(function () {

'use strict';

angular.module('OpenSlidesApp.agenda', ['OpenSlidesApp.users'])

.factory('Speaker', [
    'DS',
    function(DS) {
        return DS.defineResource({
            name: 'agenda/speaker',
            relations: {
                belongsTo: {
                    'users/user': {
                        localField: 'user',
                        localKey: 'user_id',
                    }
                }
            }
        });
    }
])

.factory('Agenda', [
    '$http',
    'DS',
    'Speaker',
    'jsDataModel',
    'Projector',
    'ProjectHelper',
    'gettextCatalog',
    'gettext',
    'CamelCase',
    'EditForm',
    function($http, DS, Speaker, jsDataModel, Projector, ProjectHelper, gettextCatalog,
        gettext, CamelCase, EditForm) {
        var name = 'agenda/item';
        return DS.defineResource({
            name: name,
            useClass: jsDataModel,
            verboseName: gettext('Agenda'),
            methods: {
                getResourceName: function () {
                    return name;
                },
                getContentObject: function () {
                    return DS.get(this.content_object.collection, this.content_object.id);
                },
                getContentObjectDetailState: function () {
                    return CamelCase(this.content_object.collection).replace('/', '.') +
                        '.detail({id: ' + this.content_object.id + '})';
                },
                getContentObjectForm: function () {
                    return EditForm.fromCollectionString(this.content_object.collection);
                },
                getContentResource: function () {
                    return DS.definitions[this.content_object.collection];
                },
                getTitle: function () {
                    var title;
                    try {
                        title =  this.getContentObject().getAgendaTitle();
                    } catch (e) {
                        // when the content object is not in the DS store.
                        title = this.title;
                    }
                    if (this.item_number) {
                        title = this.item_number + ' · ' + title;
                    }
                    return title;
                },
                getListOfSpeakersTitle: function () {
                    var title;
                    try {
                        title =  this.getContentObject().getListOfSpeakersTitle();
                        if (this.item_number) {
                            title = this.item_number + ' · ' + title;
                        }
                    } catch (e) {
                        title = this.getTitle();
                    }
                    return title;
                },
                getAgendaTitle: function () {
                    return this.title;
                },
                getCSVExportText: function () {
                    var text;
                    try {
                        text =  this.getContentObject().getCSVExportText();
                    } catch (e) {
                        // when the content object is not in the DS store
                        // or 'getCSVExportText' is not defined return nothing.
                    }
                    return text;
                },
                // link name which is shown in search result
                getSearchResultName: function () {
                    return this.getAgendaTitle();
                },
                // return true if a specific relation matches for given searchquery
                // (here: speakers)
                hasSearchResult: function (results) {
                    var item = this;
                    // search for speakers (check if any user.id from already found users matches)
                    return _.some(results, function(result) {
                        if (result.getResourceName() === "users/user") {
                            if (_.some(item.speakers, {'user_id': result.id})) {
                                return true;
                            }
                        }
                    });
                },
                getProjectorTitle: function () {
                    try {
                        return this.getContentObject().getAgendaListViewTitle();
                    } catch (e) {
                        // when the content object is not in the DS store
                        return this.list_view_title;
                    }
                },
                getListViewTitle: function () {
                    var title = this.getProjectorTitle();
                    if (this.item_number) {
                        title = this.item_number + ' · ' + title;
                    }
                    return title;
                },
                getItemNumberWithAncestors: function (agendaId) {
                    if (!agendaId) {
                        agendaId = this.id;
                    }
                    var agendaItem = DS.get(name, agendaId);
                    if (!agendaItem) {
                        return '';
                    } else if (agendaItem.item_number) {
                        return agendaItem.item_number;
                    } else if (agendaItem.parent_id) {
                        return this.getItemNumberWithAncestors(agendaItem.parent_id);
                    } else {
                        return '';
                    }
                },
                // override project function of jsDataModel factory
                project: function (projectorId, tree) {
                    if (tree) {
                        var isProjectedIds = this.isProjected(tree);
                        var requestData = {
                            clear_ids: isProjectedIds,
                        };
                        // Activate, if the projector_id is a new projector.
                        if (_.indexOf(isProjectedIds, projectorId) == -1) {
                            requestData.prune = {
                                id: projectorId,
                                element: {
                                    name: 'agenda/item-list',
                                    tree: true,
                                    id: this.id,
                                },
                            };
                        }
                        return ProjectHelper.project(requestData);
                    } else {  // Project the content object
                        var contentObject = DS.get(this.content_object.collection, this.content_object.id);
                        return contentObject.project(projectorId);
                    }
                },
                // override isProjected function of jsDataModel factory
                isProjected: function (tree) {
                    // Returns all ids of all projectors with an agenda-item element. Otherwise an empty list.
                    if (typeof tree === 'undefined') {
                        tree = false;
                    }
                    var self = this;
                    var predicate = function (element) {
                        var value;
                        if (tree) {
                            // Item tree slide for sub tree
                            value = element.name == 'agenda/item-list' &&
                                typeof element.id !== 'undefined' &&
                                element.id == self.id;
                        } else {
                            // Releated item detail slide
                            value = element.name == self.content_object.collection &&
                                typeof element.id !== 'undefined' &&
                                element.id == self.content_object.id;
                        }
                        return value;
                    };
                    var isProjectedIds = [];
                    Projector.getAll().forEach(function (projector) {
                        if (typeof _.findKey(projector.elements, predicate) === 'string') {
                            isProjectedIds.push(projector.id);
                        }
                    });
                    return isProjectedIds;
                },
                isRelatedProjected: function () {
                    // related objects for agenda items: list of speakers slide.
                    return this.isListOfSpeakersProjected();
                },
                // project list of speakers
                projectListOfSpeakers: function(projectorId) {
                    var isProjectedIds = this.isListOfSpeakersProjected();
                    var requestData = {
                        clear_ids: isProjectedIds,
                    };
                    if (_.indexOf(isProjectedIds, projectorId) == -1) {
                        requestData.prune = {
                            id: projectorId,
                            element: {
                                name: 'agenda/list-of-speakers',
                                id: this.id,
                            },
                        };
                    }
                    return ProjectHelper.project(requestData);
                },
                // check if list of speakers is projected
                isListOfSpeakersProjected: function () {
                    // Returns all ids of all projectors with an element with the
                    // name 'agenda/list-of-speakers' and the same id. Else returns an empty list.
                    var self = this;
                    var predicate = function (element) {
                        return element.name == 'agenda/list-of-speakers' &&
                               typeof element.id !== 'undefined' &&
                               element.id == self.id;
                    };
                    var isProjecteds = [];
                    Projector.getAll().forEach(function (projector) {
                        if (typeof _.findKey(projector.elements, predicate) === 'string') {
                            isProjecteds.push(projector.id);
                        }
                    });
                    return isProjecteds;
                },
                hasSubitems: function(items) {
                    var self = this;
                    var hasChild = false;
                    // Returns true if the item has at least one child item.
                    _.each(items, function (item) {
                        if (item.parent_id == self.id) {
                            hasChild = true;
                        }
                    });
                    return hasChild;
                }
            },
            relations: {
                hasMany: {
                    'core/tag': {
                        localField: 'tags',
                        localKeys: 'tags_id',
                    },
                    'agenda/speaker': {
                        localField: 'speakers',
                        foreignKey: 'item_id',
                    }
                }
            },
            beforeInject: function (resource, instance) {
                Speaker.ejectAll({where: {item_id: {'==': instance.id}}});
            }
        });
    }
])

.factory('AgendaTree', [
    function () {
        return {
            getTree: function (items) {
                // Sort items after there weight
                items.sort(function(itemA, itemB) {
                    return itemA.weight - itemB.weight;
                });

                // Build a dict with all children (dict-value) to a specific
                // item id (dict-key).
                var itemChildren = {};

                _.each(items, function (item) {
                    if (item.parent_id) {
                        // Add item to his parent. If it is the first child, then
                        // create a new list.
                        try {
                            itemChildren[item.parent_id].push(item);
                        } catch (error) {
                            itemChildren[item.parent_id] = [item];
                        }
                    }

                });

                // Recursive function that generates a nested list with all
                // items with there children
                function getChildren(items) {
                    var returnItems = [];
                    _.each(items, function (item) {
                        returnItems.push({
                            item: item,
                            children: getChildren(itemChildren[item.id]),
                            id: item.id,
                        });
                    });
                    return returnItems;
                }

                // Generates the list of root items (with no parents)
                var parentItems = items.filter(function (item) {
                    return !item.parent_id;
                });
                return getChildren(parentItems);
            },

            // Returns a list of all items as a flat tree
            getFlatTree: function(items) {
                var tree = this.getTree(items);
                var flatItems = [];

                function generateFlatTree(tree, parentCount) {
                    _.each(tree, function (item) {
                        item.item.childrenCount = item.children.length;
                        item.item.parentCount = parentCount;
                        flatItems.push(item.item);
                        generateFlatTree(item.children, parentCount + 1);
                    });
                }
                generateFlatTree(tree, 0);
                return flatItems;
            }
        };
    }
])

.factory('CurrentListOfSpeakersItem', [
    'Projector',
    'Agenda',
    function (Projector, Agenda) {
        return {
            getItem: function (projectorId) {
                var projector = Projector.get(projectorId), item;
                if (projector) {
                    _.forEach(projector.elements, function(element) {
                        if (element.agenda_item_id) {
                            item = Agenda.get(element.agenda_item_id);
                        }
                    });
                }
                return item;
            }
        };
    }
])

.factory('CurrentListOfSpeakersSlide', [
    '$http',
    'Projector',
    function($http, Projector) {
        var name = 'agenda/current-list-of-speakers';
        return {
            project: function (projectorId, overlay) {
                var isProjected = this.isProjectedWithOverlayStatus();
                _.forEach(isProjected, function (mapping) {
                    $http.post('/rest/core/projector/' + mapping.projectorId + '/deactivate_elements/',
                        [mapping.uuid]
                    );
                });

                // The slide was projected, if the id matches. If the overlay is given, also
                // the overlay is checked
                var wasProjectedBefore = _.some(isProjected, function (mapping) {
                    var value = (mapping.projectorId === projectorId);
                    if (overlay !== undefined) {
                        value = value && (mapping.overlay === overlay);
                    }
                    return value;
                });
                overlay = overlay || false; // set overlay if it wasn't defined

                if (!wasProjectedBefore) {
                    var activate = function () {
                        return $http.post(
                            '/rest/core/projector/' + projectorId + '/activate_elements/',
                             [{name: name,
                               stable: overlay, // if this is an overlay, it should not be
                                                // removed by changing the main content
                               overlay: overlay}]
                        );
                    };
                    if (!overlay) {
                        // clear all elements on this projector, because we are _not_ using the overlay.
                        $http.post('/rest/core/projector/' + projectorId + '/clear_elements/').then(activate);
                    } else {
                        activate();
                    }
                }
            },
            isProjected: function () {
                // Returns the ids of all projectors with an agenda-item element. Else return an empty list.
                var predicate = function (element) {
                    return element.name === name;
                };
                var isProjectedIds = [];
                Projector.getAll().forEach(function (projector) {
                    if (typeof _.findKey(projector.elements, predicate) === 'string') {
                        isProjectedIds.push(projector.id);
                    }
                });
                return isProjectedIds;
            },
            // Returns a list of mappings between pojector id, overlay and uuid.
            isProjectedWithOverlayStatus: function () {
                var mapping = [];
                _.forEach(Projector.getAll(), function (projector) {
                    _.forEach(projector.elements, function (element, uuid) {
                        if (element.name === name) {
                            mapping.push({
                                projectorId: projector.id,
                                uuid: uuid,
                                overlay: element.overlay || false,
                            });
                        }
                    });
                });
                return mapping;
            },
        };
    }
])



// Make sure that the Agenda resource is loaded.
.run(['Agenda', function(Agenda) {}]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.agenda.csv', [])

.factory('AgendaCsvExport', [
    'HumanTimeConverter',
    'gettextCatalog',
    'CsvDownload',
    function (HumanTimeConverter, gettextCatalog, CsvDownload) {
        var makeHeaderline = function () {
            var headerline = ['Title', 'Text', 'Duration', 'Comment', 'Internal item'];
            return _.map(headerline, function (entry) {
                return gettextCatalog.getString(entry);
            });
        };
        return {
            export: function (agenda) {
                var csvRows = [
                    makeHeaderline()
                ];
                _.forEach(agenda, function (item) {
                    var row = [];
                    var duration = item.duration ? HumanTimeConverter.secondsToHumanTime(item.duration*60,
                            { seconds: 'disabled',
                                hours: 'enabled' }) : '';
                    row.push('"' + (item.title || '') + '"');
                    row.push('"' + (item.getCSVExportText() || '') + '"');
                    row.push('"' + duration + '"');
                    row.push('"' + (item.comment || '') + '"');
                    row.push('"' + (item.is_hidden ? '1' : '')  + '"');
                    csvRows.push(row);
                });
                CsvDownload(csvRows, gettextCatalog.getString('Agenda') + '.csv');
            },
        };
    }
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.agenda.docx', ['OpenSlidesApp.core.docx'])

.factory('AgendaDocxExport', [
    '$http',
    'gettextCatalog',
    'FileSaver',
    'Agenda',
    'AgendaTree',
    'Config',
    function ($http, gettextCatalog, FileSaver, Agenda, AgendaTree, Config) {

        var getData = function (items) {
            // Item structure: The top layer has subitems, that are flat.
            // The first layer is bold and all sublayers not. The docx
            // templater cannot render items recursively, so the second
            // layer are all subitems flated out. Spacing is done with tabs.
            var tree = AgendaTree.getTree(items);
            var subitems = []; // This will be used as a temporary variable.
            var flatSubitems = function (children, parentCount) {
                _.forEach(children, function (child) {
                    var taps = _.repeat('\t', parentCount - 1);
                    subitems.push({
                        item_number: taps + child.item.item_number,
                        item_title: child.item.list_view_title,
                    });
                    flatSubitems(child.children, parentCount + 1);
                });
            };
            var twoLayerTree = _.map(tree, function (mainItem) {
                subitems = [];
                flatSubitems(mainItem.children, 1);
                return {
                    item_number: mainItem.item.item_number,
                    item_title: mainItem.item.list_view_title,
                    subitems: subitems,
                };
            });

            // header
            var headerline1 = [
                Config.translate(Config.get('general_event_name').value),
                Config.translate(Config.get('general_event_description').value)
            ].filter(Boolean).join(' – ');
            var headerline2 = [
                Config.get('general_event_location').value,
                Config.get('general_event_date').value
            ].filter(Boolean).join(', ');

            // Data structure for the docx templater.
            return {
                header: [headerline1, headerline2].join('\n'),
                agenda_translation: gettextCatalog.getString('Agenda'),
                top_list: twoLayerTree,
            };
        };

        return {
            export: function (items) {
                // TODO: use filtered items.
                var filename = gettextCatalog.getString('Agenda') + '.docx';
                $http.get('/agenda/docxtemplate/').then(function (success) {
                    var content = window.atob(success.data);
                    var doc = new Docxgen(content);

                    var data = getData(items);
                    doc.setData(data);
                    doc.render();

                    var zip = doc.getZip();
                    //zip = converter.updateZipFile(zip);

                    var out = zip.generate({type: 'blob'});
                    FileSaver.saveAs(out, filename);
                });
            },
        };
    }
]);

})();

(function () {

'use strict';

angular.module('OpenSlidesApp.agenda.pdf', ['OpenSlidesApp.core.pdf'])

.factory('AgendaContentProvider', [
    'gettextCatalog',
    'PDFLayout',
    function(gettextCatalog, PDFLayout) {

    var createInstance = function(items) {

        // page title
        var title = PDFLayout.createTitle(gettextCatalog.getString("Agenda"));

        // generate the item list with all subitems
        var createItemList = function() {
            var agenda_items = [];
            angular.forEach(items, function (item) {
                if (item.is_hidden === false) {

                    var itemIndent = item.parentCount * 20;

                    var itemStyle;
                    if (item.parentCount === 0) {
                        itemStyle = 'listParent';
                    } else {
                        itemStyle = 'listChild';
                    }

                    var itemNumberWidth;
                    if (item.item_number === "") {
                        itemNumberWidth = 0;
                    } else {
                        itemNumberWidth = 60;
                    }

                    var agendaJsonString = {
                        style: itemStyle,
                        columns: [
                            {
                                width: itemIndent,
                                text: ''
                            },
                            {
                                width: itemNumberWidth,
                                text: item.item_number
                            },
                            {
                                text: item.title
                            }
                        ]
                    };

                    agenda_items.push(agendaJsonString);
                }
            });
            return agenda_items;
        };

        var getContent = function() {
            return [
                title,
                createItemList()
            ];
        };

        return {
            getContent: getContent
        };
    };

    return {
        createInstance: createInstance
    };

}])

.factory('AgendaPdfExport', [
    'gettextCatalog',
    'AgendaContentProvider',
    'PdfMakeDocumentProvider',
    'PdfCreate',
    'Messaging',
    function (gettextCatalog, AgendaContentProvider, PdfMakeDocumentProvider, PdfCreate, Messaging) {
        return {
            export: function (items) {
                var filename = gettextCatalog.getString('Agenda') + '.pdf';
                var agendaContentProvider = AgendaContentProvider.createInstance(items);
                PdfMakeDocumentProvider.createInstance(agendaContentProvider).then(function (documentProvider) {
                    PdfCreate.download(documentProvider, filename);
                }, function (error) {
                    Messaging.addMessage(error.msg, 'error');
                });
            },
        };
    }
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.agenda.projector', ['OpenSlidesApp.agenda'])

.config([
    'slidesProvider',
    function(slidesProvider) {
        slidesProvider.registerSlide('agenda/list-of-speakers', {
            template: 'static/templates/agenda/slide-list-of-speakers.html',
        });
        slidesProvider.registerSlide('agenda/item-list', {
            template: 'static/templates/agenda/slide-item-list.html',
        });
        slidesProvider.registerSlide('agenda/current-list-of-speakers', {
            template: 'static/templates/agenda/slide-current-list-of-speakers.html',
        });
    }
])

.controller('SlideCurrentListOfSpeakersCtrl', [
    '$scope',
    'Agenda',
    'CurrentListOfSpeakersItem',
    'Config',
    'Projector',
    function ($scope, Agenda, CurrentListOfSpeakersItem, Config, Projector) {
        $scope.overlay = $scope.element.overlay;
        // Watch for changes in the current list of speakers reference
        $scope.$watch(function () {
            return Config.lastModified('projector_currentListOfSpeakers_reference');
        }, function () {
            $scope.currentListOfSpeakersReference = $scope.config('projector_currentListOfSpeakers_reference');
            $scope.updateCurrentListOfSpeakers();
        });
        // Watch for changes in the referenced projector
        $scope.$watch(function () {
            return Projector.lastModified($scope.currentListOfSpeakersReference);
        }, function () {
            $scope.updateCurrentListOfSpeakers();
        });
        // Watch for changes in the current item.
        $scope.$watch(function () {
            return Agenda.lastModified();
        }, function () {
            $scope.updateCurrentListOfSpeakers();
        });
        $scope.updateCurrentListOfSpeakers = function () {
            $scope.agendaItem = CurrentListOfSpeakersItem.getItem($scope.currentListOfSpeakersReference);
        };
    }
])

.controller('SlideListOfSpeakersCtrl', [
    '$scope',
    'Agenda',
    'User',
    function ($scope, Agenda, User) {
        // Attention! Each object that is used here has to be dealt on server side.
        // Add it to the coresponding get_requirements method of the ProjectorElement
        // class.
        var id = $scope.element.id;
        Agenda.bindOne(id, $scope, 'item');
    }
])

.controller('SlideItemListCtrl', [
    '$scope',
    '$http',
    '$filter',
    'Agenda',
    'AgendaTree',
    'Config',
    function ($scope, $http, $filter, Agenda, AgendaTree, Config) {
        // Attention! Each object that is used here has to be dealt on server side.
        // Add it to the coresponding get_requirements method of the ProjectorElement
        // class.

        // Bind agenda tree to the scope
        var items;
        $scope.$watch(function () {
            return Agenda.lastModified() +
                Config.lastModified('agenda_hide_internal_items_on_projector');
        }, function () {
            if ($scope.element.id) {
                if (Config.get('agenda_hide_internal_items_on_projector').value) {
                    items = _.filter(Agenda.getAll(), function (item) {
                        return item.type === 1;
                    });
                } else {
                    items = Agenda.getAll();
                }
                var tree = AgendaTree.getTree(items);

                var getRootNode = function (node) {
                    if (node.id == $scope.element.id) {
                        return node;
                    }
                    for (var i = 0; i < node.children.length; i++) {
                        var result = getRootNode(node.children[i]);
                        if (result) {
                            return result;
                        }
                    }
                    return false;
                };
                _.forEach(tree, function (node) {
                    var result = getRootNode(node);
                    if (result) {
                        $scope.rootItem = result.item;
                        $scope.tree = result.children;
                        return false;
                    }
                });
            } else if ($scope.element.tree) {
                items = _.filter(Agenda.getAll(), function (item) {
                    return item.type === 1;
                });
                $scope.tree = AgendaTree.getTree(items);
            } else {
                items = Agenda.filter({
                    where: { parent_id: null },
                    orderBy: 'weight'
                });
                items = _.filter(items, function (item) {
                    return item.type === 1;
                });
                $scope.tree = AgendaTree.getTree(items);
            }
        });
    }
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.agenda.site', [
    'OpenSlidesApp.agenda',
    'OpenSlidesApp.core.pdf',
    'OpenSlidesApp.agenda.pdf',
    'OpenSlidesApp.agenda.csv',
    'OpenSlidesApp.agenda.docx',
])

.config([
    'mainMenuProvider',
    'gettext',
    function (mainMenuProvider, gettext) {
        mainMenuProvider.register({
            'ui_sref': 'agenda.item.list',
            'img_class': 'calendar-o',
            'title': gettext('Agenda'),
            'weight': 200,
            'perm': 'agenda.can_see',
        });
    }
])

.config([
    'SearchProvider',
    'gettext',
    function (SearchProvider, gettext) {
        SearchProvider.register({
            'verboseName': gettext('Agenda'),
            'collectionName': 'agenda/item',
            'urlDetailState': 'agenda.item.detail',
            'weight': 200,
        });
    }
])

.config([
    '$stateProvider',
    'gettext',
    function ($stateProvider, gettext) {
        $stateProvider
            .state('agenda', {
                url: '/agenda',
                abstract: true,
                template: "<ui-view/>",
                data: {
                    title: gettext('Agenda'),
                    basePerm: 'agenda.can_see',
                },
            })
            .state('agenda.item', {
                abstract: true,
                template: "<ui-view/>",
            })
            .state('agenda.item.list', {})
            .state('agenda.item.detail', {
                resolve: {
                    itemId: ['$stateParams', function($stateParams) {
                        return $stateParams.id;
                    }],
                }
            })
            .state('agenda.item.sort', {
                url: '/sort',
                controller: 'AgendaSortCtrl',
            })
            .state('agenda.current-list-of-speakers', {
                url: '/speakers',
                controller: 'CurrentListOfSpeakersViewCtrl',
                data: {
                    title: gettext('Current list of speakers'),
                },
            });
    }
])

// Set the sensitivity of moving nodes horizontal for the ui-tree.
.config([
    'treeConfig',
    function (treeConfig) {
        treeConfig.dragMoveSensitivity = 20;
    }
])

.controller('ItemListCtrl', [
    '$scope',
    '$filter',
    '$http',
    '$state',
    'DS',
    'operator',
    'ngDialog',
    'Agenda',
    'TopicForm', // TODO: Remove this dependency. Use template hook for "New" and "Import" buttons.
    'AgendaTree',
    'Projector',
    'ProjectionDefault',
    'gettextCatalog',
    'gettext',
    'osTableFilter',
    'osTablePagination',
    'AgendaCsvExport',
    'AgendaPdfExport',
    'AgendaDocxExport',
    'ErrorMessage',
    function($scope, $filter, $http, $state, DS, operator, ngDialog, Agenda, TopicForm,
        AgendaTree, Projector, ProjectionDefault, gettextCatalog, gettext, osTableFilter,
        osTablePagination, AgendaCsvExport, AgendaPdfExport, AgendaDocxExport, ErrorMessage) {
        // Bind agenda tree to the scope
        $scope.$watch(function () {
            return Agenda.lastModified();
        }, function () {
            // Filter out items that doesn't have the list_item_title. This happens, if the
            // item is a hidden item but provides the list of speakers, but should not be
            // visible in the list view.
            var allowedItems = _.filter(Agenda.getAll(), function (item) {
                return item.list_view_title;
            });
            $scope.items = AgendaTree.getFlatTree(allowedItems);
            $scope.agendaHasSubitems = $filter('filter')($scope.items, {'parent_id': ''}).length;
        });
        Projector.bindAll({}, $scope, 'projectors');
        $scope.mainListTree = true;
        $scope.$watch(function () {
            return Projector.lastModified();
        }, function () {
            var projectiondefault =  ProjectionDefault.filter({name: 'agenda_all_items'})[0];
            if (projectiondefault) {
                $scope.defaultProjectorId_all_items = projectiondefault.projector_id;
            }
            $scope.projectionDefaults = ProjectionDefault.getAll();
        });
        $scope.alert = {};


        // Filtering
        $scope.filter = osTableFilter.createInstance('AgendaTableFilter');

        if (!$scope.filter.existsStorageEntry()) {
            $scope.filter.booleanFilters = {
                closed: {
                    value: undefined,
                    displayName: gettext('Closed items'),
                    choiceYes: gettext('Closed items'),
                    choiceNo: gettext('Open items'),
                },
                is_hidden: {
                    value: undefined,
                    displayName: gettext('Internal items'),
                    choiceYes: gettext('Internal items'),
                    choiceNo: gettext('No internal items'),
                    permission: 'agenda.can_see_hidden_items',
                },
            };
        }
        $scope.filter.propertyList = ['item_number', 'title', 'title_list_view', 'comment', 'duration'];
        $scope.filter.propertyFunctionList = [
            function (item) {return item.getListViewTitle();},
        ];
        $scope.filter.propertyDict = {
            'speakers' : function (speaker) {
                return '';
            },
        };

        // Expand all items during searching.
        $scope.filter.changed = function () {
            $scope.collapseState = true;
            $scope.toggleCollapseState();
        };

        // pagination
        $scope.pagination = osTablePagination.createInstance('AgendaTablePagination', 50);

        // parse duration for inline editing
        $scope.generateDurationText = function (item) {
            //convert data from model format (m) to view format (hh:mm)
            if (item.duration) {
                var time = "",
                    totalminutes = item.duration;
                if (totalminutes < 0) {
                    time = "-";
                    totalminutes = -totalminutes;
                }
                var hh = Math.floor(totalminutes / 60);
                var mm = Math.floor(totalminutes % 60);
                // Add leading "0" for double digit values
                mm = ("0"+mm).slice(-2);
                time += hh + ":" + mm;
                item.durationText = time;
            } else {
                item.durationText = "";
            }
        };
        $scope.setDurationText = function (item) {
            //convert data from view format (hh:mm) to model format (m)
            var time = item.durationText.replace('h', '').split(':');
            var data;
            if (time.length > 1 && !isNaN(time[0]) && !isNaN(time[1])) {
                data = (+time[0]) * 60 + (+time[1]);
                if (data < 0) {
                    data = "-"+data;
                }
                item.duration = parseInt(data);
            } else if (time.length == 1 && !isNaN(time[0])) {
                data = (+time[0]);
                item.duration = parseInt(data);
            } else {
                item.duration = 0;
            }
            $scope.save(item);
        };

        /** Duration calculations **/
        $scope.sumDurations = function () {
            var totalDuration = 0;
            $scope.items.forEach(function (item) {
                if (item.duration) {
                    totalDuration += item.duration;
                }
            });
            return totalDuration;
        };
        $scope.calculateEndTime = function () {
            var totalDuration = $scope.sumDurations();
            var startTimestamp = $scope.config('agenda_start_event_date_time');
            if (startTimestamp) {
                var endTimestamp = startTimestamp + totalDuration * 60 * 1000;
                var endDate = new Date(endTimestamp);
                var mm = ("0" + endDate.getMinutes()).slice(-2);
                var dateStr = endDate.getHours() + ':' + mm;
                return dateStr;
            } else {
                return '';
            }
        };

        // Agenda collapse function
        $scope.toggleCollapseState = function () {
            $scope.collapseState = !$scope.collapseState;
            _.forEach($scope.items, function (item) {
                item.hideChildren = $scope.collapseState;
            });
        };

        // Check, if an item has childs in all filtered items
        $scope.hasChildren = function (item) {
            return _.some($scope.itemsFiltered, function (_item) {
                return _item.parent_id == item.id;
            });
        };

        // returns true, if the agenda has at least two layers
        $scope.agendaHasMultipleLayers = function () {
            return _.some($scope.items, function (item) {
                return item.parent_id;
            });
        };

        /** Agenda item functions **/
        // open dialog for new topics // TODO Remove this. Don't forget import button in template.
        $scope.newDialog = function () {
            ngDialog.open(TopicForm.getDialog());
        };
        // save changed item
        $scope.save = function (item) {
            Agenda.save(item).then(
                function (success) {
                    $scope.alert.show = false;
                },
                function (error) {
                    $scope.alert = ErrorMessage.forAlert(error);
                });
        };
        // delete related item
        $scope.deleteRelatedItem = function (item) {
            DS.destroy(item.content_object.collection, item.content_object.id);
        };
        // auto numbering of agenda items
        $scope.autoNumbering = function() {
            $http.post('/rest/agenda/item/numbering/', {});
        };
        // check open permission
        // TODO: Use generic solution here.
        $scope.isAllowedToSeeOpenLink = function (item) {
            var collection = item.content_object.collection;
            switch (collection) {
                case 'topics/topic':
                    return operator.hasPerms('agenda.can_see');
                case 'motions/motion':
                    return operator.hasPerms('motions.can_see');
                case 'motions/motion-block':
                    return operator.hasPerms('motions.can_see');
                case 'assignments/assignment':
                    return operator.hasPerms('assignments.can_see');
                default:
                    return false;
            }
        };
        $scope.edit = function (item) {
            ngDialog.open(item.getContentObjectForm().getDialog({id: item.content_object.id}));
        };

        // export
        $scope.pdfExport = function () {
            AgendaPdfExport.export($scope.itemsFiltered);
        };
        $scope.csvExport = function () {
            AgendaCsvExport.export($scope.itemsFiltered);
        };
        $scope.docxExport = function () {
            AgendaDocxExport.export($scope.itemsFiltered);
        };

        /** select mode functions **/
        $scope.isSelectMode = false;
        // check all checkboxes
        $scope.checkAll = function () {
            $scope.selectedAll = !$scope.selectedAll;
            angular.forEach($scope.items, function (item) {
                item.selected = $scope.selectedAll;
            });
        };
        // uncheck all checkboxes if isDeleteMode is closed
        $scope.uncheckAll = function () {
            if (!$scope.isSelectMode) {
                $scope.selectedAll = false;
                angular.forEach($scope.items, function (item) {
                    item.selected = false;
                });
            }
        };
        // delete selected items
        $scope.deleteMultiple = function () {
            angular.forEach($scope.items, function (item) {
                if (item.selected) {
                    DS.destroy(item.content_object.collection, item.content_object.id);
                }
            });
            $scope.isSelectMode = false;
            $scope.uncheckAll();
        };

        /** Project functions **/
        // get ProjectionDefault for item
        $scope.getProjectionDefault = function (item) {
            if (item.tree) {
                return $scope.defaultProjectorId_all_items;
            } else {
                var app_name = item.content_object.collection.split('/')[0];
                var id = 1;
                $scope.projectionDefaults.forEach(function (projectionDefault) {
                    if (projectionDefault.name == app_name) {
                        id = projectionDefault.projector_id;
                    }
                });
                return id;
            }
        };
        // project agenda
        $scope.projectAgenda = function (projectorId, tree, id) {
            var isAgendaProjectedIds = $scope.isAgendaProjected($scope.mainListTree);
            _.forEach(isAgendaProjectedIds, function (id) {
                $http.post('/rest/core/projector/' + id + '/clear_elements/');
            });
            if (_.indexOf(isAgendaProjectedIds, projectorId) == -1) {
                $http.post('/rest/core/projector/' + projectorId + '/prune_elements/',
                    [{name: 'agenda/item-list', tree: tree, id: id}]);
            }
        };
        // change whether all items or only main items should be projected
        $scope.changeMainListTree = function () {
            var isAgendaProjectedId = $scope.isAgendaProjected($scope.mainListTree);
            $scope.mainListTree = !$scope.mainListTree;
            if (isAgendaProjectedId > 0) {
                $scope.projectAgenda(isAgendaProjectedId, $scope.mainListTree);
            }
        };
        // change whether one item or all subitems should be projected
        $scope.changeItemTree = function (item) {
            var tree = item.tree;
            item.tree = !item.tree;
            var isProjected = item.isProjected(tree);
            _.forEach(isProjected, function (projectorId) {
                // Deactivate and reactivate
                item.project(projectorId, tree).then(function (s) {
                    item.project(projectorId, !tree);
                });
            });
        };
        // check if agenda is projected
        $scope.isAgendaProjected = function (tree) {
            // Returns the ids of all projectors with an element with
            // the name 'agenda/item-list'. Else returns an empty list.
            var predicate = function (element) {
                var value;
                if (tree) {
                    // tree with all agenda items
                    value = element.name == 'agenda/item-list' &&
                        typeof element.id === 'undefined' &&
                        element.tree;
                } else {
                    // only main agenda items
                    value = element.name == 'agenda/item-list' &&
                        typeof element.id === 'undefined' &&
                        !element.tree;
                }
                return value;
            };
            var projectorIds = [];
            $scope.projectors.forEach(function (projector) {
                if (typeof _.findKey(projector.elements, predicate) === 'string') {
                    projectorIds.push(projector.id);
                }
            });
            return projectorIds;
        };
    }
])

// filter to hide collapsed items. Items has to be a flat tree.
.filter('collapsedItemFilter', [
    function () {
        return function (items) {
            return _.filter(items, function (item) {
                var index = _.findIndex(items, item);
                var parentId = item.parent_id;
                // Search for parents, if one has the hideChildren attribute set. All parents
                // have a higher index as this item, because items is a flat tree.
                // If a parent has this attribute, we should remove this item. Else if we hit
                // the top or an item on the first layer, the item is not collapsed.
                for (--index; index >= 0 && parentId !== null; index--) {
                    var p = items[index];
                    if (p.id === parentId) {
                        if (p.hideChildren) {
                            return false;
                        } else {
                            parentId = p.parent_id;
                        }
                    }
                }
                return true;
            });
        };
    }
])

.controller('ItemDetailCtrl', [
    '$scope',
    '$filter',
    'Agenda',
    'itemId',
    'Projector',
    'ProjectionDefault',
    'gettextCatalog',
    'WebpageTitle',
    'ErrorMessage',
    function ($scope, $filter, Agenda, itemId, Projector, ProjectionDefault, gettextCatalog, WebpageTitle,
        ErrorMessage) {
        $scope.alert = {};

        $scope.$watch(function () {
            return Agenda.lastModified(itemId);
        }, function () {
            $scope.item = Agenda.get(itemId);
            WebpageTitle.updateTitle(gettextCatalog.getString('List of speakers') + ' ' +
                gettextCatalog.getString('of') + ' ' + $scope.item.getTitle());
            // all speakers
            $scope.speakers = $filter('orderBy')($scope.item.speakers, 'weight');
            // next speakers
            $scope.nextSpeakers = $filter('filter')($scope.speakers, {'begin_time': null});
            // current speaker
            $scope.currentSpeaker = $filter('filter')($scope.speakers, {'begin_time': '!!', 'end_time': null});
            // last speakers
            $scope.lastSpeakers = $filter('filter')($scope.speakers, {'end_time': '!!'});
            $scope.lastSpeakers = $filter('orderBy')($scope.lastSpeakers, 'begin_time');
        });
        $scope.$watch(function () {
            return Projector.lastModified();
        }, function () {
            var item_app_name = $scope.item.content_object.collection.split('/')[0];
            var projectiondefaultItem = ProjectionDefault.filter({name: item_app_name})[0];
            if (projectiondefaultItem) {
                $scope.defaultProjectorItemId = projectiondefaultItem.projector_id;
            }
            var projectiondefaultListOfSpeakers = ProjectionDefault.filter({name: 'agenda_list_of_speakers'})[0];
            if (projectiondefaultListOfSpeakers) {
                $scope.defaultProjectorListOfSpeakersId = projectiondefaultListOfSpeakers.projector_id;
            }
        });
    }
])

/* This is the controller for the list of speakers partial management template.
 * The parent controller needs to provide a $scope.item, $scope.speakers, $scope.nextSpeakers,
 * $scope.currentSpeakers, $scope.lastSpeakers. See (as example) ItemDetailCtrl. */
.controller('ListOfSpeakersManagementCtrl', [
    '$scope',
    '$http',
    '$filter',
    'Agenda',
    'User',
    'operator',
    'ErrorMessage',
    function ($scope, $http, $filter, Agenda, User, operator, ErrorMessage) {
        User.bindAll({}, $scope, 'users');
        $scope.speakerSelectBox = {};

        // close/open list of speakers of current item
        $scope.closeList = function (listClosed) {
            $scope.item.speaker_list_closed = listClosed;
            Agenda.save($scope.item);
        };

        // add user to list of speakers
        $scope.addSpeaker = function (userId) {
            $http.post('/rest/agenda/item/' + $scope.item.id + '/manage_speaker/', {'user': userId}).then(
                function (success) {
                    $scope.alert.show = false;
                    $scope.speakerSelectBox = {};
                }, function (error) {
                    $scope.alert = ErrorMessage.forAlert(error);
                    $scope.speakerSelectBox = {};
                }
            );
        };

        // delete speaker(!) from list of speakers
        $scope.removeSpeaker = function (speakerId) {
            $http.delete(
                '/rest/agenda/item/' + $scope.item.id + '/manage_speaker/',
                {headers: {'Content-Type': 'application/json'},
                 data: JSON.stringify({speaker: speakerId})}
            )
            .then(function (success) {
            }, function (error) {
                $scope.alert = ErrorMessage.forAlert(error);
            });
        };

        //delete all speakers from list of speakers
        $scope.removeAllSpeakers = function () {
            var speakersOnList = [];
            angular.forEach($scope.item.speakers, function (speaker) {
                speakersOnList.push(speaker.id);
            });
            $http.delete(
                '/rest/agenda/item/' + $scope.item.id + '/manage_speaker/',
                {headers: {'Content-Type': 'application/json'},
                 data: JSON.stringify({speaker: speakersOnList})}
            )
            .then(function (success) {
            }, function (error) {
                $scope.alert = ErrorMessage.forAlert(error);
            });
        };

        // Return true if the requested user is allowed to do a specific action
        // and see the corresponding button (e.g. 'add me' or 'remove me').
        $scope.isAllowed = function (action) {
            var nextUsers = [];
            angular.forEach($scope.nextSpeakers, function (speaker) {
                nextUsers.push(speaker.user_id);
            });
            switch (action) {
                case 'add':
                    return (operator.hasPerms('agenda.can_be_speaker') &&
                            !$scope.item.speaker_list_closed &&
                            $.inArray(operator.user.id, nextUsers) == -1);
                case 'remove':
                    if (operator.user) {
                        return ($.inArray(operator.user.id, nextUsers) != -1);
                    }
                    return false;
                case 'removeAll':
                    return (operator.hasPerms('agenda.can_manage_list_of_speakers') &&
                            $scope.speakers.length > 0);
                case 'showLastSpeakers':
                    return $scope.lastSpeakers.length > 0;
            }
        };

        // begin speech of selected/next speaker
        $scope.beginSpeech = function (speakerId) {
            $http.put('/rest/agenda/item/' + $scope.item.id + '/speak/', {'speaker': speakerId})
            .then(function (success) {
                $scope.alert.show = false;
            }, function (error) {
                $scope.alert = ErrorMessage.forAlert(error);
            });
        };

        // end speech of current speaker
        $scope.endSpeech = function () {
            $http.delete(
                '/rest/agenda/item/' + $scope.item.id + '/speak/',
                {headers: {'Content-Type': 'application/json'}, data: {}}
            ).then(
                function (success) {},
                function (error) {
                    $scope.alert = ErrorMessage.forAlert(error);
                }
            );
        };
        // gets speech duration of selected speaker in seconds
        $scope.getDuration = function (speaker) {
            var beginTimestamp = new Date(speaker.begin_time).getTime();
            var endTimestamp = new Date(speaker.end_time).getTime();
            // calculate duration in seconds
            return Math.floor((endTimestamp - beginTimestamp) / 1000);

        };
        // save reordered list of speakers
        $scope.treeOptions = {
            dropped: function (event) {
                var sortedSpeakers = _.map($scope.nextSpeakers, function (speaker) {
                    return speaker.id;
                });
                $http.post('/rest/agenda/item/' + $scope.item.id + '/sort_speakers/',
                    {speakers: sortedSpeakers}
                );
            }
        };

        // Marking a speaker
        $scope.toggleMarked = function (speaker) {
            $http.patch('/rest/agenda/item/' + $scope.item.id + '/manage_speaker/', {
                user: speaker.user.id,
                marked: !speaker.marked,
            }).then(function (success) {
                $scope.alert.show = false;
            }, function (error) {
                $scope.alert = ErrorMessage.forAlert(error);
            });
        };
    }
])

.controller('AgendaSortCtrl', [
    '$scope',
    '$http',
    'Agenda',
    'AgendaTree',
    'ErrorMessage',
    function($scope, $http, Agenda, AgendaTree, ErrorMessage) {
        // Bind agenda tree to the scope
        $scope.$watch(function () {
            return Agenda.lastModified();
        }, function () {
            $scope.items = AgendaTree.getTree(Agenda.getAll());
        });
        $scope.showInternalItems = true;
        $scope.alert = {};

        // save parent and weight of moved agenda item (and all items on same level)
        $scope.treeOptions = {
            dropped: function(event) {
                var parentID = null;
                var droppedItemID = event.source.nodeScope.$modelValue.id;
                if (event.dest.nodesScope.item) {
                    parentID = event.dest.nodesScope.item.id;
                }
                $http.post('/rest/agenda/item/sort/', {
                    nodes: event.dest.nodesScope.$modelValue,
                    parent_id: parentID}
                ).then(
                    function(success) {},
                    function(error){
                        $scope.alert = ErrorMessage.forAlert(error);
                    }
                );
            }
        };
    }
])

.controller('CurrentListOfSpeakersViewCtrl', [
    '$scope',
    '$http',
    '$filter',
    'Projector',
    'ProjectionDefault',
    'Agenda',
    'Config',
    'CurrentListOfSpeakersItem',
    'CurrentListOfSpeakersSlide',
    'gettextCatalog',
    'WebpageTitle',
    function($scope, $http, $filter, Projector, ProjectionDefault, Agenda, Config,
        CurrentListOfSpeakersItem, CurrentListOfSpeakersSlide, gettextCatalog, WebpageTitle) {
        $scope.alert = {};
        $scope.currentListOfSpeakers = CurrentListOfSpeakersSlide;

        // Watch for changes in the current list of speakers reference
        $scope.$watch(function () {
            return Config.lastModified('projector_currentListOfSpeakers_reference');
        }, function () {
            $scope.currentListOfSpeakersReference = $scope.config('projector_currentListOfSpeakers_reference');
            $scope.updateCurrentListOfSpeakersItem();
        });
        $scope.$watch(function () {
            return Projector.lastModified();
        }, function() {
            $scope.projectors = Projector.getAll();
            // If there is just one projector we provide just the overlay.
            if ($scope.projectors.length === 1) {
                $scope.currentListOfSpeakersAsOverlay = true;
            }
            $scope.updateCurrentListOfSpeakersItem();

            $scope.listOfSpeakersDefaultProjectorId = ProjectionDefault.filter({name: 'agenda_current_list_of_speakers'})[0].projector_id;
        });

        $scope.$watch(function () {
            return $scope.item ? Agenda.lastModified($scope.item.id) : void 0;
        }, function () {
            $scope.updateCurrentListOfSpeakersItem();
        });

        $scope.updateCurrentListOfSpeakersItem = function () {
            $scope.item = CurrentListOfSpeakersItem.getItem($scope.currentListOfSpeakersReference);
            if ($scope.item) {
                // all speakers
                $scope.speakers = $filter('orderBy')($scope.item.speakers, 'weight');
                // next speakers
                $scope.nextSpeakers = $filter('filter')($scope.speakers, {'begin_time': null});
                // current speaker
                $scope.currentSpeaker = $filter('filter')($scope.speakers, {'begin_time': '!!', 'end_time': null});
                // last speakers
                $scope.lastSpeakers = $filter('filter')($scope.speakers, {'end_time': '!!'});
                $scope.lastSpeakers = $filter('orderBy')($scope.lastSpeakers, 'begin_time');
            } else {
                $scope.speakers = void 0;
                $scope.nextSpeakers = void 0;
                $scope.currentSpeaker = void 0;
                $scope.lastSpeakers = void 0;
            }
            if ($scope.item) {
                WebpageTitle.updateTitle(gettextCatalog.getString('Current list of speakers') + ' ' +
                    gettextCatalog.getString('of') + ' ' + $scope.item.getTitle());
            } else {
                WebpageTitle.updateTitle(gettextCatalog.getString('Current list of speakers'));
            }
        };

        // Set the current overlay status
        if ($scope.currentListOfSpeakers.isProjected().length) {
            var isProjected = $scope.currentListOfSpeakers.isProjectedWithOverlayStatus();
            $scope.currentListOfSpeakersAsOverlay = isProjected[0].overlay;
        } else {
            $scope.currentListOfSpeakersAsOverlay = false;
        }
        $scope.setOverlay = function (overlay) {
            $scope.currentListOfSpeakersAsOverlay = overlay;
            var isProjected = $scope.currentListOfSpeakers.isProjectedWithOverlayStatus();
            if (isProjected.length) {
                _.forEach(isProjected, function (mapping) {
                    if (mapping.overlay != overlay) { // change the overlay if it is different
                        $scope.currentListOfSpeakers.project(mapping.projectorId, overlay);
                    }
                });
            }
        };
    }
])

//mark all agenda config strings for translation with Javascript
.config([
    'gettext',
    function (gettext) {
        gettext('Enable numbering for agenda items');
        gettext('Numbering prefix for agenda items');
        gettext('This prefix will be set if you run the automatic agenda numbering.');
        gettext('Agenda');
        gettext('Invalid input.');
        gettext('Numeral system for agenda items');
        gettext('Arabic');
        gettext('Roman');
        gettext('Begin of event');
        gettext('Input format: DD.MM.YYYY HH:MM');
        gettext('Hide internal items when projecting subitems');
        gettext('Number of last speakers to be shown on the projector');
        gettext('List of speakers');
        gettext('Show orange countdown in the last x seconds of speaking time');
        gettext('Enter duration in seconds. Choose 0 to disable warning color.');
        gettext('Couple countdown with the list of speakers');
        gettext('[Begin speech] starts the countdown, [End speech] stops the ' +
                'countdown.');
    }
 ]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.assignments', [])

.factory('AssignmentPollOption', [
    'DS',
    'jsDataModel',
    'gettextCatalog',
    'Config',
    'MajorityMethods',
    function (DS, jsDataModel, gettextCatalog, Config, MajorityMethods) {
        return DS.defineResource({
            name: 'assignments/polloption',
            useClass: jsDataModel,
            methods: {
                getVotes: function () {
                    if (!this.poll.has_votes) {
                        // Return undefined if this poll has no votes.
                        return;
                    }

                    // Initial values for the option
                    var votes = [],
                        config = Config.get('assignments_poll_100_percent_base').value;

                    var base = this.poll.getPercentBase(config);
                    if (typeof base === 'object' && base !== null) {
                        // this.poll.pollmethod === 'yna'
                        base = base[this.id];
                    }

                    _.forEach(this.votes, function (vote) {
                        // Initial values for the vote
                        var order = '',
                            value = '',
                            percentStr = '',
                            percentNumber;

                        // Check for special value
                        switch (vote.weight) {
                            case -1:
                                value = gettextCatalog.getString('majority');
                                break;
                            case -2:
                                value = gettextCatalog.getString('undocumented');
                                break;
                            default:
                                if (vote.weight >= 0) {
                                    value = vote.weight;
                                } else {
                                    value = 0;  // Vote was not defined. Set value to 0.
                                }
                        }
                        switch (vote.value) {
                            case "Yes":
                                order = 1;
                                break;
                            case "No":
                                order = 2;
                                break;
                            case "Abstain":
                                order = 3;
                                break;
                            default:
                                order = 0;
                        }

                        // Special case where to skip percents
                        var skipPercents = config === 'YES_NO' && vote.value === 'Abstain';

                        if (base && !skipPercents) {
                            percentNumber = Math.round(vote.weight * 100 / base * 100) / 100;
                            if (percentNumber >= 0) {
                                percentStr = '(' + percentNumber + ' %)';
                            }
                        }
                        votes.push({
                            'order': order,
                            'label': gettextCatalog.getString(vote.value),
                            'value': value,
                            'percentStr': percentStr,
                            'percentNumber': percentNumber
                        });
                    });
                    return _.sortBy(votes, 'order');
                },

                // Returns 0 or positive integer if quorum is reached or surpassed.
                // Returns negativ integer if quorum is not reached.
                // Returns undefined if we can not calculate the quorum.
                isReached: function (method) {
                    if (!this.poll.has_votes) {
                        // Return undefined if this poll has no votes.
                        return;
                    }
                    var isReached;
                    var config = Config.get('assignments_poll_100_percent_base').value;
                    var base = this.poll.getPercentBase(config);
                    if (typeof base === 'object' && base !== null) {
                        // this.poll.pollmethod === 'yna'
                        base = base[this.id];
                    }
                    if (base) {
                        // Provide result only if base is not undefined and not 0.
                        isReached = MajorityMethods[method](this.getVoteYes(), base);
                    }
                    return isReached;
                },

                // Returns the weight for the vote or the vote 'yes' in case of YNA poll method.
                getVoteYes: function () {
                    var voteYes = 0;
                    if (this.poll.pollmethod === 'yna') {
                        var voteObj = _.find(this.votes, function (vote) {
                            return vote.value === 'Yes';
                        });
                        if (voteObj) {
                            voteYes = voteObj.weight;
                        }
                    } else {
                        // pollmethod === 'votes'
                        voteYes = this.votes[0].weight;
                    }
                    return voteYes;
                }
            },
            relations: {
                belongsTo: {
                    'assignments/poll': {
                        localField: 'poll',
                        localKey: 'poll_id',
                    },
                    'users/user': {
                        localField: 'candidate',
                        localKey: 'candidate_id',
                    }
                }
            },
        });
    }
])

.factory('AssignmentPoll', [
    '$http',
    'DS',
    'jsDataModel',
    'gettextCatalog',
    'AssignmentPollOption',
    'Config',
    function ($http, DS, jsDataModel, gettextCatalog, AssignmentPollOption, Config) {
        var name = 'assignments/poll';
        return DS.defineResource({
            name: name,
            useClass: jsDataModel,
            methods: {
                getResourceName: function () {
                    return name;
                },

                // Returns percent base. Returns undefined if calculation is not possible in general.
                getPercentBase: function (config, type) {
                    var base;
                    switch (config) {
                        case 'CAST':
                            if (this.votescast <= 0 || this.votesinvalid < 0) {
                                // It would be OK to check only this.votescast < 0 because 0
                                // is checked again later but this is a little bit faster.
                                break;
                            }
                            base = this.votescast;
                            /* falls through */
                        case 'VALID':
                            if (this.votesvalid < 0) {
                                base = void 0;
                                break;
                            }
                            if (typeof base === 'undefined' && type !== 'votescast' && type !== 'votesinvalid') {
                                base = this.votesvalid;
                            }
                            /* falls through */
                        case 'YES_NO_ABSTAIN':
                        case 'YES_NO':
                            if (this.pollmethod === 'yna') {
                                if (typeof base === 'undefined' && type !== 'votescast' && type !== 'votesinvalid' && type !== 'votesvalid') {
                                    base = {};
                                    _.forEach(this.options, function (option) {
                                        var allVotes = option.votes;
                                        if (config === 'YES_NO') {
                                            allVotes = _.filter(allVotes, function (vote) {
                                                // Extract abstain votes in case of YES_NO percent base.
                                                // Do not extract abstain vote if it is set to majority so the predicate later
                                                // fails and therefor we get an undefined base. Reason: It should not be possible
                                                // to set abstain to majority and nevertheless calculate percents of yes and no.
                                                return vote.value !== 'Abstain' || vote.weight === -1;
                                            });
                                        }
                                        var predicate = function (vote) {
                                            return vote.weight < 0;
                                        };
                                        if (_.findIndex(allVotes, predicate) === -1) {
                                            base[option.id] = _.reduce(allVotes, function (sum, vote) {
                                                return sum + vote.weight;
                                            }, 0);
                                        }
                                    });
                                }
                            } else {
                                // this.pollmethod === 'votes'
                                var predicate = function (option) {
                                    return option.votes[0].weight < 0;
                                };
                                if (_.findIndex(this.options, predicate) !== -1) {
                                    base = void 0;
                                } else {
                                    if (typeof base === 'undefined' && type !== 'votescast' && type !== 'votesinvalid' && type !== 'votesvalid') {
                                        base = _.reduce(this.options, function (sum, option) {
                                            return sum + option.votes[0].weight;
                                        }, 0);
                                    }
                                }
                            }
                    }
                    return base;
                },

                // Returns object with value and percent for this poll (for votes valid/invalid/cast only).
                getVote: function (type) {
                    if (!this.has_votes) {
                        // Return undefined if this poll has no votes.
                        return;
                    }

                    // Initial values
                    var value = '',
                        percentStr = '',
                        percentNumber,
                        vote,
                        config = Config.get('assignments_poll_100_percent_base').value;

                    switch (type) {
                        case 'votesinvalid':
                            vote = this.votesinvalid;
                            break;
                        case 'votesvalid':
                            vote = this.votesvalid;
                            break;
                        case 'votescast':
                            vote = this.votescast;
                            break;
                    }

                    // Check special values
                    switch (vote) {
                        case -1:
                            value = gettextCatalog.getString('majority');
                            break;
                        case -2:
                            value = gettextCatalog.getString('undocumented');
                            break;
                        default:
                            if (vote >= 0) {
                                value = vote;
                            } else {
                                value = 0; // value was not defined
                            }
                    }

                    // Calculate percent value
                    var base = this.getPercentBase(config, type);
                    if (base) {
                        percentNumber = Math.round(vote * 100 / (base) * 10) / 10;
                        percentStr = '(' + percentNumber + ' %)';
                    }
                    return {
                        'value': value,
                        'percentStr': percentStr,
                        'percentNumber': percentNumber,
                        'display': value + ' ' + percentStr
                    };
                }
            },
            relations: {
                belongsTo: {
                    'assignments/assignment': {
                        localField: 'assignment',
                        localKey: 'assignment_id',
                    }
                },
                hasMany: {
                    'assignments/polloption': {
                        localField: 'options',
                        foreignKey: 'poll_id',
                    }
                }
            },
        });
    }
])

.factory('AssignmentRelatedUser', [
    'DS',
    function (DS) {
        return DS.defineResource({
            name: 'assignments/relateduser',
            relations: {
                belongsTo: {
                    'users/user': {
                        localField: 'user',
                        localKey: 'user_id',
                    }
                }
            }
        });
    }
])

.factory('Assignment', [
    '$http',
    'DS',
    'Projector',
    'ProjectHelper',
    'AssignmentRelatedUser',
    'AssignmentPoll',
    'jsDataModel',
    'gettext',
    function ($http, DS, Projector, ProjectHelper, AssignmentRelatedUser, AssignmentPoll,
        jsDataModel, gettext) {
        var name = 'assignments/assignment';
        return DS.defineResource({
            name: name,
            useClass: jsDataModel,
            verboseName: gettext('Election'),
            verboseNamePlural: gettext('Elections'),
            methods: {
                getResourceName: function () {
                    return name;
                },
                getTitle: function () {
                    return this.title;
                },
                getAgendaTitle: function () {
                    return this.getTitle();
                },
                // link name which is shown in search result
                getSearchResultName: function () {
                    return this.getAgendaTitle();
                },
                // return true if a specific relation matches for given searchquery
                // (here: related_users/candidates)
                hasSearchResult: function (results) {
                    var assignment = this;
                    // search for related users (check if any user.id from already found users matches)
                    return _.some(results, function(result) {
                        if (result.getResourceName() === "users/user") {
                            if (_.some(assignment.assignment_related_users, {'user_id': result.id})) {
                                return true;
                            }
                        }
                    });
                },
                // override project function of jsDataModel factory
                project: function (projectorId, pollId) {
                    var isProjectedIds = this.isProjected(pollId);
                    var requestData = {
                        clear_ids: isProjectedIds,
                    };
                    if (_.indexOf(isProjectedIds, projectorId) == -1) {
                        requestData.prune = {
                            id: projectorId,
                            element: {
                                name: 'assignments/assignment',
                                id: this.id,
                                poll: pollId
                            },
                        };
                    }
                    return ProjectHelper.project(requestData);
                },
                // override isProjected function of jsDataModel factory
                isProjected: function (poll_id, anyPoll) {
                    // Returns the ids of all projectors with an element
                    // with the name 'assignments/assignment'. Else returns an empty list.
                    // Provide a poll_id to query a specific poll or set anyPoll to true, to
                    // query whether any poll (but not the assignment itself) is projected.
                    var self = this;
                    var predicate = function (element) {
                        var value;
                        if (typeof poll_id === 'undefined') {
                            // Assignment detail slide without poll
                            value = element.name == 'assignments/assignment' &&
                                typeof element.id !== 'undefined' &&
                                element.id == self.id &&
                                typeof element.poll === 'undefined';
                        } else if (anyPoll) {
                            // Assignment detail slide with any poll
                            value = element.name == 'assignments/assignment' &&
                                typeof element.id !== 'undefined' &&
                                element.id == self.id &&
                                typeof element.poll !== 'undefined';
                        } else {
                            // Assignment detail slide with specific poll
                            value = element.name == 'assignments/assignment' &&
                                typeof element.id !== 'undefined' &&
                                element.id == self.id &&
                                typeof element.poll !== 'undefined' &&
                                element.poll == poll_id;
                        }
                        return value;
                    };
                    var isProjectedIds = [];
                    Projector.getAll().forEach(function (projector) {
                        if (typeof _.findKey(projector.elements, predicate) === 'string') {
                            isProjectedIds.push(projector.id);
                        }
                    });
                    return isProjectedIds;
                },
                isRelatedProjected: function () {
                    var listOfSpeakers = [];
                    if (this.agenda_item) {
                        listOfSpeakers = this.agenda_item.isListOfSpeakersProjected();
                    }
                    return listOfSpeakers.concat(this.isProjected(null, true));
                },
            },
            relations: {
                belongsTo: {
                    'agenda/item': {
                        localKey: 'agenda_item_id',
                        localField: 'agenda_item',
                    }
                },
                hasMany: {
                    'core/tag': {
                        localField: 'tags',
                        localKeys: 'tags_id',
                    },
                    'assignments/relateduser': {
                        localField: 'assignment_related_users',
                        foreignKey: 'assignment_id',
                    },
                    'assignments/poll': {
                        localField: 'polls',
                        foreignKey: 'assignment_id',
                    }
                }
            },
            beforeInject: function (resource, instance) {
                AssignmentRelatedUser.ejectAll({where: {assignment_id: {'==': instance.id}}});
            }
        });
    }
])

.run(['Assignment', function(Assignment) {}]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.assignments.pdf', ['OpenSlidesApp.core.pdf'])

.factory('AssignmentContentProvider', [
    '$filter',
    'HTMLValidizer',
    'gettextCatalog',
    'PDFLayout',
    function($filter, HTMLValidizer, gettextCatalog, PDFLayout) {

        var createInstance = function(assignment) {

            // page title
            var title = PDFLayout.createTitle(assignment.title);
            var isElectedSemaphore = false;

            // open posts
            var createPreamble = function() {
                var preambleText = gettextCatalog.getString("Number of persons to be elected") + ": ";
                var memberNumber = ""+assignment.open_posts;
                var preamble = {
                    text: [
                        {
                            text: preambleText,
                            bold: true,
                            style: 'textItem'
                        },
                        {
                            text: memberNumber,
                            style: 'textItem'
                        }
                    ]
                };
                return preamble;
            };

            // description
            var createDescription = function() {
                if (assignment.description) {
                    var html = HTMLValidizer.validize(assignment.description);
                    var descriptionText = gettextCatalog.getString("Description") + ":";
                    var description = [
                        {
                            text: descriptionText,
                            bold: true,
                            style: 'textItem'
                        },
                        {
                            text: $(html).text(),
                            style: 'textItem',
                            margin: [10, 0, 0, 0]
                        }
                    ];
                    return description;
                } else {
                    return "";
                }
            };

            // show candidate list (if assignment phase is not 'finished')
            var createCandidateList = function() {
                if (assignment.phase != 2) {
                    var candidates = $filter('orderBy')(assignment.assignment_related_users, 'weight');
                    var candidatesText = gettextCatalog.getString("Candidates") + ": ";
                    var userList = [];

                    angular.forEach(candidates, function(assignmentsRelatedUser) {
                        userList.push({
                                text: assignmentsRelatedUser.user.get_full_name(),
                                margin: [0, 0, 0, 10],
                            }
                        );
                    });

                    var cadidateList = {
                        columns: [
                            {
                                text: candidatesText,
                                bold: true,
                                width: "25%",
                                style: 'textItem'
                            },
                            {
                                ul: userList,
                                style: 'textItem'
                            }
                        ]
                    };
                    return cadidateList;
                } else {
                    return "";
                }
            };

            // handles the case if a candidate is elected or not
            var electedCandidateLine = function(candidateName, pollOption, pollTableBody) {
                if (pollOption.is_elected) {
                    isElectedSemaphore = true;
                    return {
                        text: candidateName + "*",
                        bold: true,
                        style: PDFLayout.flipTableRowStyle(pollTableBody.length)
                    };
                } else {
                    return {
                        text: candidateName,
                        style: PDFLayout.flipTableRowStyle(pollTableBody.length)
                    };
                }
            };

            //creates the voting string for the result table and differentiates between special values
            var parseVoteValue = function(voteObject, printLabel) {
                var voteVal = "";
                if (voteObject) {
                    if (printLabel) {
                        voteVal += voteObject.label + ": ";
                    }
                    voteVal += voteObject.value;

                    if (voteObject.percentStr) {
                        voteVal += " " + voteObject.percentStr;
                    }
                }
                voteVal += "\n";
                return voteVal;
            };

            // creates the election result table
            var createPollResultTable = function() {
                var resultBody = [];
                angular.forEach(assignment.polls, function(poll, pollIndex) {
                    if (poll.published) {
                        var pollTableBody = [];

                        resultBody.push({
                            text: gettextCatalog.getString("Ballot") + " " + (pollIndex+1),
                            bold: true,
                            style: 'textItem',
                            margin: [0, 15, 0, 0]
                        });

                        pollTableBody.push([
                            {
                                text: gettextCatalog.getString("Candidates"),
                                style: 'tableHeader',
                            },
                            {
                                text: gettextCatalog.getString("Votes"),
                                style: 'tableHeader',
                            }
                        ]);

                        angular.forEach(poll.options, function(pollOption, optionIndex) {
                            var candidateName = pollOption.candidate.get_full_name();
                            var votes = pollOption.getVotes(); // 0 = yes, 1 = no, 2 = abstain
                            var tableLine = [];

                            tableLine.push(electedCandidateLine(candidateName, pollOption, pollTableBody));
                            if (poll.pollmethod == 'votes') {
                                tableLine.push(
                                    {
                                        text: parseVoteValue(votes[0], false),
                                        style: PDFLayout.flipTableRowStyle(pollTableBody.length)
                                    }
                                );
                            } else {
                                var resultBlock = [];
                                angular.forEach(votes, function(vote) {
                                    resultBlock.push(parseVoteValue(vote, true));
                                });
                                tableLine.push({
                                        text: resultBlock,
                                        style: PDFLayout.flipTableRowStyle(pollTableBody.length)
                                    }
                                );
                            }
                            pollTableBody.push(tableLine);
                        });

                        if (poll.votesvalid) {
                            pollTableBody.push([
                                {
                                    text: gettextCatalog.getString("Valid ballots"),
                                    style: 'tableConclude'
                                },
                                {
                                    text: parseVoteValue(poll.getVote('votesvalid'), false),
                                    style: 'tableConclude'
                                },
                            ]);
                        }

                        if (poll.votesinvalid) {
                            pollTableBody.push([
                                {
                                    text: gettextCatalog.getString("Invalid ballots"),
                                    style: 'tableConclude'
                                },
                                {
                                    text: parseVoteValue(poll.getVote('votesinvalid'), false),
                                    style: 'tableConclude'
                                },
                            ]);
                        }

                        if (poll.votescast) {
                            pollTableBody.push([
                                {
                                    text: gettextCatalog.getString("Casted ballots"),
                                    style: 'tableConclude'
                                },
                                {
                                    text: parseVoteValue(poll.getVote('votescast'), false),
                                    style: 'tableConclude'
                                },
                            ]);
                        }

                        var resultTableJsonSting = {
                            table: {
                                widths: ['64%','33%'],
                                headerRows: 1,
                                body: pollTableBody,
                            },
                            layout: 'headerLineOnly',
                        };

                        resultBody.push(resultTableJsonSting);
                    }
                });

                // add the legend to the result body
                if (assignment.polls.length > 0 && isElectedSemaphore) {
                    resultBody.push({
                        text: "* = " + gettextCatalog.getString("is elected"),
                        margin: [0, 5, 0, 0],
                    });
                }

                return resultBody;
            };

            var getContent = function() {
                return [
                    title,
                    createPreamble(),
                    createDescription(),
                    createCandidateList(),
                    createPollResultTable()
                ];
            };

            return {
                getContent: getContent,
                title: assignment.title
            };
        };

        return {
            createInstance: createInstance
        };
    }
])

.factory('BallotContentProvider', [
    '$q',
    '$filter',
    'gettextCatalog',
    'PDFLayout',
    'Config',
    'User',
    'ImageConverter',
    function($q, $filter, gettextCatalog, PDFLayout, Config, User, ImageConverter) {
        var createInstance = function(assignment, poll, pollNumber) {

            var logoBallotPaperUrl = Config.get('logo_pdf_ballot_paper').value.path;
            var imageMap = {};

            // PDF header
            var header = function() {
                var columns = [];

                // logo
                if (logoBallotPaperUrl) {
                    columns.push({
                        image: logoBallotPaperUrl,
                        fit: [90,20],
                        width: '20%'
                    });
                }
                var text = Config.get('general_event_name').value;
                columns.push({
                    text: text,
                    fontSize: 8,
                    alignment: 'right',
                });

                return {
                    color: '#555',
                    margin: [30, 10, 10, -10], // [left, top, right, bottom]
                    columns: columns,
                    columnGap: 10
                };
            };

            // page title
            var createTitle = function() {
                return {
                    text: assignment.title,
                    style: 'title',
                };
            };

            // poll description
            var createPollHint = function() {
                var description = poll.description ? ': ' + poll.description : '';
                return {
                    text: gettextCatalog.getString("Ballot") + " " + pollNumber + description,
                    style: 'description',
                };
            };

            // election entries
            var createYNBallotEntry = function(decision) {
                var YNColumn = [
                    {
                        width: "auto",
                        stack: [
                            PDFLayout.createBallotEntry(gettextCatalog.getString("Yes"))
                        ]
                    },
                    {
                        width: "auto",
                        stack: [
                            PDFLayout.createBallotEntry(gettextCatalog.getString("No"))
                        ]
                    },
                ];

                if (poll.pollmethod == 'yna') {
                    YNColumn.push({
                        width: "auto",
                        stack: [
                            PDFLayout.createBallotEntry(gettextCatalog.getString("Abstain"))
                        ]
                    });
                }

                return [
                    {
                        text: decision,
                        margin: [40, 10, 0, 0],
                    },
                    {
                        columns: YNColumn
                    }
                ];
            };

            var createSelectionField = function() {
                var candidates = $filter('orderBy')(poll.options, 'weight');
                var candidateBallotList = [];

                if (poll.pollmethod == 'votes') {
                    angular.forEach(candidates, function(option) {
                        var candidate = option.candidate.get_full_name();
                        candidateBallotList.push(PDFLayout.createBallotEntry(candidate));
                    });
                } else {
                    angular.forEach(candidates, function(option) {
                        var candidate;
                        if (option.candidate) {
                            candidate = option.candidate.get_full_name();
                        }
                        candidateBallotList.push(createYNBallotEntry(candidate));
                    });
                }
                return candidateBallotList;
            };

            var createSection = function(marginTop) {

                // since it is not possible to give a column a fixed height, we draw an "empty" column
                // with a one px width and a fixed top-margin
                return {
                    columns: [
                        {
                            width: 1,
                            margin: [0, marginTop],
                            text: '',
                        },
                        {
                            width: '*',
                            stack: [
                                header(),
                                createTitle(),
                                createPollHint(),
                                createSelectionField(),
                            ],
                        },
                    ]
                };
            };

            var createTableBody = function(numberOfRows, sheetend, maxballots) {
                var ballotstoprint = numberOfRows * 2;
                if (Number.isInteger(maxballots) && maxballots > 0 && maxballots < ballotstoprint) {
                    ballotstoprint = maxballots;
                }
                var tableBody = [];
                while (ballotstoprint > 1){
                    tableBody.push([createSection(sheetend), createSection(sheetend)]);
                    ballotstoprint -= 2;
                }
                if (ballotstoprint == 1) {
                    tableBody.push([createSection(sheetend), '']);
                }
                return tableBody;
            };

            var createContentTable = function() {
                // first, determine how many ballots we need
                var amount;
                var amount_method = Config.get('assignments_pdf_ballot_papers_selection').value;
                switch (amount_method) {
                    case 'NUMBER_OF_ALL_PARTICIPANTS':
                        amount = User.getAll().length;
                        break;
                    case 'NUMBER_OF_DELEGATES':
                        //TODO: assumption that DELEGATES is always group id 2. This may not be true
                        var group_id = 2;
                        amount = User.filter({where: {'groups_id': {contains:group_id} }}).length;
                        break;
                    case 'CUSTOM_NUMBER':
                        amount = Config.get('assignments_pdf_ballot_papers_number').value;
                        break;
                    default:
                        // should not happen.
                        amount = 0;
                }
                var tabledContent = [];
                var rowsperpage;
                var sheetend;
                if (poll.pollmethod == 'votes') {
                    if (poll.options.length <= 4) {
                        sheetend = 105;
                        rowsperpage = 4;
                    } else if (poll.options.length <= 8) {
                        sheetend = 140;
                        rowsperpage = 3;
                    } else if (poll.options.length <= 12) {
                        sheetend = 210;
                        rowsperpage = 2;
                    }
                    else { //works untill ~30 people
                        sheetend = 418;
                        rowsperpage = 1;
                    }
                } else {
                    if (poll.options.length <= 2) {
                        sheetend = 105;
                        rowsperpage = 4;
                    } else if (poll.options.length <= 4) {
                        sheetend = 140;
                        rowsperpage = 3;
                    } else if (poll.options.length <= 6) {
                        sheetend = 210;
                        rowsperpage = 2;
                    } else {
                        sheetend = 418;
                        rowsperpage = 1;
                    }
                }
                var page_entries = rowsperpage * 2;
                var fullpages = Math.floor(amount / page_entries);
                for (var i=0; i < fullpages; i++) {
                    tabledContent.push({
                        table: {
                            headerRows: 1,
                            widths: ['50%', '50%'],
                            body: createTableBody(rowsperpage, sheetend),
                            pageBreak: 'after'
                            },
                            layout: PDFLayout.getBallotLayoutLines(),
                            rowsperpage: rowsperpage
                        });
                }
                // fill the last page only partially
                var lastpage_ballots = amount - (fullpages * page_entries);
                if (lastpage_ballots < page_entries && lastpage_ballots > 0){
                    var partialpage = createTableBody(rowsperpage, sheetend, lastpage_ballots);
                    tabledContent.push({
                        table: {
                            headerRows: 1,
                            widths: ['50%', '50%'],
                            body: partialpage
                        },
                        layout: PDFLayout.getBallotLayoutLines(),
                        rowsperpage: rowsperpage
                    });
                }
                return tabledContent;
            };

            var getContent = function() {
                return createContentTable();
            };

            var getImageMap = function () {
                return imageMap;
            };

            return $q(function (resolve, reject) {
                var imageSources = [
                    logoBallotPaperUrl,
                ];
                ImageConverter.toBase64(imageSources).then(function (_imageMap) {
                    imageMap = _imageMap;
                    resolve({
                        getContent: getContent,
                        getImageMap: getImageMap,
                    });
                }, reject);
            });
        };

        return {
            createInstance: createInstance
        };
    }
])

.factory('AssignmentCatalogContentProvider', [
    'gettextCatalog',
    'PDFLayout',
    'Config',
    function(gettextCatalog, PDFLayout, Config) {

        var createInstance = function(allAssignments) {

            var title = PDFLayout.createTitle(
                    Config.translate(Config.get('assignments_pdf_title').value)
            );

            var createPreamble = function() {
                var preambleText = Config.get('assignments_pdf_preamble').value;
                if (preambleText) {
                    return {
                        text: preambleText,
                        style: "preamble"
                    };
                } else {
                    return "";
                }
            };

            var createTOContent = function(assignmentTitles) {
                var heading = {
                    text: gettextCatalog.getString("Table of contents"),
                    style: "heading2",
                };

                var toc = [];
                angular.forEach(assignmentTitles, function(title) {
                    toc.push({
                        text: title,
                        style: "tableofcontent"
                    });
                });

                return [
                    heading,
                    toc,
                    PDFLayout.addPageBreak()
                ];
            };

            var getContent = function() {
                var content = [];
                var assignmentContent = [];
                var assignmentTitles = [];

                _.forEach(allAssignments, function(assignment, key) {
                    assignmentTitles.push(assignment.title);
                    assignmentContent.push(assignment.getContent());
                    if (key < allAssignments.length - 1) {
                        assignmentContent.push(PDFLayout.addPageBreak());
                    }
                });

                content.push(title);
                content.push(createPreamble());
                content.push(createTOContent(assignmentTitles));
                content.push(assignmentContent);
                return content;
            };

            return {
                getContent: getContent
            };
        };

        return {
            createInstance: createInstance
        };
    }
])

.factory('AssignmentPdfExport', [
    'gettextCatalog',
    'AssignmentContentProvider',
    'AssignmentCatalogContentProvider',
    'PdfMakeDocumentProvider',
    'BallotContentProvider',
    'PdfMakeBallotPaperProvider',
    'PdfCreate',
    'Messaging',
    function (gettextCatalog, AssignmentContentProvider, AssignmentCatalogContentProvider,
        PdfMakeDocumentProvider, BallotContentProvider, PdfMakeBallotPaperProvider, PdfCreate,
        Messaging) {
        return {
            export: function (assignments, singleAssignment) {
                var filename = singleAssignment ?
                    gettextCatalog.getString('Election') + '_' + assignments.title :
                    gettextCatalog.getString('Elections');
                filename += '.pdf';
                if (singleAssignment) {
                    assignments = [assignments];
                }

                // Convert the assignments to content providers
                var assignmentContentProviderArray = _.map(assignments, function (assignment) {
                    return AssignmentContentProvider.createInstance(assignment);
                });

                var documentProviderPromise;
                if (singleAssignment) {
                    documentProviderPromise =
                        PdfMakeDocumentProvider.createInstance(assignmentContentProviderArray[0]);
                } else {
                    var assignmentCatalogContentProvider =
                        AssignmentCatalogContentProvider.createInstance(assignmentContentProviderArray);
                    documentProviderPromise =
                        PdfMakeDocumentProvider.createInstance(assignmentCatalogContentProvider);
                }
                documentProviderPromise.then(function (documentProvider) {
                    PdfCreate.download(documentProvider, filename);
                }, function (error) {
                    Messaging.addMessage(error.msg, 'error');
                });
            },
            createBallotPdf: function (assignment, pollId) {
                var thePoll;
                var pollNumber;
                angular.forEach(assignment.polls, function(poll, pollIndex) {
                    if (poll.id == pollId) {
                        thePoll = poll;
                        pollNumber = pollIndex+1;
                    }
                });
                var filename = gettextCatalog.getString('Ballot') + '_' + pollNumber + '_' + assignment.title + '.pdf';
                BallotContentProvider.createInstance(assignment, thePoll, pollNumber).then(function (ballotContentProvider) {
                    var documentProvider = PdfMakeBallotPaperProvider.createInstance(ballotContentProvider);
                    PdfCreate.download(documentProvider, filename);
                }, function (error) {
                    Messaging.addMessage(error.msg, 'error');
                });
            },
        };
    }
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.assignments.projector', ['OpenSlidesApp.assignments'])

.config([
    'slidesProvider',
    function(slidesProvider) {
        slidesProvider.registerSlide('assignments/assignment', {
            template: 'static/templates/assignments/slide_assignment.html',
        });
    }
])

.controller('SlideAssignmentCtrl', [
    '$scope',
    'Assignment',
    'AssignmentPhases',
    'User',
    function($scope, Assignment, AssignmentPhases, User) {
        // Attention! Each object that is used here has to be dealt on server side.
        // Add it to the coresponding get_requirements method of the ProjectorElement
        // class.
        var id = $scope.element.id;
        $scope.showResult = $scope.element.poll;

        Assignment.bindOne(id, $scope, 'assignment');
        $scope.phases = AssignmentPhases;
        User.bindAll({}, $scope, 'users');
    }
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.assignments.site', [
    'OpenSlidesApp.assignments',
    'OpenSlidesApp.core.pdf',
    'OpenSlidesApp.assignments.pdf',
    'OpenSlidesApp.poll.majority'
])

.config([
    'mainMenuProvider',
    'gettext',
    function (mainMenuProvider, gettext) {
        mainMenuProvider.register({
            'ui_sref': 'assignments.assignment.list',
            'img_class': 'pie-chart',
            'title': gettext('Elections'),
            'weight': 400,
            'perm': 'assignments.can_see'
        });
    }
])

.config([
    'SearchProvider',
    'gettext',
    function (SearchProvider, gettext) {
        SearchProvider.register({
            'verboseName': gettext('Elections'),
            'collectionName': 'assignments/assignment',
            'urlDetailState': 'assignments.assignment.detail',
            'weight': 400,
        });
    }
])

.config([
    '$stateProvider',
    'gettext',
    function($stateProvider, gettext) {
        $stateProvider
            .state('assignments', {
                url: '/assignments',
                abstract: true,
                template: "<ui-view/>",
                data: {
                    title: gettext('Elections'),
                    basePerm: 'assignments.can_see',
                },
            })
            .state('assignments.assignment', {
                abstract: true,
                template: "<ui-view/>",
            })
            .state('assignments.assignment.list', {})
            .state('assignments.assignment.detail', {
                controller: 'AssignmentDetailCtrl',
                resolve: {
                    assignmentId: ['$stateParams', function($stateParams) {
                        return $stateParams.id;
                    }],
                }
            })
            // redirects to assignment detail and opens assignment edit form dialog, uses edit url,
            // used by ui-sref links from agenda only
            // (from assignment controller use AssignmentForm factory instead to open dialog in front
            // of current view without redirect)
            .state('assignments.assignment.detail.update', {
                onEnter: ['$stateParams', '$state', 'ngDialog',
                    function($stateParams, $state, ngDialog) {
                        ngDialog.open({
                            template: 'static/templates/assignments/assignment-form.html',
                            controller: 'AssignmentUpdateCtrl',
                            className: 'ngdialog-theme-default wide-form',
                            closeByEscape: false,
                            closeByDocument: false,
                            resolve: {
                                assignmentId: function() {
                                    return $stateParams.id;
                                },
                            },
                            preCloseCallback: function() {
                                $state.go('assignments.assignment.detail', {assignment: $stateParams.id});
                                return true;
                            }
                        });
                    }
                ]
            });
    }
])

// Service for generic assignment form (create and update)
.factory('AssignmentForm', [
    'gettextCatalog',
    'operator',
    'Editor',
    'Mediafile',
    'Tag',
    'Assignment',
    'Agenda',
    'AgendaTree',
    function (gettextCatalog, operator, Editor, Mediafile, Tag, Assignment, Agenda, AgendaTree) {
        return {
            // ngDialog for assignment form
            getDialog: function (assignment) {
                return {
                    template: 'static/templates/assignments/assignment-form.html',
                    controller: (assignment) ? 'AssignmentUpdateCtrl' : 'AssignmentCreateCtrl',
                    className: 'ngdialog-theme-default wide-form',
                    closeByEscape: false,
                    closeByDocument: false,
                    resolve: {
                        assignmentId: function () {return assignment ? assignment.id : void 0;}
                    },
                };
            },
            // angular-formly fields for assignment form
            getFormFields: function (isCreateForm) {
                var images = Mediafile.getAllImages();
                var formFields = [
                {
                    key: 'title',
                    type: 'input',
                    templateOptions: {
                        label: gettextCatalog.getString('Title'),
                        required: true
                    }
                },
                {
                    key: 'description',
                    type: 'editor',
                    templateOptions: {
                        label: gettextCatalog.getString('Description')
                    },
                    data: {
                        ckeditorOptions: Editor.getOptions(images)
                    }
                },
                {
                    key: 'open_posts',
                    type: 'input',
                    templateOptions: {
                        label: gettextCatalog.getString('Number of persons to be elected'),
                        type: 'number',
                        min: 1,
                        required: true
                    }
                },
                {
                    key: 'poll_description_default',
                    type: 'input',
                    templateOptions: {
                        label: gettextCatalog.getString('Default comment on the ballot paper')
                    }
                }];

                // show as agenda item + parent item
                if (isCreateForm) {
                    formFields.push({
                        key: 'showAsAgendaItem',
                        type: 'checkbox',
                        templateOptions: {
                            label: gettextCatalog.getString('Show as agenda item'),
                            description: gettextCatalog.getString('If deactivated the election appears as internal item on agenda.')
                        },
                        hide: !(operator.hasPerms('assignments.can_manage') && operator.hasPerms('agenda.can_manage'))
                    });
                    formFields.push({
                        key: 'agenda_parent_id',
                        type: 'select-single',
                        templateOptions: {
                            label: gettextCatalog.getString('Parent item'),
                            options: AgendaTree.getFlatTree(Agenda.getAll()),
                            ngOptions: 'item.id as item.getListViewTitle() for item in to.options | notself : model.agenda_item_id',
                            placeholder: gettextCatalog.getString('Select a parent item ...')
                        },
                        hide: !operator.hasPerms('agenda.can_manage')
                    });
                }
                // more (with tags field)
                if (Tag.getAll().length > 0) {
                    formFields.push(
                        {
                            key: 'more',
                            type: 'checkbox',
                            templateOptions: {
                                label: gettextCatalog.getString('Show extended fields')
                            },
                            hide: !operator.hasPerms('assignments.can_manage')
                        },
                        {
                            template: '<hr class="smallhr">',
                            hideExpression: '!model.more'
                        },
                        {
                            key: 'tags_id',
                            type: 'select-multiple',
                            templateOptions: {
                                label: gettextCatalog.getString('Tags'),
                                options: Tag.getAll(),
                                ngOptions: 'option.id as option.name for option in to.options',
                                placeholder: gettextCatalog.getString('Select or search a tag ...')
                            },
                            hideExpression: '!model.more'
                        }
                    );
                }

                return formFields;
            }
        };
    }
])

// Cache for AssignmentPollDetailCtrl so that users choices are keeped during user actions (e. g. save poll form).
.value('AssignmentPollDetailCtrlCache', {})

// Child controller of AssignmentDetailCtrl for each single poll.
.controller('AssignmentPollDetailCtrl', [
    '$scope',
    'MajorityMethodChoices',
    'Config',
    'AssignmentPollDetailCtrlCache',
    'AssignmentPoll',
    function ($scope, MajorityMethodChoices, Config, AssignmentPollDetailCtrlCache, AssignmentPoll) {
        // Define choices.
        $scope.methodChoices = MajorityMethodChoices;
        // TODO: Get $scope.baseChoices from config_variables.py without copying them.

        // Setup empty cache with default values.
        if (typeof AssignmentPollDetailCtrlCache[$scope.poll.id] === 'undefined') {
            AssignmentPollDetailCtrlCache[$scope.poll.id] = {
                method: $scope.config('assignments_poll_default_majority_method'),
            };
        }

        // Fetch users choices from cache.
        $scope.method = AssignmentPollDetailCtrlCache[$scope.poll.id].method;

        $scope.recalculateMajorities = function (method) {
            $scope.method = method;
            _.forEach($scope.poll.options, function (option) {
                option.majorityReached = option.isReached(method);
            });
        };
        $scope.recalculateMajorities($scope.method);

        $scope.saveDescriptionChange = function (poll) {
            AssignmentPoll.save(poll);
        };

        // Save current values to cache on destroy of this controller.
        $scope.$on('$destroy', function() {
            AssignmentPollDetailCtrlCache[$scope.poll.id] = {
                method: $scope.method,
            };
        });
    }
])

.controller('AssignmentListCtrl', [
    '$scope',
    'ngDialog',
    'AssignmentForm',
    'Assignment',
    'Tag',
    'Agenda',
    'Projector',
    'ProjectionDefault',
    'gettextCatalog',
    'User',
    'osTableFilter',
    'osTableSort',
    'osTablePagination',
    'gettext',
    'AssignmentPhases',
    'AssignmentPdfExport',
    function($scope, ngDialog, AssignmentForm, Assignment, Tag, Agenda, Projector,
        ProjectionDefault, gettextCatalog, User, osTableFilter, osTableSort, osTablePagination,
        gettext, AssignmentPhases, AssignmentPdfExport) {
        $scope.$watch(function () {
            return Assignment.lastModified();
        }, function () {
            $scope.assignments = _.orderBy(Assignment.getAll(), ['title']);
        });
        Tag.bindAll({}, $scope, 'tags');
        $scope.$watch(function () {
            return Projector.lastModified();
        }, function () {
            var projectiondefault = ProjectionDefault.filter({name: 'assignments'})[0];
            if (projectiondefault) {
                $scope.defaultProjectorId = projectiondefault.projector_id;
            }
        });
        $scope.phases = AssignmentPhases;
        $scope.alert = {};

        // Filtering
        $scope.filter = osTableFilter.createInstance('AssignmentTableFilter');

        if (!$scope.filter.existsStorageEntry()) {
            $scope.filter.multiselectFilters = {
                tag: [],
                phase: [],
            };
        }
        $scope.filter.propertyList = ['title', 'description'];
        $scope.filter.propertyFunctionList = [
            function (assignment) {
                return gettextCatalog.getString($scope.phases[assignment.phase].display_name);
            },
        ];
        $scope.filter.propertyDict = {
            'assignment_related_users': function (candidate) {
                return candidate.user.get_short_name();
            },
            'tags': function (tag) {
                return tag.name;
            },
        };
        $scope.getItemId = {
            tag: function (assignment) {return assignment.tags_id;},
            phase: function (assignment) {return assignment.phase;},
        };

        // Sorting
        $scope.sort = osTableSort.createInstance('AssignmentTableSort');
        if (!$scope.sort.column) {
            $scope.sort.column = 'title';
        }
        $scope.sortOptions = [
            {name: 'agenda_item.getItemNumberWithAncestors()',
             display_name: gettext('Item')},
            {name: 'title',
             display_name: gettext('Title')},
            {name: 'phase',
             display_name: gettext('Phase')},
            {name: 'assignment_related_users.length',
             display_name: gettext('Number of candidates')},
        ];
        $scope.hasTag = function (assignment, tag) {
            return _.indexOf(assignment.tags_id, tag.id) > -1;
        };
        $scope.toggleTag = function (assignment, tag) {
            if ($scope.hasTag(assignment, tag)) {
                assignment.tags_id = _.filter(assignment.tags_id, function (tag_id){
                    return tag_id != tag.id;
                });
            } else {
                assignment.tags_id.push(tag.id);
            }
            Assignment.save(assignment);
        };

        // Pagination
        $scope.pagination = osTablePagination.createInstance('AssignmentTablePagination');

        // update phase
        $scope.updatePhase = function (assignment, phase_id) {
            assignment.phase = phase_id;
            Assignment.save(assignment);
        };
        // open new/edit dialog
        $scope.openDialog = function (assignment) {
            ngDialog.open(AssignmentForm.getDialog(assignment));
        };
        // *** select mode functions ***
        $scope.isSelectMode = false;
        // check all checkboxes
        $scope.checkAll = function () {
            $scope.selectedAll = !$scope.selectedAll;
            angular.forEach($scope.assignments, function (assignment) {
                assignment.selected = $scope.selectedAll;
            });
        };
        // uncheck all checkboxes if isSelectMode is closed
        $scope.uncheckAll = function () {
            if (!$scope.isSelectMode) {
                $scope.selectedAll = false;
                angular.forEach($scope.assignments, function (assignment) {
                    assignment.selected = false;
                });
            }
        };
        // delete all selected assignments
        $scope.deleteMultiple = function () {
            angular.forEach($scope.assignments, function (assignment) {
                if (assignment.selected)
                    Assignment.destroy(assignment.id);
            });
            $scope.isSelectMode = false;
            $scope.uncheckAll();
        };
        // delete single assignment
        $scope.delete = function (assignment) {
            Assignment.destroy(assignment.id);
        };
        // create the PDF List
        $scope.pdfExport = function () {
            AssignmentPdfExport.export($scope.assignmentsFiltered);
        };
    }
])

.controller('AssignmentDetailCtrl', [
    '$scope',
    '$http',
    '$filter',
    '$timeout',
    'filterFilter',
    'gettext',
    'ngDialog',
    'AssignmentForm',
    'operator',
    'Assignment',
    'User',
    'assignmentId',
    'Projector',
    'ProjectionDefault',
    'gettextCatalog',
    'AssignmentPhases',
    'AssignmentPdfExport',
    'WebpageTitle',
    'ErrorMessage',
    function($scope, $http, $filter, $timeout, filterFilter, gettext, ngDialog, AssignmentForm, operator,
        Assignment, User, assignmentId, Projector, ProjectionDefault, gettextCatalog, AssignmentPhases,
        AssignmentPdfExport, WebpageTitle, ErrorMessage) {
        User.bindAll({}, $scope, 'users');
        var assignment = Assignment.get(assignmentId);
        Assignment.loadRelations(assignment, 'agenda_item');
        // This flag is for setting 'activeTab' to recently added (last) ballot tab.
        // Set this flag, if ballots are added/removed. When the next autoupdate comes
        // in, the tabset will be updated.
        var updateBallotTabsFlag = true;
        $scope.$watch(function () {
            return Projector.lastModified();
        }, function () {
            var projectiondefault = ProjectionDefault.filter({name: 'assignments'})[0];
            if (projectiondefault) {
                $scope.defaultProjectorId = projectiondefault.projector_id;
            }
        });
        $scope.$watch(function () {
            return Assignment.lastModified(assignmentId);
        }, function () {
            // setup sorting of candidates
            $scope.relatedUsersSorted = $filter('orderBy')(assignment.assignment_related_users, 'weight');
            $scope.assignment = Assignment.get(assignment.id);
            if (updateBallotTabsFlag) {
                $scope.activeTab = $scope.assignment.polls.length - 1;
                updateBallotTabsFlag = false;
            }
            WebpageTitle.updateTitle(gettextCatalog.getString('Election') + ' ' + $scope.assignment.title);
        });
        $scope.candidateSelectBox = {};
        $scope.phases = AssignmentPhases;
        $scope.alert = {};

        // open edit dialog
        $scope.openDialog = function () {
            ngDialog.open(AssignmentForm.getDialog($scope.assignment));
        };
        // add (nominate) candidate
        $scope.addCandidate = function (userId) {
            $http.post('/rest/assignments/assignment/' + assignmentId + '/candidature_other/', {'user': userId})
                .then(function (success){
                    $scope.alert.show = false;
                    $scope.candidateSelectBox = {};
                }, function (error){
                    $scope.alert = ErrorMessage.forAlert(error);
                    $scope.candidateSelectBox = {};
                });
        };
        // remove candidate
        $scope.removeCandidate = function (userId) {
            $http.delete('/rest/assignments/assignment/' + assignmentId + '/candidature_other/',
                    {headers: {'Content-Type': 'application/json'},
                     data: JSON.stringify({user: userId})})
                .then(function (success) {},
                    function (error) {
                        $scope.alert = ErrorMessage.forAlert(error);
                    }
                );
        };
        // add me (nominate self as candidate)
        $scope.addMe = function () {
            $http.post('/rest/assignments/assignment/' + assignmentId + '/candidature_self/', {}).then(
                function (success) {
                    $scope.alert.show = false;
                }, function (error) {
                    $scope.alert = ErrorMessage.forAlert(error);
                }
            );
        };
        // remove me (withdraw own candidature)
        $scope.removeMe = function () {
            $http.delete('/rest/assignments/assignment/' + assignmentId + '/candidature_self/').then(
                function (success) {
                    $scope.alert.show = false;
                }, function (error) {
                    $scope.alert = ErrorMessage.forAlert(error);
                }
            );
        };
        // check if current user is already a candidate (elected==false)
        $scope.isCandidate = function () {
            var check = $scope.assignment.assignment_related_users.map(function(candidate) {
                if (!candidate.elected) {
                    return candidate.user_id;
                }
            }).indexOf(operator.user.id);
            if (check > -1) {
                return true;
            } else {
                return false;
            }
        };
        // Sort all candidates
        $scope.treeOptions = {
            dropped: function () {
                var sortedCandidates = [];
                _.forEach($scope.relatedUsersSorted, function (user) {
                    sortedCandidates.push(user.id);
                });
                $http.post('/rest/assignments/assignment/' + assignmentId + '/sort_related_users/',
                    {related_users: sortedCandidates}
                );
            }
        };
        // update phase
        $scope.updatePhase = function (phase_id) {
            $scope.assignment.phase = phase_id;
            Assignment.save($scope.assignment);
        };
        // create new ballot
        $scope.createBallot = function () {
            $http.post('/rest/assignments/assignment/' + assignmentId + '/create_poll/').then(
                function (success) {
                    $scope.alert.show = false;
                    if (assignment.phase === 0) {
                        $scope.updatePhase(1);
                    }
                    updateBallotTabsFlag = true;
                }, function (error) {
                    $scope.alert = ErrorMessage.forAlert(error);
                }
            );
        };
        // delete ballot
        $scope.deleteBallot = function (poll) {
            poll.DSDestroy().then(
                function (success) {
                    $scope.activeTab = $scope.activeTab - 1;
                    updateBallotTabsFlag = true;
                }
            );
        };
        // edit poll dialog
        $scope.editPollDialog = function (poll, ballot) {
            ngDialog.open({
                template: 'static/templates/assignments/assignmentpoll-form.html',
                controller: 'AssignmentPollUpdateCtrl',
                className: 'ngdialog-theme-default',
                closeByEscape: false,
                closeByDocument: false,
                resolve: {
                    assignmentpollId: function () {return poll.id;},
                    ballot: function () {return ballot;},
                }
            });
        };
        // publish ballot
        $scope.togglePublishBallot = function (poll) {
            poll.DSUpdate({
                    assignment_id: assignmentId,
                    published: !poll.published,
            })
            .then(function (success) {
                $scope.alert.show = false;
            }, function (error) {
                $scope.alert = ErrorMessage.forAlert(error);
            });
        };

        // mark candidate as (not) elected
        $scope.markElected = function (user, reverse) {
            if (reverse) {
                $http.delete(
                    '/rest/assignments/assignment/' + assignmentId + '/mark_elected/', {
                        headers: {
                            'Content-Type': 'application/json'
                        },
                        data: JSON.stringify({user: user})
                });
            } else {
                $http.post('/rest/assignments/assignment/' + assignmentId + '/mark_elected/', {'user': user});
            }

        };

        // Creates the document as pdf
        $scope.pdfExport = function() {
            AssignmentPdfExport.export($scope.assignment, true);
        };
        // Creates the ballotpaper as pdf
        $scope.ballotpaperExport = function(pollId) {
            AssignmentPdfExport.createBallotPdf($scope.assignment, pollId);
        };

        // Just mark some vote value strings for translation.
        gettext('Yes');
        gettext('No');
        gettext('Abstain');
    }
])

.controller('AssignmentCreateCtrl', [
    '$scope',
    '$state',
    'Assignment',
    'AssignmentForm',
    'Agenda',
    'ErrorMessage',
    function($scope, $state, Assignment, AssignmentForm, Agenda, ErrorMessage) {
        $scope.model = {};
        // set default value for open posts form field
        $scope.model.open_posts = 1;
        // get all form fields
        $scope.formFields = AssignmentForm.getFormFields(true);
        // save assignment
        $scope.save = function(assignment, gotoDetailView) {
            assignment.agenda_type = assignment.showAsAgendaItem ? 1 : 2;
            // The attribute assignment.agenda_parent_id is set by the form, see form definition.
            Assignment.create(assignment).then(
                function (success) {
                    if (gotoDetailView) {
                        $state.go('assignments.assignment.detail', {id: success.id});
                    }
                    $scope.closeThisDialog();
                },
                function (error) {
                    $scope.alert = ErrorMessage.forAlert(error);
                }
            );
        };
    }
])

.controller('AssignmentUpdateCtrl', [
    '$scope',
    '$state',
    'Assignment',
    'AssignmentForm',
    'Agenda',
    'assignmentId',
    'ErrorMessage',
    function($scope, $state, Assignment, AssignmentForm, Agenda, assignmentId, ErrorMessage) {
        var assignment = Assignment.get(assignmentId);
        $scope.alert = {};
        // set initial values for form model by create deep copy of assignment object
        // so list/detail view is not updated while editing
        $scope.model = angular.copy(assignment);
        // get all form fields
        $scope.formFields = AssignmentForm.getFormFields();

        // save assignment
        $scope.save = function (assignment, gotoDetailView) {
            // inject the changed assignment (copy) object back into DS store
            Assignment.inject(assignment);
            // save changed assignment object on server
            Assignment.save(assignment).then(
                function(success) {
                    if (gotoDetailView) {
                        $state.go('assignments.assignment.detail', {id: success.id});
                    }
                    $scope.closeThisDialog();
                },
                function (error) {
                    // save error: revert all changes by restore
                    // (refresh) original assignment object from server
                    Assignment.refresh(assignment);
                    $scope.alert = ErrorMessage.forAlert(error);
                }
            );
        };
    }
])

.controller('AssignmentPollUpdateCtrl', [
    '$scope',
    '$filter',
    'gettextCatalog',
    'AssignmentPoll',
    'assignmentpollId',
    'ballot',
    'ErrorMessage',
    function($scope, $filter, gettextCatalog, AssignmentPoll, assignmentpollId, ballot, ErrorMessage) {
        // set initial values for form model by create deep copy of assignmentpoll object
        // so detail view is not updated while editing poll
        var assignmentpoll = angular.copy(AssignmentPoll.get(assignmentpollId));
        $scope.model = assignmentpoll;
        $scope.ballot = ballot;
        $scope.formFields = [];
        $scope.alert = {};

        // add dynamic form fields
        var options = $filter('orderBy')(assignmentpoll.options, 'weight');
        options.forEach(function(option) {
            var defaultValue;
            if (assignmentpoll.pollmethod == 'yna' || assignmentpoll.pollmethod == 'yn') {
                defaultValue = {};
                _.forEach(option.votes, function (vote) {
                    defaultValue[vote.value.toLowerCase()] = vote.weight;
                });

                var columnClass = (assignmentpoll.pollmethod === 'yna') ? 'col-xs-4' : 'col-xs-6';
                var columns = [
                    {
                        key: 'yes_' + option.candidate_id,
                        type: 'input',
                        className: columnClass + ' no-padding-left',
                        templateOptions: {
                            label: gettextCatalog.getString('Yes'),
                            type: 'number',
                            min: -2,
                            required: true
                        },
                        defaultValue: defaultValue.yes
                    },
                    {
                        key: 'no_' + option.candidate_id,
                        type: 'input',
                        className: columnClass + ' no-padding-left' +
                            (assignmentpoll.pollmethod === 'yn' ? ' no-padding-right' : ''),
                        templateOptions: {
                            label: gettextCatalog.getString('No'),
                            type: 'number',
                            min: -2,
                            required: true
                        },
                        defaultValue: defaultValue.no
                    }
                ];
                if (assignmentpoll.pollmethod == 'yna'){
                    columns.push({
                        key:'abstain_' + option.candidate_id,
                        type: 'input',
                        className: columnClass + ' no-padding-left no-padding-right',
                        templateOptions: {
                            label: gettextCatalog.getString('Abstain'),
                            type: 'number',
                            min: -2,
                            required: true
                        },
                        defaultValue: defaultValue.abstain
                    });
                }
                $scope.formFields.push({
                    noFormControl: true,
                    template: '<strong>' + option.candidate.get_full_name() + '</strong>'
                },
                {
                    className: 'row',
                    fieldGroup: columns,
                });
            } else { // votes method
                if (option.votes.length) {
                    defaultValue = option.votes[0].weight;
                }
                $scope.formFields.push({
                    key: 'vote_' + option.candidate_id,
                    type: 'input',
                    templateOptions: {
                        label: option.candidate.get_full_name(),
                        type: 'number',
                        min: -2,
                        required: true,
                    },
                    defaultValue: defaultValue
                });
            }
        });
        // add general form fields
        $scope.formFields.push(
                {
                    key: 'votesvalid',
                    type: 'input',
                    templateOptions: {
                        label: gettextCatalog.getString('Valid ballots'),
                        type: 'number',
                        min: -2,
                    }
                },
                {
                    key: 'votesinvalid',
                    type: 'input',
                    templateOptions: {
                        label: gettextCatalog.getString('Invalid ballots'),
                        type: 'number',
                        min: -2,
                    }
                },
                {
                    key: 'votescast',
                    type: 'input',
                    templateOptions: {
                        label: gettextCatalog.getString('Casted ballots'),
                        type: 'number',
                        min: -2,
                    }
                }
        );

        // save assignmentpoll
        $scope.save = function (poll) {
            var votes = [];
            if (assignmentpoll.pollmethod == 'yna') {
                assignmentpoll.options.forEach(function(option) {
                    votes.push({
                        "Yes": poll['yes_' + option.candidate_id],
                        "No": poll['no_' + option.candidate_id],
                        "Abstain": poll['abstain_' + option.candidate_id]
                    });
                });
            } else if (assignmentpoll.pollmethod == 'yn') {
                    assignmentpoll.options.forEach(function(option) {
                        votes.push({
                            "Yes": poll['yes_' + option.candidate_id],
                            "No": poll['no_' + option.candidate_id]
                            });
                        });
            } else {
                assignmentpoll.options.forEach(function(option) {
                    votes.push({
                        "Votes": poll['vote_' + option.candidate_id],
                    });
                });
            }
            // save change poll object on server
            poll.DSUpdate({
                assignment_id: poll.assignment_id,
                votes: votes,
                votesvalid: poll.votesvalid,
                votesinvalid: poll.votesinvalid,
                votescast: poll.votescast
            })
            .then(function(success) {
                $scope.alert.show = false;
                $scope.closeThisDialog();
            }, function (error) {
                $scope.alert = ErrorMessage.forAlert(error);
            });
        };
    }
])

//mark all assignment config strings for translation with Javascript
.config([
    'gettext',
    function (gettext) {
        gettext('Election method');
        gettext('Automatic assign of method');
        gettext('Always one option per candidate');
        gettext('Always Yes-No-Abstain per candidate');
        gettext('Always Yes/No per candidate');
        gettext('Elections');
        gettext('Ballot and ballot papers');
        gettext('The 100-%-base of an election result consists of');
        gettext('For Yes/No/Abstain per candidate and Yes/No per candidate the 100-%-base ' +
                'depends on the election method: If there is only one option per candidate, ' +
                'the sum of all votes of all candidates is 100 %. Otherwise for each ' +
                'candidate the sum of all votes is 100 %.');
        gettext('Yes/No/Abstain per candidate');
        gettext('Yes/No per candidate');
        gettext('All valid ballots');
        gettext('All casted ballots');
        gettext('Disabled (no percents)');
        gettext('Number of ballot papers (selection)');
        gettext('Number of all delegates');
        gettext('Number of all participants');
        gettext('Use the following custom number');
        gettext('Custom number of ballot papers');
        gettext('Required majority');
        gettext('Default method to check whether a candidate has reached the required majority.');
        gettext('Simple majority');
        gettext('Two-thirds majority');
        gettext('Three-quarters majority');
        gettext('Disabled');
        gettext('Put all candidates on the list of speakers');
        gettext('Title for PDF document (all elections)');
        gettext('Preamble text for PDF document (all elections)');
        //other translations
        gettext('Searching for candidates');
        gettext('Voting');
        gettext('Finished');
    }
]);

}());

(function () {

'use strict';

// The core module used for the OpenSlides site and the projector
angular.module('OpenSlidesApp.core', [
    'js-data',
    'gettext',
    'ngAnimate',
    'ngBootbox',
    'ngSanitize',  // TODO: only use this in functions that need it.
    'ngStorage',
    'ui.bootstrap',
    'ui.bootstrap.datetimepicker',
    'ui.tree',
    'pdf',
    'OpenSlidesApp-templates',
])

.config([
    'DSProvider',
    'DSHttpAdapterProvider',
    function(DSProvider, DSHttpAdapterProvider) {
        DSProvider.defaults.reapAction = 'none';
        DSProvider.defaults.basePath = '/rest';
        DSProvider.defaults.afterReap = function(model, items) {
            if (items.length > 5) {
                model.findAll({}, {bypassCache: true});
            } else {
                _.forEach(items, function (item) {
                    model.refresh(item[model.idAttribute]);
                });
            }
        };
        DSHttpAdapterProvider.defaults.forceTrailingSlash = true;
    }
])

.factory('ProjectorID', [
    function () {
        return function () {
            return /projector\/(\d+)\//.exec(location.pathname)[1];
        };
    }
])

.config([
    '$sessionStorageProvider',
    function ($sessionStorageProvider) {
        $sessionStorageProvider.setKeyPrefix('OpenSlides');
    }
])

.factory('autoupdate', [
    'DS',
    'REALM',
    'ProjectorID',
    '$q',
    '$timeout',
    'ErrorMessage',
    function (DS, REALM, ProjectorID, $q, $timeout, ErrorMessage) {
        var socket = null;
        var retryConnectCallbacks = [];

        var websocketProtocol;
        if (location.protocol == 'https:') {
            websocketProtocol = 'wss:';
        } else {
            websocketProtocol = 'ws:';
        }

        var websocketPath;
        if (REALM === 'site') {
          websocketPath = '/ws/site/';
        } else if (REALM === 'projector') {
          websocketPath = '/ws/projector/' + ProjectorID() + '/';
        } else {
          console.error('The constant REALM is not set properly.');
        }

        // Get a random retry timeout between 2000 and 5000 ms.
        var getTimeoutTime = function () {
            return Math.floor(Math.random() * 3000 + 2000);
        };

        /* The callbacks are invoked if the ws connection closed and this factory tries to
         * reconnect after 1 second. The callbacks should return a promise. If the promise
         * resolves, the retry-process is stopped, so the callback can indicate whether it
         * has managed the reconnecting different.*/
        var runRetryConnectCallbacks = function () {
            var callbackPromises = _.map(retryConnectCallbacks, function (callback) {
                return callback();
            });
            $q.all(callbackPromises).then(function (success) {
                ErrorMessage.clearConnectionError();
            }, function (error) {
                $timeout(runRetryConnectCallbacks, getTimeoutTime());
            });
        };

        var Autoupdate = {};
        Autoupdate.messageReceivers = [];
        // We use later a promise to defer the first message of the established ws connection.
        Autoupdate.firstMessageDeferred = $q.defer();
        Autoupdate.onMessage = function (receiver) {
            Autoupdate.messageReceivers.push(receiver);
        };
        Autoupdate.newConnect = function () {
            socket = new WebSocket(websocketProtocol + '//' + location.host + websocketPath);
            // Make shure the servers state hasn't changed: Send a whoami request. If no users is logged and
            // anonymous are deactivated, reboot the client in fact that the server has lost all login information.
            socket.onclose = function (event) {
                socket = null;
                if (event.code !== 1000) { // 1000 is a normal close, like the close on logout
                    ErrorMessage.setConnectionError();
                }
                $timeout(runRetryConnectCallbacks, getTimeoutTime());
            };
            socket.onmessage = function (event) {
                var dataList = [];
                try {
                    dataList = JSON.parse(event.data);
                    _.forEach(Autoupdate.messageReceivers, function (receiver) {
                        receiver(dataList);
                    });
                } catch(err) {
                    console.error(err);
                }
                // Check if the promise is not resolved yet.
                if (Autoupdate.firstMessageDeferred.promise.$$state.status === 0) {
                    Autoupdate.firstMessageDeferred.resolve();
                }
                ErrorMessage.clearConnectionError();
            };
        };
        Autoupdate.send = function (message) {
            if (socket) {
                socket.send(JSON.stringify(message));
            }
        };
        Autoupdate.closeConnection = function () {
            if (socket) {
                socket.close();
            }
            Autoupdate.firstMessageDeferred = $q.defer();
        };
        Autoupdate.registerRetryConnectCallback = function (callback) {
            retryConnectCallbacks.push(callback);
        };
        return Autoupdate;
    }
])

.factory('operator', [
    'User',
    'Group',
    function (User, Group) {
        var setUserCallbacks = [];
        var operator = {
            user: null,
            perms: [],
            isAuthenticated: function () {
                return !!this.user;
            },
            setUser: function(user_id, user_data) {
                if (user_id && user_data) {
                    operator.user = User.inject(user_data);
                } else {
                    operator.user = null;
                }
                operator.reloadPerms();
                _.forEach(setUserCallbacks, function (cb) {
                    cb(operator.user);
                });
            },
            // Returns true if the operator has at least one perm of the perms-list.
            hasPerms: function(perms) {
                if (typeof perms === 'string') {
                    perms = perms.split(' ');
                }
                return _.intersection(perms, operator.perms).length > 0;
            },
            reloadPerms: function () {
                if (operator.user) {
                    operator.perms = operator.user.getPerms();
                } else {
                    var defaultGroup = Group.get(1);
                    operator.perms = defaultGroup ? defaultGroup.permissions : [];
                }
            },
            // Returns true if the operator is a member of group.
            isInGroup: function(group) {
                var groups = operator.user.groups_id;
                if (groups.length === 0) {
                    groups = [1]; // Set the default group, if no other groups are set.
                }
                return _.indexOf(groups, group.id) > -1;
            },
            registerSetUserCallback: function (cb) {
                setUserCallbacks.push(cb);
            },
        };
        return operator;
    }
])

// gets all in OpenSlides available languages
.factory('Languages', [
    '$sessionStorage',
    '$ngBootbox',
    'gettext',
    'gettextCatalog',
    'OpenSlidesPlugins',
    function ($sessionStorage, $ngBootbox, gettext, gettextCatalog, OpenSlidesPlugins) {
        return {
            // get all available languages
            getLanguages: function () {
                var current = $sessionStorage.language;
                // Define here new languages...
                var languages = [
                    { code: 'en', name: 'English' },
                    { code: 'de', name: 'Deutsch' },
                    { code: 'fr', name: 'Français' },
                    { code: 'es', name: 'Español' },
                    { code: 'pt', name: 'Português' },
                    { code: 'cs', name: 'Čeština'},
                    { code: 'ru', name: 'русский'},
                ];
                angular.forEach(languages, function (language) {
                    if (language.code == current)
                        language.selected = true;
                });
                return languages;
            },
            // get detected browser language code
            getBrowserLanguage: function () {
                var lang = navigator.language || navigator.userLanguage;
                if (!navigator.language && !navigator.userLanguage) {
                    lang = 'en';
                } else {
                    if (lang.indexOf('-') !== -1)
                        lang = lang.split('-')[0];
                    if (lang.indexOf('_') !== -1)
                        lang = lang.split('_')[0];
                }
                return lang;
            },
            // set current language and return updated languages object array
            setCurrentLanguage: function (lang) {
                var languages = this.getLanguages();
                var plugins = OpenSlidesPlugins.getAll();
                angular.forEach(languages, function (language) {
                    language.selected = false;
                    if (language.code == lang) {
                        language.selected = true;
                        $sessionStorage.language = lang;
                        gettextCatalog.setCurrentLanguage(lang);
                        // Plugins
                        if (lang != 'en') {
                            gettextCatalog.loadRemote("static/i18n/" + lang + ".json").then(function (success) {
                                // translate ng-bootbox directives when the translations are available.
                                $ngBootbox.addLocale(lang, {
                                    OK: gettextCatalog.getString('OK'),
                                    CANCEL: gettextCatalog.getString('Cancel'),
                                    CONFIRM: gettextCatalog.getString('OK'), // Yes, 'OK' is the original string.
                                });
                                $ngBootbox.setLocale(lang);
                            });
                            // load language files from plugins
                            angular.forEach(plugins, function (plugin) {
                                if (plugin.languages.indexOf(lang) != -1) {
                                    gettextCatalog.loadRemote("static/i18n/" + plugin.name + '/' + lang + ".json");
                                }
                            });
                        }
                    }
                });
                return languages;
            }
        };
    }
])

// Hook into gettextCatalog to include custom translations by wrapping
// the getString method. The translations are stored in the config.
.decorator('gettextCatalog', [
    '$delegate',
    '$rootScope',
    function ($delegate, $rootScope) {
        var oldGetString = $delegate.getString;
        var customTranslations = {};

        $delegate.getString = function () {
            var translated = oldGetString.apply($delegate, arguments);
            if (customTranslations[translated]) {
                translated = customTranslations[translated];
            }
            return translated;
        };
        $delegate.setCustomTranslations = function (translations) {
            customTranslations = translations;
            $rootScope.$broadcast('gettextLanguageChanged');
        };

        return $delegate;
    }
])

.run([
    '$rootScope',
    'Config',
    'gettextCatalog',
    function ($rootScope, Config, gettextCatalog) {
        $rootScope.$watch(function () {
            return Config.lastModified('translations');
        }, function () {
            var translations = Config.get('translations');
            if (translations) {
                var customTranslations = {};
                _.forEach(translations.value, function (entry) {
                    customTranslations[entry.original] = entry.translation;
                });
                // Update all translate directives
                gettextCatalog.setCustomTranslations(customTranslations);
            }
        });
    }
])

// set browser language as default language for OpenSlides
.run([
    '$sessionStorage',
    'gettextCatalog',
    'Languages',
    function($sessionStorage, gettextCatalog, Languages) {
        // set detected browser language as default language (fallback: 'en')
        if ($sessionStorage.language) {
            Languages.setCurrentLanguage($sessionStorage.language);
        } else {
            Languages.setCurrentLanguage(Languages.getBrowserLanguage());
        }   
        // Set this to true for debug. Helps to find untranslated strings by
        // adding "[MISSING]:".
        gettextCatalog.debug = false;
    }
])

.factory('dsEject', [
    'DS',
    function (DS) {
        return function (collection, instance) {
            var Resource = DS.definitions[collection];
            if (Resource.relationList) {
                Resource.relationList.forEach(function (relationDef) {
                    if (relationDef.foreignKey && !relationDef.osProtectedRelation) {
                        var query = {};
                        query[relationDef.foreignKey] = instance[Resource.idAttribute];
                        Resource.getResource(relationDef.relation).ejectAll(query);
                    }
                });
            }
        };
    }
])

.run([
    'DS',
    'autoupdate',
    'dsEject',
    function (DS, autoupdate, dsEject) {
        // Handler for normal autoupdate messages.
        autoupdate.onMessage(function(dataList) {
            var dataListByCollection = _.groupBy(dataList, 'collection');
            _.forEach(dataListByCollection, function (list, key) {
                var changedElements = [];
                var deletedElements = [];
                var collectionString = key;
                _.forEach(list, function (data) {
                    // Uncomment this line for debugging to log all autoupdates:
                    // console.log("Received object: " + data.collection + ", " + data.id);

                    // Now handle autoupdate message but do not handle notify messages.
                    if (data.collection !== 'notify') {
                        // remove (=eject) object from local DS store
                        var instance = DS.get(data.collection, data.id);
                        if (instance) {
                            dsEject(data.collection, instance);
                        }
                        // check if object changed or deleted
                        if (data.action === 'changed') {
                            changedElements.push(data.data);
                        } else if (data.action === 'deleted') {
                            deletedElements.push(data.id);
                        } else {
                            console.error('Error: Undefined action for received object' +
                                '(' + data.collection + ', ' + data.id + ')');
                        }
                    }
                });
                // add (=inject) all given objects into local DS store
                if (changedElements.length > 0) {
                    DS.inject(collectionString, changedElements);
                }
                // delete (=eject) all given objects from local DS store
                // (note: js-data does not provide 'bulk eject' as for DS.inject)
                _.forEach(deletedElements, function(id) {
                    DS.eject(collectionString, id);
                });
            });
        });
    }
])

.factory('Notify', [
    'autoupdate',
    'operator',
    function (autoupdate, operator) {
        var anonymousTrackId;

        // Handler for notify messages.
        autoupdate.onMessage(function(dataList) {
            var dataListByCollection = _.groupBy(dataList, 'collection');
            _.forEach(dataListByCollection.notify, function (notifyItem) {
                // Check, if this current user (or anonymous instance) has send this notify.
                if (notifyItem.senderUserId) {
                    if (operator.user) { // User send to user
                        notifyItem.sendBySelf = (notifyItem.senderUserId === operator.user.id);
                    } else { // User send to anonymous
                        notifyItem.sendBySelf = false;
                    }
                } else {
                    if (operator.user) { // Anonymous send to user
                        notifyItem.sendBySelf = false;
                    } else { // Anonymous send to anonymous
                        notifyItem.sendBySelf = (notifyItem.anonymousTrackId === anonymousTrackId);
                    }
                }
                // notify registered receivers.
                _.forEach(callbackReceivers[notifyItem.name], function (item) {
                    item.fn(notifyItem);
                });
            });
        });

        var callbackReceivers = {};
        /* Structure of callbackReceivers:
         * event_name_one: [ {id:0, fn:fn}, {id:3, fn:fn} ],
         * event_name_two: [ {id:2, fn:fn} ],
         * */
        var idCounter = 0;
        var eventNameRegex = new RegExp('^[a-zA-Z0-9_-]+$');
        var externIdRegex = new RegExp('^[a-zA-Z0-9_-]+\/[0-9]+$');
        return {
            registerCallback: function (eventName, fn) {
                if (!eventNameRegex.test(eventName)) {
                    throw 'eventName should only consist of [a-zA-Z0-9_-]';
                } else if (typeof fn === 'function') {
                    var id = idCounter++;

                    if (!callbackReceivers[eventName]) {
                        callbackReceivers[eventName] = [];
                    }
                    callbackReceivers[eventName].push({
                        id: id,
                        fn: fn,
                    });
                    return eventName + '/' + id;
                } else {
                    throw 'fn should be a function.';
                }
            },
            deregisterCallback: function (externId) {
                if (externIdRegex.test(externId)){
                    var split = externId.split('/');
                    var eventName = split[0];
                    var id = parseInt(split[1]);
                    callbackReceivers[eventName] = _.filter(callbackReceivers[eventName], function (item) {
                        return item.id !== id;
                    });
                } else {
                    throw externId + ' is not a valid id';
                }
            },
            // variable length of parameters, just pass ids.
            deregisterCallbacks: function () {
                _.forEach(arguments, this.deregisterCallback);
            },
            notify: function(eventName, params, users, channels) {
                if (eventNameRegex.test(eventName)) {
                    if (!params || typeof params !== 'object') {
                        params = {};
                    }

                    var notifyItem = {
                        collection: 'notify',
                        name: eventName,
                        params: params,
                        users: users,
                        replyChannels: channels,
                    };
                    if (!operator.user) {
                        if (!anonymousTrackId) {
                            anonymousTrackId = Math.floor(Math.random()*1000000);
                        }
                        notifyItem.anonymousTrackId = anonymousTrackId;
                    }
                    autoupdate.send([notifyItem]);
                } else {
                    throw 'eventName should only consist of [a-zA-Z0-9_-]';
                }
            },
        };
    }
])

// Save the server time to the rootscope.
.run([
    '$http',
    '$rootScope',
    function ($http, $rootScope) {
        // Loads server time and calculates server offset
        $rootScope.serverOffset = 0;
        $http.get('/core/servertime/')
        .then(function(data) {
            $rootScope.serverOffset = Math.floor(Date.now() / 1000 - data.data);
        });
    }
])

.run([
    'Config',
    '$rootScope',
    function (Config, $rootScope) {
        $rootScope.config = function (key) {
            try {
                return Config.get(key).value;
            }
            catch(err) {
                return '';
            }
        };
    }
])

// Make the indexOf available in every scope; needed for the projectorbuttons
.run([
    '$rootScope',
    function ($rootScope) {
        $rootScope.inArray = function (array, value) {
            return _.indexOf(array, value) > -1;
        };
    }
])

// Put the Math object into every scope.
.run([
    '$rootScope',
    function ($rootScope) {
        $rootScope.Math = window.Math;
    }
])

// Template hooks
// Possible uses:
// 1. { id: 'myHookId', template: '<button>click me</button>' }
// 2. { id: 'myHookId', templateUrl: '/static/templates/plugin_name/my-hook.html' }
// 3. { id: 'myHookId' }
//
// Deprecated: Give the id with 'Id'. Please use 'id'.
//
// Option 3 is for just changing the scope (see below), but not the original content. This
// is usefull to alter a JS behavior, e.g. on a ng-click. In this case, override is false
// for this template hook.
//
// It is possible to provide a scope, that is merged into the surrounding scope.
// You can override functions or values of the surrounding scope by providing them:
// { id: 'hookId', template: '<button ng-click="customOrOverwritten()">click me</button>',
//   scope: {
//     customOrOverwritten: function () { /*Do something */ },
//   },
// }
// Or you provide a function that returns an object of functions/values to overwrite to
// get access to the scope merged in:
// { id: 'hookId', template: '<button ng-click="customFn()">click me</button>',
//   scope: function (scope) {
//     return {
//       customOrOverwritten: function () {
//         scope.value = /* change it */;
//       },
//     };
//   },
// }
//
// As a default, template hooks in flavour of option 1 and 2 override the content that was
// originally there. Provide 'override: false', to prevent overriding the original content.
.factory('templateHooks', [
    function () {
        var hooks = {};
        return {
            hooks: hooks,
            registerHook: function (hook) {
                // Deprecated: Set the new style 'id', if 'Id' is given.
                if (hook.id === void 0) {
                    hook.id = hook.Id;
                }

                if (hooks[hook.id] === void 0) {
                    hooks[hook.id] = [];
                }
                // set override default
                if (hook.override === void 0) {
                    hook.override = !!(hook.template || hook.templateUrl);
                }
                hooks[hook.id].push(hook);
            }
        };
    }
])

.directive('templateHook', [
    '$compile',
    '$http',
    '$q',
    '$templateCache',
    '$timeout',
    'templateHooks',
    function ($compile, $http, $q, $templateCache, $timeout, templateHooks) {
        return {
            restrict: 'E',
            template: '',
            link: function (scope, iElement, iAttr) {
                var hooks = templateHooks.hooks[iAttr.hookName];
                if (hooks) {
                    // Populate scopes
                    _.forEach(hooks, function (hook) {
                        var _scope = hook.scope;
                        // If it is a function, get the scope from the function and provide
                        // the original scope.
                        if (typeof hook.scope === 'function') {
                            _scope = hook.scope(scope);
                        }

                        _.forEach(_scope, function (value, key) {
                            scope[key] = value;
                        });
                    });

                    // Check, if at least one hook overrides the original content.
                    var override = _.some(hooks, function (hook) {
                        return hook.override;
                    });

                    // filter hooks, that does actually have a template
                    hooks = _.filter(hooks, function (hook) {
                        return hook.template || hook.templateUrl;
                    });

                    // Get all templates
                    var templates = _.map(hooks, function (hook) {
                        // Either a template (html given as string) or a templateUrl has
                        // to be given. If a scope is provided, the schope of this templateHook
                        // is populated with the given functions/values.
                        if (hook.template) {
                            return hook.template;
                        } else {
                            return $templateCache.get(hook.templateUrl);
                        }
                    });

                    // Wait for the dom to build up, so we can retrieve the inner html of iElement.
                    $timeout(function () {
                        var html = override ? '' : iElement.html();
                        if (templates.length) {
                            html += templates.join('');
                        }

                        iElement.empty();
                        iElement.append($compile(html)(scope));
                    });
                }
            }
        };
    }
])

/*
 * This places a projector button in the document.
 *
 * Example: <projector-button model="motion" default-projector.id="defPrId"
 *           arg="2" content="{{ 'project' | translate }}"></projector-button>
 * This button references to model (in this example 'motion'). Also a defaultProjectionId
 * has to be given. In the example it's a scope variable. The next two parameters are additional:
 *   - arg: Then the model.project and model.isProjected will be called with
 *          this argument (e. g.: model.project(2))
 *   - content: A text placed behind the projector symbol.
 */
.directive('projectorButton', [
    'Projector',
    function (Projector) {
        return {
            restrict: 'E',
            templateUrl: 'static/templates/projector-button.html',
            link: function (scope, element, attributes) {
                if (!attributes.model) {
                    throw 'A model has to be given!';
                } else if (!attributes.defaultProjectorId) {
                    throw 'A default-projector-id has to be given!';
                }

                Projector.bindAll({}, scope, 'projectors');

                scope.$watch(attributes.model, function (model) {
                    scope.model = model;
                });

                scope.$watch(attributes.defaultProjectorId, function (defaultProjectorId) {
                    scope.defaultProjectorId = defaultProjectorId;
                });

                if (attributes.arg) {
                    scope.$watch(attributes.arg, function (arg) {
                        scope.arg = arg;
                    });
                }

                scope.content = '';
                if (attributes.content) {
                    attributes.$observe('content', function (content) {
                        scope.content = content;
                    });
                }
            }
        };
    }
])

.factory('jsDataModel', [
    '$http',
    'Projector',
    'ProjectHelper',
    function($http, Projector, ProjectHelper) {
        var BaseModel = function() {};
        BaseModel.prototype.project = function(projectorId) {
            // if this object is already projected on projectorId, delete this element from this projector
            var isProjectedIds = this.isProjected();
            var requestData = {
                clear_ids: isProjectedIds,
            };
            // Show the element, if it was not projected before on the given projector
            if (_.indexOf(isProjectedIds, projectorId) == -1) {
                requestData.prune = {
                    id: projectorId,
                    element: {name: this.getResourceName(), id: this.id},
                };
            }
            return ProjectHelper.project(requestData);
        };
        BaseModel.prototype.isProjected = function() {
            // Returns the ids of all projectors if there is a projector element
            // with the same name and the same id. Else returns an empty list.
            var self = this;
            var predicate = function (element) {
                return element.name == self.getResourceName() &&
                    typeof element.id !== 'undefined' &&
                    element.id == self.id;
            };
            var isProjectedIds = [];
            Projector.getAll().forEach(function (projector) {
                if (typeof _.findKey(projector.elements, predicate) === 'string') {
                    isProjectedIds.push(projector.id);
                }
            });
            return isProjectedIds;
        };
        // Override this method to get object specific behavior
        BaseModel.prototype.isRelatedProjected = function() {
            throw "needs to be implemented!";
        };
        return BaseModel;
    }
])

.factory('ErrorMessage', [
    '$timeout',
    'gettextCatalog',
    'Messaging',
    function ($timeout, gettextCatalog, Messaging) {
        return {
            forAlert: function (error) {
                var message = gettextCatalog.getString('Error') + ': ';

                if (!error.data) {
                    message += gettextCatalog.getString("The server didn't respond.");
                } else if (error.data.detail) {
                    message += error.data.detail;
                } else if (error.status > 500) { // Some kind of server error.
                    message += gettextCatalog.getString("A server error occurred (%%code%%). Please check the system logs.");
                    message = message.replace('%%code%%', error.status);
                } else {
                    for (var e in error.data) {
                        message += e + ': ' + error.data[e] + ' ';
                    }
                }
                return { type: 'danger', msg: message, show: true };
            },
            setConnectionError: function () {
                $timeout(function () {
                    Messaging.createOrEditMessage(
                        'connectionLostMessage',
                        gettextCatalog.getString('Offline mode: You can use OpenSlides but changes are not saved.'),
                        'warning',
                        {noClose: true});
                }, 1);
            },
            clearConnectionError: function () {
                $timeout(function () {
                    Messaging.deleteMessage('connectionLostMessage');
                }, 1);
            },
        };
    }
])

/* Messaging factory. The text is html-binded into the document, so you can
 * provide also html markup for the messages. There are 4 types: 'info',
 * 'success', 'warning', 'error'. The timeout is for autodeleting the message.
 * Args that could be provided:
 * - timeout: Milliseconds until autoclose the message (default: not set, no auto close)
 * - noClose: Whether to show the close button (default: false)
 */
.factory('Messaging', [
    '$timeout',
    function($timeout) {
        var callbackList = [],
            messages = {},
            idCounter = 0;

        var onChange = function () {
            _.forEach(callbackList, function (callback) {
                callback();
            });
        };

        return {
            addMessage: function (text, type, args) {
                var id = idCounter++;
                return this.createOrEditMessage(id, text, type, args);
            },
            createOrEditMessage: function (id, text, type, args) {
                if (!args) {
                    args = {};
                }
                if (messages[id] && messages[id].timeout) {
                    $timeout.cancel(messages[id].timeout);
                }
                messages[id] = {
                    text: text,
                    type: type,
                    id: id,
                    args: args,
                };
                if (typeof args.timeout === 'number' && args.timeout > 0) {
                    var self = this;
                    messages[id].timeout = $timeout(function () {
                        self.deleteMessage(id);
                    }, args.timeout);
                }
                onChange();
                return id;
            },
            deleteMessage: function (id) {
                delete messages[id];
                onChange();
            },
            getMessages: function () {
                return messages;
            },
            registerMessageChangeCallback: function (fn) {
                if (typeof fn === 'function') {
                    callbackList.push(fn);
                } else {
                    throw 'fn has to be a function';
                }
            },
        };
    }
])

.factory('Logos', [
    'Config',
    'gettext',
    function (Config, gettext) {
        return {
            getKeys: function () {
                return Config.get('logos_available').value;
            },
            getAll: function () {
                var self = this;
                return _.map(this.getKeys(), function (key) {
                    return self.get(key);
                });
            },
            get: function (key) {
                var config = Config.get(key);
                if (config) {
                    config.value.key = key;
                    return config.value;
                }
            },
            set: function (key, path) {
                var config = Config.get(key);
                if (config) {
                    config.value.path = path || '';
                    Config.save(key);
                }
            },
        };
    }
])

.factory('Fonts', [
    'Config',
    'gettext',
    function (Config, gettext) {
        var extensionFormatMap = {
            'ttf': 'truetype',
            'woff': 'woff',
        };

        return {
            getKeys: function () {
                return Config.get('fonts_available').value;
            },
            getAll: function () {
                var self = this;
                return _.map(this.getKeys(), function (key) {
                    return self.get(key);
                });
            },
            get: function (key) {
                var config = Config.get(key);
                if (config) {
                    config.value.key = key;
                    return config.value;
                }
            },
            getUrl: function (key) {
                var font = this.get(key);
                if (font) {
                    var path = font.path;
                    if (!path) {
                        return font.default;
                    }
                    return path;
                }
            },
            getForCss: function (key) {
                var url = this.getUrl(key);
                if (url) {
                    var ext = _.last(url.split('.'));
                    return "url('" + url + "') format('" +
                        extensionFormatMap[ext] + "')";
                }
            },
            set: function (key, path) {
                var config = Config.get(key);
                if (config) {
                    config.value.path = path || '';
                    Config.save(key);
                }
            },
        };
    }
])

.factory('Tag', [
    'DS',
    function(DS) {
        return DS.defineResource({
            name: 'core/tag',
        });
    }
])

.factory('Config', [
    '$http',
    'gettextCatalog',
    'DS',
    function($http, gettextCatalog, DS) {
        var configOptions;
        return DS.defineResource({
            name: 'core/config',
            idAttribute: 'key',
            translate: function (value) {
                return gettextCatalog.getString(value);
            }
        });
    }
])

.factory('ChatMessage', [
    'DS',
    function(DS) {
        return DS.defineResource({
            name: 'core/chat-message',
            relations: {
                belongsTo: {
                    'users/user': {
                        localField: 'user',
                        localKey: 'user_id',
                    }
                }
            }
        });
    }
])

/*
 * Provides a function for plugins to register as new plugin.
 *
 * Get all registerd plugins via 'OpenSlidesPlugins.getAll()'.
 *
 * Example code for plugins:
 *
 *  .config([
 *      'OpenSlidesPluginsProvider',
 *       function(OpenSlidesPluginsProvider) {
 *          OpenSlidesPluginsProvider.registerPlugin({
 *              name: 'openslides_votecollector',
 *              display_name: 'VoteCollector',
 *              languages: ['de']
 *          });
 *      }
 *  ])
 */
.provider('OpenSlidesPlugins', [
    function () {
        var provider = this;
        provider.plugins = [];
        provider.registerPlugin = function (plugin) {
            provider.plugins.push(plugin);
        };
        provider.$get = [
            function () {
                return {
                    getAll: function () {
                        return provider.plugins;
                    }
                };
            }
        ];
    }
])


// Configs for CKEditor which has to set while startup of OpenSlides
.config(
    function() {
        CKEDITOR.disableAutoInline = true;
    }
)

// Options for CKEditor used in various create and edit views.
// Required in core/base.js because MotionComment factory which used this
// factory has to placed in motions/base.js.
.factory('Editor', [
    'gettextCatalog',
    function (gettextCatalog) {
        var extraPlugins = [];
        return {
            registerDialog: function (name, dialog) {
                CKEDITOR.dialog.add(name, dialog);
            },
            registerPlugin: function (name, plugin) {
                CKEDITOR.plugins.add(name, plugin);
                extraPlugins.push(name);
            },
            /* Provide special keyword in the arguments for a special behaviour:
             * Example: getOptions('inline', 'YOffset')
             * Available keywords:
             *  - inline: smaller toolbar
             *  - YOffset: move the editor toolbar 40px up
             */
            getOptions: function () {
                var extraPluginsString = 'colorbutton,colordialog,find,sourcedialog,justify,showblocks';
                var registeredPluginsString = extraPlugins.join(',');
                if (registeredPluginsString) {
                    extraPluginsString += ',' + registeredPluginsString;
                }
                var options = {
                    on: {
                        instanceReady: function() {
                            // This adds a listener to ckeditor to remove unwanted blank lines on import.
                            // Clipboard content varies heavily in structure and html code, depending on the "sender".
                            // Here it is first parsed into a pseudo-DOM (two lines taken from a ckeditor
                            // paste example on the ckeditor site).
                            this.on('paste', function(evt) {
                                if (evt.data.type == 'html') {
                                    var fragment = CKEDITOR.htmlParser.fragment.fromHtml(evt.data.dataValue);
                                    var writer = new CKEDITOR.htmlParser.basicWriter();
                                    // html content will now be in a dom-like structure inside 'fragment'.
                                    this.filter.applyTo(fragment);
                                    if (fragment.children) {
                                        // If this fragment is DOM-like, it may contain nested properties
                                        // (being html nodes). Traverse the children and check if it is a
                                        // child only containing empty <br> or <p>.
                                        // new_content_children will finally contain all nodes that are
                                        // not empty.
                                        var new_content_children = [];
                                        _.forEach(fragment.children, function (child) {
                                            var empty = true;
                                            if (child.children){
                                                _.forEach(child.children, function(grandchild) {
                                                    if (grandchild.name != 'p' && grandchild.name != 'br') {
                                                        empty = false;
                                                    } else if (grandchild.isEmpty !== true) {
                                                        empty = false;
                                                    }
                                                });
                                                if (empty === false) {
                                                    new_content_children.push(child);
                                                }
                                            } else {
                                                if (child.name != 'p' && child.name != 'br' &&
                                                    child.isEmpty !== true){
                                                    new_content_children.push(child);
                                                }
                                            }
                                        });
                                        fragment.children = new_content_children;
                                    }
                                    fragment.writeHtml(writer);
                                    // Return the re-created fragment without the empty <p> and <br> into the
                                    // editor import processing (same as at the begin of the function: by ckeditor)
                                    evt.data.dataValue = writer.getHtml();
                                }
                            });
                        }
                    },
                    customConfig: '',
                    floatSpaceDockedOffsetY: _.indexOf(arguments, 'YOffset') > -1 ? 35 : 0,
                    disableNativeSpellChecker: false,
                    language_list: [
                        'fr:français',
                        'es:español',
                        'pt:português',
                        'en:english',
                        'de:deutsch',
                        'cs:čeština'],
                    language: gettextCatalog.getCurrentLanguage(),
                    allowedContent:
                        'h1 h2 h3 b i u strike sup sub strong em;' +
                        'blockquote p pre table' +
                        '(text-align-left,text-align-center,text-align-right,text-align-justify,os-split-before,os-split-after){text-align, float, padding};' +
                        'a[!href];' +
                        'img[!src,alt]{width,height,float, padding};' +
                        'tr th td caption;' +
                        'li(os-split-before,os-split-after); ol(os-split-before,os-split-after)[start]{list-style-type};' +
                        'ul(os-split-before,os-split-after){list-style};' +
                        'span[!*]{color,background-color}(os-split-before,os-split-after,os-line-number,line-number-*);' +
                        'br(os-line-break);',

                    // there seems to be an error in CKeditor that parses spaces in extraPlugins as part of the plugin name.
                    extraPlugins: extraPluginsString,
                    removePlugins: 'wsc,scayt,a11yhelp,filebrowser,sourcearea,liststyle,tabletools,tableselection,contextmenu,image',
                    removeButtons: 'Scayt,Anchor,Styles,HorizontalRule',
                };
                if (_.indexOf(arguments, 'inline') > -1) {
                    options.toolbarGroups = [
                        { name: 'basicstyles', groups: [ 'basicstyles', 'cleanup' ] },
                        { name: 'colors', groups: [ 'colors' ] },
                        { name: 'paragraph', groups: [ 'list'] },
                        { name: 'links', groups: [ 'links' ] },
                        { name: 'clipboard', groups: [ 'undo' ] },
                        { name: 'document', groups: [ 'mode' ] },
                    ];
                    options.removeButtons = 'Underline,Subscript,Superscript,PasteFromWord,PasteText,Scayt,Link,Unlink,Anchor,HorizontalRule,Table,Image,Maximize,Source,Format,About,Paste,Cut,Copy';
                } else {
                    options.toolbarGroups = [
                        { name: 'clipboard', groups: [ 'clipboard', 'undo' ] },
                        { name: 'editing', groups: [ 'find', 'selection', 'spellchecker', 'editing' ] },
                        { name: 'links', groups: [ 'links' ] },
                        { name: 'insert', groups: [ 'insert' ] },
                        { name: 'tools', groups: [ 'tools' ] },
                        { name: 'document', groups: [ 'mode' ] },
                        '/',
                        { name: 'styles', groups: [ 'styles' ] },
                        { name: 'basicstyles', groups: [ 'basicstyles', 'cleanup' ] },
                        { name: 'colors', groups: [ 'colors' ] },
                        { name: 'paragraph', groups: [ 'list', 'indent' ] },
                        { name: 'align'},
                        { name: 'paragraph', groups: [ 'blocks' ] }
                    ];
                }
                return options;
            }
        };
    }
])

/* Model for a projector.
 *
 * At the moment we use only one projector, so there will be only one object
 * in this model. It has the id 1. For later releases there will be multiple
 * projector objects.
 *
 * This model uses onConfilict: 'replace' instead of 'merge'. This is necessary
 * because the keys of the projector objects can change and old keys have to
 * be removed. See http://www.js-data.io/docs/dsdefaults#onconflict for
 * more information.
 */
.factory('Projector', [
    'DS',
    '$http',
    'EditForm',
    'Config',
    function(DS, $http, EditForm, Config) {
        return DS.defineResource({
            name: 'core/projector',
            onConflict: 'replace',
            relations: {
                hasMany: {
                    'core/projection-default': {
                        localField: 'projectiondefaults',
                        foreignKey: 'projector_id',
                    }
                },
            },
            methods: {
                controlProjector: function(action, direction) {
                    $http.post('/rest/core/projector/' + this.id + '/control_view/',
                            {"action": action, "direction": direction}
                    );
                },
                getFormOrStateForCurrentSlide: function () {
                    var return_dict;
                    angular.forEach(this.elements, function(value, key) {
                        if (value.name == 'agenda/list-of-speakers') {
                            return_dict = {
                                state: 'agenda.item.detail',
                                id: value.id,
                            };
                        } else if (
                            // TODO:
                            // Find generic solution for whitelist in getFormOrStateForCurrentSlide
                            // see https://github.com/OpenSlides/OpenSlides/issues/3130
                            value.name === 'topics/topic' ||
                            value.name === 'motions/motion' ||
                            value.name === 'motions/motion-block' ||
                            value.name === 'assignments/assignment' ||
                            value.name === 'mediafiles/mediafile' ||
                            value.name === 'users/user') {
                                return_dict = {
                                    form: EditForm.fromCollectionString(value.name),
                                    id: value.id,
                                };
                        }
                    });
                    return return_dict;
                },
                toggleBlank: function () {
                    $http.post('/rest/core/projector/' + this.id + '/control_blank/',
                        !this.blank
                    );
                },
                toggleBroadcast: function () {
                    $http.post('/rest/core/projector/' + this.id + '/broadcast/');
                }
            },
        });
    }
])

// This factory sends a request to /rest/core/projectors/project
// with the given data. Also it does the changes done by the server
// locally and may reverts them, if something went wrong.
.factory('ProjectHelper', [
    '$q',
    '$http',
    'Projector',
    function ($q, $http, Projector) {
        var uuid4 = function () {
            function s8() {
                return Math.floor((1 + Math.random()) * 0x100000000)
                    .toString(16)
                    .substring(1);
            }
            return s8() + s8() + s8() + s8();
        };

        return {
            project: function (data) {
                var projector;
                // get all projectors that will be changed.
                var projectorsChanged = _.filter(_.map(data.clear_ids, function (id) {
                    return Projector.get(id);
                }));
                if (data.prune && !_.includes(data.clear_ids, data.prune.id)) {
                    projector = Projector.get(data.prune.id);
                    if (projector) {
                        projectorsChanged.push(projector);
                    }
                }

                // copy original projectors in case we have to reconstruct those
                // _.cloneDeep and angular.clone does not work here; I'm not
                // exactly sure why..
                var originalProjectors = _.map(projectorsChanged, function (projector) {
                    var elements = {};
                    _.forEach(projector.elements, function (element, key) {
                        elements[key] = _.cloneDeep(element);
                    });
                    return {
                        id: projector.id,
                        elements: elements,
                        scroll: projector.scroll,
                        scale: projector.scale,
                        name: projector.name,
                        blank: projector.blank,
                        width: projector.width,
                        height: projector.height,
                        projectiondefaults: _.cloneDeep(projector.projectiondefaults),
                    };
                });

                // Clear every projector
                _.forEach(projectorsChanged, function (projector) {
                    var elements = {};
                    _.forEach(projector.elements, function (element, key) {
                        if (element.stable) {
                            elements[key] = element;
                        }
                    });
                    projector.elements = elements;
                });

                // Add the prune element if given
                if (data.prune) {
                    projector = _.find(projectorsChanged, function (projector) {
                        return projector.id === data.prune.id;
                    });
                    if (projector) {
                        projector.scroll = 0;
                        projector.elements[uuid4()] = data.prune.element;
                    }
                }

                Projector.inject(projectorsChanged);

                return $q(function (resolve, reject) {
                    $http.post('/rest/core/projector/project/', data).then(function (success) {
                        resolve(success);
                    }, function (error) {
                        // revert the changed made earlier
                        Projector.inject(originalProjectors);
                        reject(error);
                    });
                });
            },
        };
    }
])

/* Model for all projection defaults */
.factory('ProjectionDefault', [
    'DS',
    function(DS) {
        return DS.defineResource({
            name: 'core/projection-default',
            relations: {
                belongsTo: {
                    'core/projector': {
                        localField: 'projector',
                        localKey: 'projector_id',
                    }
                }
            }
        });
    }
])

/* Model for ProjectorMessages */
.factory('ProjectorMessage', [
    'DS',
    'jsDataModel',
    'gettext',
    '$http',
    'Projector',
    function(DS, jsDataModel, gettext, $http, Projector) {
        var name = 'core/projector-message';
        return DS.defineResource({
            name: name,
            useClass: jsDataModel,
            verboseName: gettext('Message'),
            verbosenamePlural: gettext('Messages'),
            methods: {
                getResourceName: function () {
                    return name;
                },
                // Override the BaseModel.project function
                project: function(projectorId) {
                    // if this object is already projected on projectorId, delete this element from this projector
                    var isProjectedIds = this.isProjected();
                    var self = this;
                    var predicate = function (element) {
                        return element.name === name && element.id === self.id;
                    };
                    _.forEach(isProjectedIds, function (id) {
                        var uuid = _.findKey(Projector.get(id).elements, predicate);
                        $http.post('/rest/core/projector/' + id + '/deactivate_elements/', [uuid]);
                    });
                    // if it was the same projector before, just delete it but not show again
                    if (_.indexOf(isProjectedIds, projectorId) == -1) {
                        // Now check whether other messages are already projected and delete them
                        var elements = Projector.get(projectorId).elements;
                        _.forEach(elements, function (element, uuid) {
                            if (element.name === name) {
                                $http.post('/rest/core/projector/' + projectorId + '/deactivate_elements/', [uuid]);
                            }
                        });
                        return $http.post(
                            '/rest/core/projector/' + projectorId + '/activate_elements/',
                            [{name: name, id: self.id, stable: true}]
                        );
                    }
                },
            }
        });
    }
])

/* Model for Countdowns */
.factory('Countdown', [
    'DS',
    'jsDataModel',
    'gettext',
    '$rootScope',
    '$http',
    'Projector',
    function(DS, jsDataModel, gettext, $rootScope, $http, Projector) {
        var name = 'core/countdown';
        return DS.defineResource({
            name: name,
            useClass: jsDataModel,
            verboseName: gettext('Countdown'),
            verbosenamePlural: gettext('Countdowns'),
            methods: {
                getResourceName: function () {
                    return name;
                },
                start: function () {
                    // calculate end point of countdown (in seconds!)
                    var endTimestamp = Date.now() / 1000 - $rootScope.serverOffset + this.countdown_time;
                    this.running = true;
                    this.countdown_time = endTimestamp;
                    DS.save(name, this.id);
                },
                stop: function () {
                    // calculate rest duration of countdown (in seconds!)
                    var newDuration = Math.floor( this.countdown_time - Date.now() / 1000 + $rootScope.serverOffset );
                    this.running = false;
                    this.countdown_time = newDuration;
                    DS.save(name, this.id);
                },
                reset: function () {
                    this.running = false;
                    this.countdown_time = this.default_time;
                    DS.save(name, this.id);
                },
                // Override the BaseModel.project function
                project: function(projectorId) {
                    // if this object is already projected on projectorId, delete this element from this projector
                    var isProjectedIds = this.isProjected();
                    var self = this;
                    var predicate = function (element) {
                        return element.name == name && element.id == self.id;
                    };
                    _.forEach(isProjectedIds, function (id) {
                        var uuid = _.findKey(Projector.get(id).elements, predicate);
                        $http.post('/rest/core/projector/' + id + '/deactivate_elements/', [uuid]);
                    });
                    // if it was the same projector before, just delete it but not show again
                    if (_.indexOf(isProjectedIds, projectorId) == -1) {
                        return $http.post(
                            '/rest/core/projector/' + projectorId + '/activate_elements/',
                            [{name: name, id: self.id, stable: true}]
                        );
                    }
                },
            },
        });
    }
])

/* Two functions to convert between time duration in seconds <-> human readable time span.
 * E.g. 90 sec <-> 1:30 (min), 3661 sec <-> 1:01:01 (h)
 *
 * secondsToHumanTime: Expects seconds and give [h*:]mm[:ss]. The minutes part is always given, the hours
 *      and minutes could be controlled. The default are forced seconds and hours just if it is not 0.
 *      - seconds ('enabled', 'auto', 'disabled'): Whether to show seconds (Default 'enabled')
 *      - hours ('enabled', 'auto', 'disabled'): Whether to show hours (Default 'auto')
 *
 * humanTimeToSeconds: Expects [h*:]m*[:s*] with each part could have a variable length. The parsed time is
 *      in seconds. Minutes have to be given and hours and seconds are optional. One have to set 'seconds' or
 *      'hours' to true toparse these.
 *
 * params could be an object with the given settings, e.g. {ignoreHours: true}
 */
.factory('HumanTimeConverter', [
    function () {
        return {
            secondsToHumanTime: function (seconds, params) {
                if (!params) {
                    params = {seconds: 'enabled', hours: 'auto'};
                }
                if (!params.seconds) {
                    params.seconds = 'enabled';

                }
                if (!params.hours) {
                    params.hours = 'auto';
                }
                var time;
                // floor returns the largest integer of the absolut value of seconds
                var total = Math.floor(Math.abs(seconds));
                var h = Math.floor(total / 3600);
                var m = Math.floor(total % 3600 / 60);
                var s = Math.floor(total % 60);
                // Add leading "0" for double digit values
                time = ('0'+m).slice(-2); //minutes
                if ((params.seconds == 'auto' && s > 0) || params.seconds == 'enabled') {
                    s = ('0'+s).slice(-2);
                    time =  time + ':' + s;
                }
                if ((params.hours == 'auto' && h > 0) || params.hours == 'enabled') {
                    time = h + ':' + time;
                }
                if (seconds < 0) {
                    time = '-'+time;
                }
                return time;
            },
            humanTimeToSeconds: function (data, params) {
                if (!params) {
                    params = {seconds: false, hours: false};
                }
                var minLength = 1;
                if (params.seconds) {
                    minLength++;
                }
                if (params.hours){
                    minLength++;
                }

                var negative = data.charAt(0) == '-';
                var time = data.split(':');
                data = 0;
                if (time.length >= minLength) {
                    for (var i = 0; i < minLength; i++) {
                        data = data*60;
                        if (!isNaN(+time[i])) {
                            data += (+time[i]);
                        }
                    }
                    if (!params.seconds) { // the last field was minutes (e.g. h:mm)
                        data *= 60;
                    }
                    if (negative) {
                        data = -data;
                    }
                }
                return data;
            },
        };
    }
])

/* Converts a snake-case string to camelCase. Example:
 * 'motion-block-config' -> 'motionBlockConfig' */
.factory('CamelCase', [
    function () {
        return function (str) {
            return str.replace(/-([a-z])/g, function (match) {
                return match[1].toUpperCase();
            });
        };
    }
])

/* Return the specific EditForm for a given model. */
.factory('EditForm', [
    '$injector',
    'CamelCase',
    function ($injector, CamelCase) {
        return {
            fromCollectionString: function (collection) {
                var modelName = CamelCase(collection).split('/')[1];
                // Convert modelModel to ModelModelForm
                var formName = modelName.charAt(0).toUpperCase() + modelName.slice(1) + 'Form';
                return $injector.get(formName);
            },
        };
    }
])

/* Converts number of seconds into string "h:mm:ss" or "mm:ss" */
.filter('osSecondsToTime', [
    'HumanTimeConverter',
    function (HumanTimeConverter) {
        return function (seconds) {
            return HumanTimeConverter.secondsToHumanTime(seconds);
        };
    }
])

/* Converts number of minutes into string "h:mm" or "hh:mm" */
.filter('osMinutesToTime', [
    'HumanTimeConverter',
    function (HumanTimeConverter) {
        return function (minutes) {
            return HumanTimeConverter.secondsToHumanTime(minutes*60,
                { seconds: 'disabled',
                    hours: 'enabled' }
            );
        };
    }
])

// mark HTML as "trusted"
.filter('trusted', [
    '$sce',
    function ($sce) {
        return function(text) {
            return $sce.trustAsHtml(text);
        };
    }
])

// filters the requesting object (id=selfid) from a list of input objects
.filter('notself', function () {
    return function (input, selfid) {
        var result;
        if (selfid) {
            result = [];
            for (var key in input){
                var obj = input[key];
                if (selfid != obj.id) {
                    result.push(obj);
                }
            }
        } else {
            result = input;
        }
        return result;
    };
})

// Wraps the orderBy filter. But puts ("", null, undefined) last.
.filter('orderByEmptyLast', [
    '$filter',
    '$parse',
    function ($filter, $parse) {
        return function (array, sortPredicate, reverseOrder, compareFn) {
            var parsed = $parse(sortPredicate);
            var falsyItems = [];
            var truthyItems = _.filter(array, function (item) {
                var falsy = parsed(item) === void 0 || parsed(item) === null || parsed(item) === '';
                if (falsy) {
                    falsyItems.push(item);
                }
                return !falsy;
            });
            truthyItems = $filter('orderBy')(truthyItems, sortPredicate, reverseOrder, compareFn);
            return _.concat(truthyItems, falsyItems);
        };
    }
])

// Make sure that the DS factories are loaded by making them a dependency
.run([
    'ChatMessage',
    'Config',
    'Countdown',
    'ProjectorMessage',
    'Projector',
    'ProjectionDefault',
    'Tag',
    'Notify', // For setting up the autoupdate callback
    function (ChatMessage, Config, Countdown, ProjectorMessage, Projector, ProjectionDefault, Tag, Notify) {}
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.core.csv', [])

.factory('CsvDownload', [
    'Config',
    'FileSaver',
    function (Config, FileSaver) {
        var utf8_BOM = decodeURIComponent('%EF%BB%BF');
        return function (contentRows, filename) {
            var separator = Config.get('general_csv_separator').value;
            var rows = _.map(contentRows, function (row) {
                return row.join(separator);
            });
            var blob = new Blob([utf8_BOM + rows.join('\n')]);
            FileSaver.saveAs(blob, filename);
        };
    }
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.core.docx', [])

.factory('Html2DocxConverter', [
    '$q',
    'ImageConverter',
    function ($q, ImageConverter) {
        var PAGEBREAK = '<w:p><w:r><w:br w:type="page" /></w:r></w:p>';

        var createInstance = function () {
            var converter = {
                imageMap: {},
                documentImages: [],
                relationships: [],
                contentTypes: [],
            };

            var html2docx = function (html) {
                var docx = '';
                var tagStack = [];

                // With this variable, we keep track, if we are currently inside or outside of a paragraph.
                var inParagraph = true;
                // the text may not begin with a paragraph. If so, append one because word needs it.
                var skipFirstParagraphClosing = true;

                var handleTag = function (tag) {
                    if (tag.charAt(0) == "/") {  // A closing tag
                        // remove from stack
                        tagStack.pop();

                        // Special: end paragraphs
                        if (tag.indexOf('/p') === 0) {
                            docx += '</w:p>';
                            inParagraph = false;
                        }
                    } else { // now all other tags
                        var tagname = tag.split(' ')[0];
                        handleNamedTag(tagname, tag);
                    }
                    return docx;
                };
                var handleNamedTag = function (tagname, fullTag) {
                    var tag = {
                        tag: tagname,
                        attrs: {},
                    };
                    switch (tagname) {
                        case 'p':
                            if (inParagraph && !skipFirstParagraphClosing) {
                                // End the paragrapth, if there is one
                                docx += '</w:p>';
                            }
                            skipFirstParagraphClosing = false;
                            docx += '<w:p>';
                            inParagraph = true;
                            break;
                        case 'span':
                            var styleRegex = /(?:\"|\;\s?)([a-zA-z\-]+)\:\s?([a-zA-Z0-9\-\#]+)/g, matchSpan;
                            while ((matchSpan = styleRegex.exec(fullTag)) !== null) {
                                switch (matchSpan[1]) {
                                    case 'color':
                                            tag.attrs.color = matchSpan[2].slice(1); // cut off the #
                                        break;
                                    case 'background-color':
                                            tag.attrs.backgroundColor = matchSpan[2].slice(1); // cut off the #
                                        break;
                                    case 'text-decoration':
                                        if (matchSpan[2] === 'underline') {
                                            tag.attrs.underline = true;
                                        } else if (matchSpan[2] === 'line-through') {
                                            tag.attrs.strike = true;
                                        }
                                        break;
                                }
                            }
                            break;
                        case 'a':
                            var hrefRegex = /href="([^"]+)"/g;
                            var href = hrefRegex.exec(fullTag)[1];
                            tag.href = href;
                            break;
                        case 'img':
                            imageTag(tag, fullTag);
                            break;
                    }
                    if (tagname !== 'img' && tagname !== 'p') {
                        tagStack.push(tag);
                    }
                };
                var imageTag = function (tag, fullTag) {
                    // images has to be placed instantly, so there is no use of 'tag'.
                    var image = {};
                    var attributeRegex = /(\w+)=\"([^\"]*)\"/g, attributeMatch;
                    while ((attributeMatch = attributeRegex.exec(fullTag)) !== null) {
                        image[attributeMatch[1]] = attributeMatch[2];
                    }
                    if (image.src && converter.imageMap[image.src]) {
                        image.width = converter.imageMap[image.src].width;
                        image.height = converter.imageMap[image.src].height;

                        var rrId = converter.relationships.length + 1;
                        var imageId = converter.documentImages.length + 1;

                        // set name ('pic.jpg'), title, ext ('jpg'), mime ('image/jpeg')
                        image.name = _.last(image.src.split('/'));

                        var tmp = image.name.split('.');
                        image.ext = tmp.splice(-1);

                        // set name without extension as title if there isn't a title
                        if (!image.title) {
                            image.title = tmp.join('.');
                        }

                        image.mime = 'image/' + image.ext;
                        if (image.ext == 'jpe' || image.ext == 'jpg') {
                            image.mime = 'image/jpeg';
                        }

                        // x and y for the container and picture size in EMU (assuming 96dpi)!
                        var x = image.width * 914400 / 96;
                        var y = image.height * 914400 / 96;

                        // the image does not belong into a paragraph in ooxml
                        if (inParagraph) {
                            docx += '</w:p>';
                        }
                        docx += '<w:p><w:r><w:drawing><wp:inline distT="0" distB="0" distL="0" distR="0"><wp:extend cx="' + x +'" cy="' + y + '"/><wp:effectExtent l="0" t="0" r="0" b="0"/>' +
                            '<wp:docPr id="' + imageId + '" name="' + image.name + '" title="' + image.title + '" descr="' + image.title + '"/><wp:cNvGraphicFramePr>' +
                            '<a:graphicFrameLocks xmlns:a="http://schemas.openxmlformats.org/drawingml/2006/main" noChangeAspect="1"/></wp:cNvGraphicFramePr>' +
                            '<a:graphic xmlns:a="http://schemas.openxmlformats.org/drawingml/2006/main"><a:graphicData uri="http://schemas.openxmlformats.org/drawingml/2006/picture">' +
                            '<pic:pic xmlns:pic="http://schemas.openxmlformats.org/drawingml/2006/picture"><pic:nvPicPr><pic:cNvPr id="' + imageId + '" name="' +
                            image.name + '" title="' + image.title + '" descr="' + image.title + '"/><pic:cNvPicPr/></pic:nvPicPr><pic:blipFill><a:blip r:embed="rrId' + rrId + '"/><a:stretch>' +
                            '<a:fillRect/></a:stretch></pic:blipFill><pic:spPr bwMode="auto"><a:xfrm><a:off x="0" y="0"/><a:ext cx="' + x + '" cy="' + y + '"/></a:xfrm>' +
                            '<a:prstGeom prst="rect"><a:avLst/></a:prstGeom></pic:spPr></pic:pic></a:graphicData></a:graphic></wp:inline></w:drawing></w:r></w:p>';

                        // inParagraph stays untouched, the documents paragraph state is restored here
                        if (inParagraph) {
                            docx += '<w:p>';
                        }

                        // entries in documentImages, relationships and contentTypes
                        converter.documentImages.push({
                            src: image.src,
                            zipPath: 'word/media/' + image.name
                        });
                        converter.relationships.push({
                            Id: 'rrId' + rrId,
                            Type: 'http://schemas.openxmlformats.org/officeDocument/2006/relationships/image',
                            Target: 'media/' + image.name
                        });
                        converter.contentTypes.push({
                            PartName: '/word/media/' + image.name,
                            ContentType: image.mime
                        });
                    }
                };
                var handleText = function (text) {
                    // Start a new paragraph, if only loose text is there
                    if (!inParagraph) {
                        docx += '<w:p>';
                        inParagraph = true;
                    }
                    var docxPart = '<w:r><w:rPr>';
                    var hyperlink = false;
                    tagStack.forEach(function (tag) {
                        switch (tag.tag) {
                            case 'b':
                            case 'strong':
                                docxPart += '<w:b/><w:bCs/>';
                                break;
                            case 'em':
                            case 'i':
                                docxPart += '<w:i/><w:iCs/>';
                                break;
                            case 'span':
                                for (var key in tag.attrs) {
                                    switch (key) {
                                        case 'color':
                                            docxPart += '<w:color w:val="' + tag.attrs[key] + '"/>';
                                            break;
                                        case 'backgroundColor':
                                            docxPart += '<w:shd w:fill="' + tag.attrs[key] + '"/>';
                                            break;
                                        case 'underline':
                                            docxPart += '<w:u w:val="single"/>';
                                            break;
                                        case 'strike':
                                            docxPart += '<w:strike/>';
                                            break;
                                    }
                                }
                                break;
                            case 'u':
                                docxPart += '<w:u w:val="single"/>';
                                break;
                            case 'strike':
                                docxPart += '<w:strike/>';
                                break;
                            case 'a':
                                var id = converter.relationships.length + 1;
                                docxPart = '<w:hyperlink r:id="rrId' + id + '">' + docxPart;
                                docxPart += '<w:rStyle w:val="Internetlink"/>';
                                converter.relationships.push({
                                    Id: 'rrId' + id,
                                    Type: 'http://schemas.openxmlformats.org/officeDocument/2006/relationships/hyperlink',
                                    Target: tag.href,
                                    TargetMode: 'External'
                                });
                                hyperlink = true;
                                break;
                        }
                    });
                    docxPart += '</w:rPr><w:t>' + text + '</w:t></w:r>';
                    if (hyperlink) {
                        docxPart += '</w:hyperlink>';
                    }

                    // append to docx
                    docx += docxPart;
                    return docx;
                };

                var replaceEntities = function () {
                    // replacing of special symbols:
                    docx = docx.replace(new RegExp('\&auml\;', 'g'), 'ä');
                    docx = docx.replace(new RegExp('\&uuml\;', 'g'), 'ü');
                    docx = docx.replace(new RegExp('\&ouml\;', 'g'), 'ö');
                    docx = docx.replace(new RegExp('\&Auml\;', 'g'), 'Ä');
                    docx = docx.replace(new RegExp('\&Uuml\;', 'g'), 'Ü');
                    docx = docx.replace(new RegExp('\&Ouml\;', 'g'), 'Ö');
                    docx = docx.replace(new RegExp('\&szlig\;', 'g'), 'ß');
                    docx = docx.replace(new RegExp('\&nbsp\;', 'g'), ' ');
                    docx = docx.replace(new RegExp('\&sect\;', 'g'), '§');

                    // remove all entities except gt, lt and amp
                    var entityRegex = /\&(?!gt|lt|amp)\w+\;/g, matchEntry, indexes = [];
                    while ((matchEntry = entityRegex.exec(docx)) !== null) {
                        indexes.push({
                            startId: matchEntry.index,
                            stopId: matchEntry.index + matchEntry[0].length
                        });
                    }
                    for (var i = indexes.length - 1; i>=0; i--) {
                        docx = docx.substring(0, indexes[i].startId) + docx.substring(indexes[i].stopId, docx.length);
                    }
                };

                var parse = function () {
                    if (html.substring(0,3) != '<p>') {
                        docx += '<w:p>';
                        skipFirstParagraphClosing = false;
                    }
                    html = html.split(/(<|>)/g);
                    // remove whitespaces and > brackets. Leave < brackets in there to check, whether
                    // the following string is a tag or text.
                    html = _.filter(html, function (part) {
                        var skippedCharsRegex = new RegExp('^([\s\n\r]|>)*$', 'g');
                        return !skippedCharsRegex.test(part);
                    });

                    for (var i = 0; i < html.length; i++) {
                        if (html[i] === '<') {
                            i++;
                            handleTag(html[i]);
                        } else {
                            handleText(html[i]);
                        }
                    }
                    // for finishing close the last paragraph (if open)
                    if (inParagraph) {
                        docx += '</w:p>';
                    }

                    replaceEntities();

                    return docx;
                };

                return parse();
            };

            // return a wrapper function for html2docx, that fetches all the images.
            converter.html2docx = function (html) {
                var imageSources = _.map($(html).find('img'), function (element) {
                    return element.getAttribute('src');
                });
                // Don't get images multiple times; just if the converter has not seen them befor.
                imageSources = _.filter(imageSources, function (src) {
                    return !converter.imageMap[src];
                });
                return $q(function (resolve) {
                    ImageConverter.toBase64(imageSources).then(function (_imageMap) {
                        _.forEach(_imageMap, function (value, key) {
                            converter.imageMap[key] = value;
                        });
                        var docx = html2docx(html);
                        resolve(docx);
                    });
                });
            };

            converter.updateZipFile = function (zip) {
                var updateRelationships = function (oldContent) {
                    var content = oldContent.split('\n');
                    _.forEach(converter.relationships, function (relationship) {
                        content[1] += '<Relationship';
                        _.forEach(relationship, function (value, key) {
                            content[1] += ' ' + key + '="' + value + '"';
                        });
                        content[1] += '/>';
                    });
                    return content.join('\n');
                };
                var updateContentTypes = function (oldContent) {
                    var content = oldContent.split('\n');
                    _.forEach(converter.contentTypes, function (type) {
                        content[1] += '<Override';
                        _.forEach(type, function (value, key) {
                            content[1] += ' ' + key + '="' + value + '"';
                        });
                        content[1] += '/>';
                    });
                    return content.join('\n');
                };
                // update relationships from 'relationships'
                var relationships = updateRelationships(zip.file('word/_rels/document.xml.rels').asText());
                zip.file('word/_rels/document.xml.rels', relationships);

                // update content type from 'contentTypes'
                var contentTypes = updateContentTypes(zip.file('[Content_Types].xml').asText());
                zip.file('[Content_Types].xml', contentTypes);

                converter.documentImages = _.uniqBy(converter.documentImages, 'src');
                _.forEach(converter.documentImages, function (image) {
                    var dataUrl = converter.imageMap[image.src].data;
                    var base64 = dataUrl.split(',')[1];
                    zip.file(image.zipPath, base64, {base64: true});
                });
                return zip;
            };

            return converter;
        };

        return {
            createInstance: createInstance,
        };
    }
]);

})();

(function () {

'use strict';

angular.module('OpenSlidesApp.core.pdf', [])

/*
 * General layout functions for building PDFs with pdfmake.
 */
.factory('PDFLayout', [
    'gettextCatalog',
    function(gettextCatalog) {
        var PDFLayout = {};
        var BallotCircleDimensions = {
            yDistance: 6,
            size: 8
        };

        // page title
        PDFLayout.createTitle = function(title) {
            return {
                text: title,
                style: "title"
            };
        };

        // page subtitle
        PDFLayout.createSubtitle = function(subtitle) {
            return {
                text: subtitle.join('\n'),
                style: "subtitle"
            };
        };

        // pagebreak
        PDFLayout.addPageBreak = function() {
            return [
                {
                    text: '',
                    pageBreak: 'after'
                }
            ];
        };

        // table row style
        PDFLayout.flipTableRowStyle = function(currentTableSize) {
            if (currentTableSize % 2 === 0) {
                return "tableEven";
            } else {
                return "tableOdd";
            }
        };

        // draws a circle
        PDFLayout.drawCircle = function(y, size) {
            return [
                {
                    type: 'ellipse',
                    x: 0,
                    y: y,
                    lineColor: 'black',
                    r1: size,
                    r2: size
                }
            ];
        };

        // returns an entry in the ballot with a circle to draw into
        PDFLayout.createBallotEntry = function(decision) {
            return {
                margin: [40+BallotCircleDimensions.size, 10, 0, 0],
                columns: [
                    {
                        width: 15,
                        canvas: PDFLayout.drawCircle(BallotCircleDimensions.yDistance,
                                BallotCircleDimensions.size)
                    },
                    {
                        width: "auto",
                        text: decision
                    }
                ],
            };
        };

        // crop marks for ballot papers
        PDFLayout.getBallotLayoutLines = function() {
            return '{{ballot-placeholder-to-insert-functions-here}}';
        };

        // returns a promise for converting an image in data URL format with size information
        PDFLayout.imageURLtoBase64 = function(url) {
            var promise = new Promise(function(resolve, reject) {
                var img = new Image();
                img.crossOrigin = "Anonymous";
                img.onerror = function () {
                    reject({
                        msg: '<i class="fa fa-exclamation-triangle fa-lg spacer-right"></i>' +
                             gettextCatalog.getString('Error while generating PDF file') +
                             ': <code>' + gettextCatalog.getString('Cannot load image') + ' ' + url + '</code>',
                    });
                };
                img.onload = function () {
                    var canvas = document.createElement("canvas");
                    canvas.width = img.width;
                    canvas.height = img.height;
                    var ctx = canvas.getContext("2d");
                    ctx.drawImage(img, 0, 0);
                    var dataURL = canvas.toDataURL("image/png");
                    var imageData = {
                        data: dataURL,
                        width: img.width,
                        height: img.height
                    };
                    resolve(imageData);
                };
                img.src = url;
            });
            return promise;
        };

        return PDFLayout;
    }
])


.factory('HTMLValidizer', function() {
    var HTMLValidizer = {};

    // In some cases copying from word to OpenSlides results in umlauts
    // that are the base letter and then the entity #776; to make the dots
    // above the base letter. This breaks the PDF.
    HTMLValidizer.replaceMalformedUmlauts = function (text) {
        return text.replace(/([aeiouAEIOUy])[\u0308]/g, function (match, baseChar) {
            return '&' + baseChar + 'uml;';
        });
    };


    //checks if str is valid HTML. Returns valid HTML if not,
    //return emptystring if empty
    HTMLValidizer.validize = function(str) {
        if (str) {
            str = HTMLValidizer.replaceMalformedUmlauts(str);
            // Sometimes, some \n are in the text instead of whitespaces. Replace them.
            str = str.replace(/\n/g, ' ');

            var a = document.createElement('div');
            a.innerHTML = str;
            angular.forEach(a.childNodes, function (child) {
                if (child.nodeType == 1) {
                    return str;
                }
            });
            return '<p>' + str + '</p>';
        } else {
            return ''; //needed for blank "reasons" field
        }
    };
    return HTMLValidizer;
})


.factory('PdfMakeDocumentProvider', [
    '$q',
    'Config',
    'PDFLayout',
    'ImageConverter',
    function($q, Config, PDFLayout, ImageConverter) {
        /**
         * Provides the global document
         * @constructor
         * @param {object} contentProvider - Object with on method `getContent`, which
         * returns an array for content
         */
        //images shall contain the the logos as URL: base64Str, just like the converter
        var createInstance = function(contentProvider, noFooter) {
            // Logo urls
            var logoHeaderUrl = Config.get('logo_pdf_header').value.path,
                logoFooterUrl = Config.get('logo_pdf_footer').value.path;
            var imageMap = contentProvider.getImageMap ? contentProvider.getImageMap() : {};

            // PDF header
            var getHeader = function() {
                var columns = [];

                if (logoHeaderUrl) {
                    if (logoHeaderUrl.indexOf('/') === 0) {
                        logoHeaderUrl = logoHeaderUrl.substr(1); // remove trailing /
                    }
                    columns.push({
                        image: logoHeaderUrl,
                        fit: [180, 40],
                        width: '20%'
                    });
                }

                var line1 = [
                    Config.translate(Config.get('general_event_name').value),
                    Config.translate(Config.get('general_event_description').value)
                ].filter(Boolean).join(' – ');
                var line2 = [
                    Config.get('general_event_location').value,
                    Config.get('general_event_date').value
                ].filter(Boolean).join(', ');
                var text = [line1, line2].join('\n');
                columns.push({
                    text: text,
                    fontSize: 10,
                    alignment: 'right',
                    margin: [0, 10, 0, 0],
                });
                return {
                    color: '#555',
                    fontSize: 9,
                    margin: [75, 30, 75, 10], // [left, top, right, bottom]
                    columns: columns,
                    columnGap: 10,
                };
            };


            // PDF footer
            // Used placeholder for currentPage and pageCount which
            // are replaced by dynamic footer function in pdf-worker.js.
            var getFooter = function() {
                var columns = [];

                if (logoFooterUrl) {
                    if (logoFooterUrl.indexOf('/') === 0) {
                        logoFooterUrl = logoFooterUrl.substr(1); // remove trailing /
                    }
                    columns.push({
                        image: logoFooterUrl,
                        fit: [400,50],
                        width: '80%'
                    });
                }
                columns.push({
                    text: '{{currentPage}} / {{pageCount}}',
                    color: '#555',
                    fontSize: 9,
                    alignment: Config.get('general_export_pdf_pagenumber_alignment').value,
                    margin: [0, 15, 0, 0],
                });
                return {
                    margin: [75, 0, 75, 10],
                    columns: columns,
                    columnGap: 10,
                };
            };
            // Generates the document(definition) for pdfMake
            var getDocument = function() {
                var content = contentProvider.getContent();
                var standardFontsize = Config.get('general_export_pdf_fontsize').value;
                return {
                    pageSize: 'A4',
                    pageMargins: [75, 90, 75, 75],
                    defaultStyle: {
                        font: 'PdfFont',
                        fontSize: standardFontsize
                    },
                    header: getHeader(),
                    footerTpl: noFooter ? '' : getFooter(),
                    content: content,
                    styles: {
                        title: {
                            fontSize: 18,
                            margin: [0,0,0,20],
                            bold: true
                        },
                        subtitle: {
                            fontSize: 9,
                            margin: [0,-20,0,20],
                            color: 'grey'
                        },
                        preamble: {
                            fontSize: standardFontsize,
                            margin: [0,0,0,10],
                        },
                        userDataTitle: {
                            fontSize: 26,
                            margin: [0,0,0,0],
                            bold: true
                        },
                        textItem: {
                            fontSize: 11,
                            margin: [0,7]
                        },
                        heading2: {
                            fontSize: 14,
                            margin: [0,0,0,10],
                            bold: true
                        },
                        heading3: {
                            fontSize: 12,
                            margin: [0,10,0,0],
                            bold: true
                        },
                        userDataHeading: {
                            fontSize: 14,
                            margin: [0,10],
                            bold: true
                        },
                        userDataTopic: {
                            fontSize: 12,
                            margin: [0,5]
                        },
                        userDataValue: {
                            fontSize: 12,
                            margin: [15,5]
                        },
                        tableofcontent: {
                            fontSize: 12,
                            margin: [0,3]
                        },
                        listParent: {
                            fontSize: 12,
                            margin: [0,5]
                        },
                        listChild: {
                            fontSize: 10,
                            margin: [0,5]
                        },
                        tableHeader: {
                            bold: true,
                            fillColor: 'white'
                        },
                        tableEven: {
                            fillColor: 'white'
                        },
                        tableOdd: {
                            fillColor: '#eee'
                        },
                        tableConclude: {
                            fillColor: '#ddd',
                            bold: true
                        },
                        grey: {
                            fillColor: '#ddd',
                        },
                        lightgrey: {
                            fillColor: '#aaa',
                        },
                        bold: {
                            bold: true,
                        },
                        small: {
                            fontSize: 8,
                        }
                    }
                };
            };

            var getImageMap = function () {
                return imageMap;
            };

            return $q(function (resolve, reject) {
                var imageSources = [
                    logoHeaderUrl,
                    logoFooterUrl
                ];
                ImageConverter.toBase64(imageSources).then(function (_imageMap) {
                    _.forEach(_imageMap, function (data, path) {
                        if (!imageMap[path]) {
                            imageMap[path] = data;
                        }
                    });
                    resolve({
                        getDocument: getDocument,
                        getImageMap: getImageMap,
                    });
                }, reject);
            });
        };
        return {
            createInstance: createInstance,
        };
    }
])

.factory('PdfMakeBallotPaperProvider', [
    'PDFLayout',
    function(PDFLayout) {
        /**
         * Provides the global Document
         * @constructor
         * @param {object} contentProvider - Object with on method `getContent`, which returns an array for content
         */
        var createInstance = function(contentProvider) {
            /**
             * Generates the document(definition) for pdfMake
             * @function
             */
            var getDocument = function() {
                var content = contentProvider.getContent();
                return {
                    pageSize: 'A4',
                    pageMargins: [0, 0, 0, 0],
                    defaultStyle: {
                        font: 'PdfFont',
                        fontSize: 10
                    },
                    content: content,
                    styles: {
                        title: {
                            fontSize: 14,
                            bold: true,
                            margin: [30, 30, 0, 0]
                        },
                        description: {
                            fontSize: 11,
                            margin: [30, 0, 0, 0]
                        }
                    }
                };
            };

            var getImageMap = function() {
                return contentProvider.getImageMap();
            };

            return {
                getDocument: getDocument,
                getImageMap: getImageMap,
            };
        };
        return {
            createInstance: createInstance
        };
    }
])

.factory('PdfMakeConverter', [
    'HTMLValidizer',
    'Config',
    function(HTMLValidizer, Config) {
        /**
         * Converter component for HTML->JSON for pdfMake
         * @constructor
         * @param {object} images   - Key-Value structure representing image.src/BASE64 of images
         */
        var createInstance = function(images) {
            var slice = Function.prototype.call.bind([].slice),
                map = Function.prototype.call.bind([].map),

                DIFF_MODE_NORMAL = 0,
                DIFF_MODE_INSERT = 1,
                DIFF_MODE_DELETE = 2,

                // Space between list elements
                LI_MARGIN_BOTTOM = 8,

                /**
                 * Convertes HTML for use with pdfMake
                 * @function
                 * @param {object} html - html
                 * @param {string} lineNumberMode - [inline, outside, none]
                 */
                convertHTML = function(html, lineNumberMode) {
                    var elementStyles = {
                            "b": ["font-weight:bold"],
                            "strong": ["font-weight:bold"],
                            "u": ["text-decoration:underline"],
                            "em": ["font-style:italic"],
                            "i": ["font-style:italic"],
                            "h1": ["font-size:14", "font-weight:bold"],
                            "h2": ["font-size:12", "font-weight:bold"],
                            "h3": ["font-size:10", "font-weight:bold"],
                            "h4": ["font-size:10", "font-style:italic"],
                            "h5": ["font-size:10"],
                            "h6": ["font-size:10"],
                            "a": ["color:blue", "text-decoration:underline"],
                            "strike": ["text-decoration:line-through"],
                            "del": ["color:red", "text-decoration:line-through"],
                            "ins": ["color:green", "text-decoration:underline"]
                        },
                        classStyles = {
                            "delete": ["color:red", "text-decoration:line-through"],
                            "insert": ["color:green", "text-decoration:underline"]
                        },
                        getLineNumber = function (element) {
                            if (element && element.nodeName == 'SPAN' && element.getAttribute('class') &&
                                element.getAttribute('class').indexOf('os-line-number') > -1) {
                                return element.getAttribute('data-line-number');
                            }
                        },
                        /**
                         *
                         * Removes all line number nodes (not line-breaks)
                         * and returns an array containing the reoved numbers in this format:
                         * { lineNumber: '<lineNumber>', marginBottom: <number> }
                         * where marginBottom is optional.
                         *
                         * @function
                         * @param {object} element
                         */
                        extractLineNumbers = function(element) {
                            var foundLineNumbers = [];
                            var lineNumber = getLineNumber(element);
                            if (lineNumber) {
                                foundLineNumbers.push({lineNumber: lineNumber});
                                element.parentNode.removeChild(element);
                            } else if (element.nodeName === 'BR') {
                                // Check if there is a new line, but it does not get a line number.
                                // If so, insert a dummy line, so the line nubers stays aligned with
                                // the text.
                                if (!getLineNumber(element.nextSibling)) {
                                    foundLineNumbers.push({lineNumber: ''});
                                }
                            } else {
                                var children = element.childNodes,
                                    childrenLength = children.length,
                                    childrenLineNumbers = [];
                                for (var i = 0; i < children.length; i++) {
                                    childrenLineNumbers = _.concat(childrenLineNumbers, extractLineNumbers(children[i]));
                                    if (children.length < childrenLength) {
                                        i -= (childrenLength - children.length);
                                        childrenLength = children.length;
                                    }
                                }
                                // If this is an list item, add some space to the lineNumbers:
                                if (childrenLineNumbers.length && element.nodeName === 'LI') {
                                    _.last(childrenLineNumbers).marginBottom = LI_MARGIN_BOTTOM;
                                }
                                foundLineNumbers = _.concat(foundLineNumbers, childrenLineNumbers);
                            }
                            return foundLineNumbers;
                        },
                        /**
                         * Parses Children of the current paragraph
                         * @function
                         * @param {object} converted  -
                         * @param {object} element   -
                         * @param {object} currentParagraph -
                         * @param {object} styles -
                         * @param {number} diff_mode
                         */
                        parseChildren = function(converted, element, currentParagraph, styles, diff_mode) {
                            var elements = [];
                            var children = element.childNodes;
                            if (children.length !== 0) {
                                _.forEach(children, function(child) {
                                    currentParagraph = ParseElement(elements, child, currentParagraph, styles, diff_mode);
                                });
                            }
                            if (elements.length !== 0) {
                                _.forEach(elements, function(el) {
                                    converted.push(el);
                                });
                            }
                            return currentParagraph;
                        },
                        /**
                         * Returns the color in a hex format (e.g. #12ff00).
                         * Tries to convert the rgb form into this.
                         * @function
                         * @param {string} color
                         */
                        parseColor = function (color) {
                            var hexRegex = new RegExp('^#([0-9a-f]{3}|[0-9a-f]{6})$');
                            // e.g. #fff or #ff0048
                            var rgbRegex = new RegExp('^rgb\\((\\d+),\\s*(\\d+),\\s*(\\d+)\\)$');
                            // e.g. rgb(0,255,34) or rgb(22, 0, 0)
                            var nameRegex = new RegExp('^[a-z]+$');
                            // matches just text like 'red', 'black', 'green'

                            if (hexRegex.test(color)) {
                                return color;
                            } else if (rgbRegex.test(color)) {
                                var decimalColors = rgbRegex.exec(color).slice(1);
                                for (var i = 0; i < 3; i++) {
                                    var decimalValue = parseInt(decimalColors[i]);
                                    if (decimalValue > 255) {
                                        decimalValue = 255;
                                    }
                                    var hexString = '0' + decimalValue.toString(16);
                                    hexString = hexString.slice(-2);
                                    decimalColors[i] = hexString;
                                }
                                return '#' + decimalColors.join('');
                            } else if (nameRegex.test(color)) {
                                return color;
                            } else {
                                console.error('Could not parse color "' + color + '"');
                                return color;
                            }
                        },
                        /**
                         * Extracts the style from an object
                         * @function
                         * @param {object} o       - the current object
                         * @param {object} styles  - an array with styles
                         */
                        ComputeStyle = function(o, styles) {
                            styles.forEach(function(singleStyle) {
                                var styleDefinition = singleStyle.trim().toLowerCase().split(":");
                                var style = styleDefinition[0];
                                var value = styleDefinition[1];
                                if (styleDefinition.length === 2) {
                                    switch (style) {
                                        case "padding-left":
                                            o.margin = [parseInt(value), 0, 0, 0];
                                            break;
                                        case "font-size":
                                            o.fontSize = parseInt(value);
                                            break;
                                        case "text-align":
                                            switch (value) {
                                                case "right":
                                                case "center":
                                                case "justify":
                                                    o.alignment = value;
                                                    break;
                                            }
                                            break;
                                        case "font-weight":
                                            switch (value) {
                                                case "bold":
                                                    o.bold = true;
                                                    break;
                                            }
                                            break;
                                        case "text-decoration":
                                            switch (value) {
                                                case "underline":
                                                    o.decoration = "underline";
                                                    break;
                                                case "line-through":
                                                    o.decoration = "lineThrough";
                                                    break;
                                            }
                                            break;
                                        case "font-style":
                                            switch (value) {
                                                case "italic":
                                                    o.italics = true;
                                                    break;
                                            }
                                            break;
                                        case "color":
                                            o.color = parseColor(value);
                                            break;
                                        case "background-color":
                                            o.background = parseColor(value);
                                            break;
                                    }
                                }
                            });
                        },
                        // A little helper function to check, if an element has the given class.
                        hasClass = function (element, className) {
                            var classes = element.getAttribute('class');
                            if (classes) {
                                classes = classes.toLowerCase().split(' ');
                                return _.indexOf(classes, className) > -1;
                            } else {
                                return false;
                            }
                        },
                        // Helper function for determinating whether a parent of element is a list item.
                        isInsideAList = function (element) {
                            var parent = element.parentNode;
                            while(parent !== null) {
                                if (parent.nodeName.toLowerCase() === 'li') {
                                    return true;
                                }
                                parent = parent.parentNode;
                            }
                            return false;
                        },
                        /**
                         * Parses a single HTML element
                         * @function
                         * @param {object} alreadyConverted  -
                         * @param {object} element   -
                         * @param {object} currentParagraph -
                         * @param {object} styles -
                         * @param {number} diff_mode
                         */
                        ParseElement = function(alreadyConverted, element, currentParagraph, styles, diff_mode) {
                            styles = styles ? _.clone(styles) : [];
                            var classes = [];
                            if (element.getAttribute) {
                                var nodeStyle = element.getAttribute("style");
                                if (nodeStyle) {
                                    nodeStyle.split(";").forEach(function(nodeStyle) {
                                        var tmp = nodeStyle.replace(/\s/g, '');
                                        styles.push(tmp);
                                    });
                                }
                                var nodeClass = element.getAttribute("class");
                                if (nodeClass) {
                                    classes = nodeClass.toLowerCase().split(" ");
                                    classes.forEach(function(nodeClass) {
                                        if (typeof(classStyles[nodeClass]) != 'undefined') {
                                            classStyles[nodeClass].forEach(function(style) {
                                                styles.push(style);
                                            });
                                        }
                                        if (nodeClass == 'insert') {
                                            diff_mode = DIFF_MODE_INSERT;
                                        }
                                        if (nodeClass == 'delete') {
                                            diff_mode = DIFF_MODE_DELETE;
                                        }
                                    });
                                }
                            }
                            var nodeName = element.nodeName.toLowerCase();
                            switch (nodeName) {
                                case "h1":
                                case "h2":
                                case "h3":
                                case "h4":
                                case "h5":
                                case "h6":
                                    if (lineNumberMode === "outside" &&
                                            element.childNodes.length > 0 &&
                                            element.childNodes[0].getAttribute) {
                                        // A heading may have multiple lines, so handle line by line separated by line number elements
                                        var outerStack = create("stack");
                                        var currentCol;
                                        _.forEach(element.childNodes, function (node) {
                                            if (node.getAttribute && node.getAttribute('data-line-number')) {
                                                if (currentCol) {
                                                    ComputeStyle(currentCol, elementStyles[nodeName]);
                                                    outerStack.stack.push(currentCol);
                                                }
                                                currentCol = {
                                                    columns: [
                                                        getLineNumberObject({
                                                            lineNumber: node.getAttribute('data-line-number')
                                                        }),
                                                    ],
                                                    margin: [0, 2, 0, 0],
                                                };
                                            } else if (node.textContent) {
                                                var HeaderText = {
                                                    text: node.textContent,
                                                };
                                                currentCol.columns.push(HeaderText);
                                            }
                                        });
                                        ComputeStyle(currentCol, elementStyles[nodeName]);
                                        outerStack.stack.push(currentCol);
                                        outerStack.margin = [0, 0, 0, 0];
                                        if (!/h[1-6]/.test(element.previousSibling.nodeName.toLowerCase())) {
                                            outerStack.margin[1] = 10;
                                        }
                                        alreadyConverted.push(outerStack);
                                    } else {
                                        currentParagraph = create("text");
                                        currentParagraph.marginBottom = 4;
                                        currentParagraph.marginTop = 10;
                                        currentParagraph = parseChildren(alreadyConverted, element, currentParagraph, styles.concat(elementStyles[nodeName]), diff_mode);
                                        alreadyConverted.push(currentParagraph);
                                    }
                                    break;
                                case "a":
                                case "b":
                                case "strong":
                                case "u":
                                case "em":
                                case "i":
                                case "ins":
                                case "del":
                                case "strike":
                                    currentParagraph = parseChildren(alreadyConverted, element, currentParagraph, styles.concat(elementStyles[nodeName]), diff_mode);
                                    break;
                                case "table":
                                    var t = create("table", {
                                        widths: [],
                                        body: []
                                    });
                                    var border = element.getAttribute("border");
                                    var isBorder = false;
                                    if (border) {
                                        isBorder = (parseInt(border) === 1);
                                    } else {
                                        t.layout = 'noBorders';
                                    }
                                    currentParagraph = parseChildren(t.table.body, element, currentParagraph, styles, diff_mode);
                                    var widths = element.getAttribute("widths");
                                    if (!widths) {
                                        if (t.table.body.length !== 0) {
                                            if (t.table.body[0].length !== 0)
                                                for (var k = 0; k < t.table.body[0].length; k++)
                                                    t.table.widths.push("*");
                                        }
                                    } else {
                                        var w = widths.split(",");
                                        for (var ko = 0; ko < w.length; ko++) t.table.widths.push(w[ko]);
                                    }
                                    alreadyConverted.push(t);
                                    break;
                                case "tbody":
                                    currentParagraph = parseChildren(alreadyConverted, element, currentParagraph, styles, diff_mode);
                                    break;
                                case "tr":
                                    var row = [];
                                    currentParagraph = parseChildren(row, element, currentParagraph, styles, diff_mode);
                                    alreadyConverted.push(row);
                                    break;
                                case "td":
                                    currentParagraph = create("text");
                                    var st = create("stack");
                                    st.stack.push(currentParagraph);
                                    var rspan = element.getAttribute("rowspan");
                                    if (rspan)
                                        st.rowSpan = parseInt(rspan);
                                    var cspan = element.getAttribute("colspan");
                                    if (cspan)
                                        st.colSpan = parseInt(cspan);
                                    currentParagraph = parseChildren(st.stack, element, currentParagraph, styles, diff_mode);
                                    alreadyConverted.push(st);
                                    break;
                                case "span":
                                    if (element.getAttribute("data-line-number")) {
                                        if (lineNumberMode === "inline") {
                                            if (diff_mode !== DIFF_MODE_INSERT) {
                                                var lineNumberInline = element.getAttribute("data-line-number"),
                                                    lineNumberObjInline = {
                                                        text: lineNumberInline,
                                                        color: "gray",
                                                        fontSize: 5
                                                    };
                                                currentParagraph.text.push(lineNumberObjInline);
                                            }
                                        } else if (lineNumberMode === "outside") {
                                            var lineNumberOutline;
                                            if (diff_mode === DIFF_MODE_INSERT) {
                                                lineNumberOutline = "";
                                            } else {
                                                lineNumberOutline = element.getAttribute("data-line-number");
                                            }
                                            var col = {
                                                columns: [
                                                    getLineNumberObject({
                                                        lineNumber: lineNumberOutline,
                                                    }),
                                                ]
                                            };
                                            currentParagraph = create("text");
                                            currentParagraph.lineHeight = 1.25;
                                            col.columns.push(currentParagraph);
                                            alreadyConverted.push(col);
                                        }
                                    }
                                    else {
                                        currentParagraph = parseChildren(alreadyConverted, element, currentParagraph, styles, diff_mode);
                                    }
                                    break;
                                case "br":
                                    var brParent = element.parentNode;
                                    var brParentNodeName = brParent.nodeName;
                                    //in case of no or inline-line-numbers and the ignore os-line-breaks.
                                    if ((lineNumberMode === 'inline' || lineNumberMode === 'none') &&
                                                hasClass(element, 'os-line-break')) {
                                        break;
                                    } else {
                                        currentParagraph = create("text");
                                        if (lineNumberMode === "outside" &&
                                                brParentNodeName !== "LI" &&
                                                element.parentNode.parentNode.nodeName !== "LI") {
                                            if (brParentNodeName === 'INS' || brParentNodeName === 'DEL') {

                                                var hasPrevSiblingALineNumber = function (element) {
                                                    // Iterare all nodes up to the top from element.
                                                    while (element) {
                                                        if (getLineNumber(element)) {
                                                            return true;
                                                        }
                                                        if (element.previousSibling) {
                                                            element = element.previousSibling;
                                                        } else {
                                                            element = element.parentNode;
                                                        }
                                                    }
                                                    return false;
                                                };
                                                if (hasPrevSiblingALineNumber(brParent)) {
                                                     currentParagraph.margin = [20, 0, 0, 0];
                                                 }
                                             } else {
                                                 currentParagraph.margin = [20, 0, 0, 0];
                                             }
                                        }
                                        // Add a dummy line, if the next tag is a BR tag again. The line could
                                        // not be empty otherwise it will be removed and the empty line is not displayed
                                        if (element.nextSibling && element.nextSibling.nodeName === 'BR') {
                                            currentParagraph.text.push(create('text', ' '));
                                        }
                                        currentParagraph.lineHeight = 1.25;
                                        alreadyConverted.push(currentParagraph);
                                    }
                                    break;
                                case "li":
                                case "div":
                                    currentParagraph = create("text");
                                    currentParagraph.lineHeight = 1.25;
                                    var stackDiv = create("stack");
                                    if (_.indexOf(classes, 'os-split-before') > -1) {
                                        stackDiv.listType = 'none';
                                    }
                                    if (nodeName === 'li') {
                                        stackDiv.marginBottom = LI_MARGIN_BOTTOM;
                                    }
                                    stackDiv.stack.push(currentParagraph);
                                    ComputeStyle(stackDiv, styles);
                                    currentParagraph = parseChildren(stackDiv.stack, element, currentParagraph, [], diff_mode);
                                    alreadyConverted.push(stackDiv);
                                    break;
                                case "p":
                                    var pObjectToPush; //determine what to push later
                                    currentParagraph = create("text");
                                    // If this element is inside a list (happens if copied from word), do not set spaces
                                    // and margins. Just leave the paragraph there..
                                    if (!isInsideAList(element)) {
                                        currentParagraph.margin = [0, 0, 0, 0];
                                        if (classes.indexOf('os-split-before') === -1) {
                                            currentParagraph.margin[1] = 8;
                                        }
                                        if (classes.indexOf('insert') > -1) {
                                            currentParagraph.margin[0] = 20;
                                        }
                                    }
                                    currentParagraph.lineHeight = 1.25;
                                    var stackP = create("stack");
                                    stackP.stack.push(currentParagraph);
                                    ComputeStyle(stackP, styles);
                                    currentParagraph = parseChildren(stackP.stack, element, currentParagraph, [], diff_mode);
                                    pObjectToPush = stackP; //usually we want to push stackP
                                    if (lineNumberMode === "outside") {
                                        if (element.childNodes.length > 0) { //if we hit = 0, the code would fail
                                            // add empty line number column for inline diff or pragraph diff mode
                                            if (element.childNodes[0].tagName === "INS" ||
                                                element.childNodes[0].tagName === "DEL") {
                                                var pLineNumberPlaceholder = {
                                                    width: 20,
                                                    text: "",
                                                    fontSize: 8,
                                                    margin: [0, 2, 0, 0]
                                                };
                                                var pLineNumberPlaceholderCol = {
                                                    columns: [
                                                        pLineNumberPlaceholder,
                                                        stackP
                                                    ]
                                                };
                                                pObjectToPush = pLineNumberPlaceholderCol; //overwrite the object to push
                                            }
                                        }
                                    }
                                    alreadyConverted.push(pObjectToPush);
                                    break;
                                case "img":
                                    var path = element.getAttribute('src');
                                    var height = images[path].height;
                                    var width = images[path].width;
                                    var maxWidth = 450;
                                    var scale = 100;

                                    var style = element.getAttribute('style');
                                    if (style) {
                                        var match = style.match(/width:\s*(\d+)\%/);
                                        if (match) {
                                            scale = parseInt(match[1]);
                                        }
                                    }

                                    // scale image
                                    width = (width * scale) / 100;
                                    height = (height * scale) / 100;

                                    if (width > maxWidth) {
                                        height = (height * maxWidth) / width;
                                        width = maxWidth;
                                    }

                                    // remove trailing / for the virtual file system (there is no root)
                                    if (path.indexOf('/') === 0) {
                                        path = path.substr(1);
                                    }
                                    alreadyConverted.push({
                                        image: path,
                                        width: width,
                                        height: height,
                                    });
                                    break;
                                case "ul":
                                case "ol":
                                    var list = create(nodeName);
                                    if (nodeName == 'ol') {
                                        var start = element.getAttribute('start');
                                        if (start) {
                                            list.start = start;
                                        }
                                    }
                                    ComputeStyle(list, styles);
                                    if (lineNumberMode === "outside") {
                                        var lines = extractLineNumbers(element);
                                        currentParagraph = parseChildren(list[nodeName], element, currentParagraph, styles, diff_mode);
                                        if (lines.length > 0) {
                                            var listCol = {
                                                columns: [{
                                                    width: 20,
                                                    stack: []
                                                }]
                                            };
                                            _.forEach(lines, function(line) {
                                                listCol.columns[0].stack.push(getLineNumberObject(line));
                                            });
                                            listCol.columns.push(list);
                                            if (!hasClass(element, 'os-split-before')) {
                                                listCol.margin = [0, 5, 0, 0];
                                            }
                                            alreadyConverted.push(listCol);
                                        } else {
                                            list.margin = [20, 0, 0, 0];
                                            alreadyConverted.push(list);
                                        }
                                    } else {
                                        list.margin = [0, LI_MARGIN_BOTTOM, 0, 0];
                                        currentParagraph = parseChildren(list[nodeName], element, currentParagraph, styles, diff_mode);
                                        alreadyConverted.push(list);
                                    }
                                    break;
                                default:
                                    var defaultText = create("text", element.textContent.replace(/\n/g, ""));
                                    ComputeStyle(defaultText, styles);
                                    if (!currentParagraph) {
                                        currentParagraph = {};
                                        currentParagraph.text = [];
                                    }
                                    currentParagraph.text.push(defaultText);
                                    break;
                            }
                            return currentParagraph;
                        },
                        /**
                         * Parses HTML
                         * @function
                         * @param {string} converted      -
                         * @param {object} htmlText   -
                         */
                        ParseHtml = function(converted, htmlText) {
                            var html = HTMLValidizer.validize(htmlText);
                            html = $(html.replace(/\t/g, "").replace(/\n/g, ""));
                            var emptyParagraph = create("text");
                            slice(html).forEach(function(element) {
                                ParseElement(converted, element, null, [], DIFF_MODE_NORMAL);
                            });
                        },
                        /* Returns the object to push first into every column, that represents the given line. */
                        getLineNumberObject = function (line) {
                            var standardFontsize = Config.get('general_export_pdf_fontsize').value;
                            return {
                                width: 20,
                                text: [
                                    {
                                        text: ' ', // Add a blank with the normal font size here, so in rare cases the text
                                                   // is rendered on the next page and the linenumber on the previous page.
                                        fontSize: standardFontsize,
                                        decoration: '',
                                    },
                                    {
                                        text: line.lineNumber,
                                        color: "gray",
                                        fontSize: standardFontsize - 2,
                                        decoration: '',
                                    },
                                ],
                                marginBottom: line.marginBottom,
                                lineHeight: 1.25,
                            };
                        },
                        content = [];
                    ParseHtml(content, html);
                    return content;
                },
                /**
                 * Creates containerelements for pdfMake
                 * e.g create("text":"MyText") result in { text: "MyText" }
                 * or complex objects create("stack", [{text:"MyText"}, {text:"MyText2"}])
                 *for units / paragraphs of text
                 *
                 * @function
                 * @param {string} name      - name of the attribute holding content
                 * @param {object} content   - the actual content (maybe empty)
                 */
                create = function(name, content) {
                    var o = {};
                    content = content || [];
                    o[name] = content;
                    return o;
                };
            return {
                convertHTML: convertHTML,
                createElement: create
            };
        };
        return {
            createInstance: createInstance
        };
    }
])

.factory('ImageConverter', [
    '$q',
    'PDFLayout',
    function ($q, PDFLayout) {
        return {
            toBase64: function (imageSources) {
                var imageMap = {};
                var imagePromises = _.map(imageSources, function (imageSource) {
                    if (imageSource) {
                        return PDFLayout.imageURLtoBase64(imageSource).then(function (imgInfo) {
                            imageMap[imageSource] = imgInfo;
                        });
                    }
                });

                return $q(function (resolve, reject) {
                    //resolve promises to get base64
                    $q.all(imagePromises).then(function() {
                        resolve(imageMap);
                    }, reject);
                });
            }
        };
    }
])

// Creates the virtual filesystem for PdfMake.
.factory('PdfVfs', [
    '$q',
    '$http',
    'Fonts',
    'Config',
    function ($q, $http, Fonts, Config) {
        var urlCache = {}; // Caches the get request. Maps urls to base64 data ready to use.

        var loadFont = function (url) {
            return $q(function (resolve, reject) {
                // Get font
                return $http.get(url, {responseType: 'blob'}).then(function (success) {
                    // Convert to base64
                    var reader = new FileReader();
                    reader.readAsDataURL(success.data);
                    reader.onloadend = function() {
                        resolve(reader.result.split(',')[1]);
                    };
                }, function (error) {
                    reject(error);
                });
            });
        };

        /*
         * Returns a map from urls to arrays of font types used by PdfMake.
         * E.g. if the font "regular" and bold" have the urls "fonts/myFont.ttf",
         * the map fould be "fonts/myFont.ttf": ["OSFont-regular.ttf", "OSFont-bold.ttf"]
         */
        var getUrlMapping = function () {
            var urlMap = {};
            var fonts = ['regular', 'italic', 'bold', 'bold_italic'];
            _.forEach(fonts, function (font) {
                var url = Fonts.getUrl('font_' + font);
                if (!urlMap[url]) {
                    urlMap[url] = [];
                }
                urlMap[url].push('OSFont-' + font + '.ttf');
            });
            return urlMap;
        };

        /*
         * Create the virtual filesystem needed by PdfMake for the fonts. Gets the url
         * mapping and loads all fonts via get requests or the urlCache.
         * Adds all image sources to the vfs given by the imageMap.
         */
        var getVfs = function (imageMap) {
            var vfs = {};
            _.forEach(imageMap || {}, function (data, path) {
                if (path.indexOf('/') === 0) {
                    path = path.substr(1); // remove trailing /
                }
                vfs[path] = data.data.split(',')[1];
            });
            return $q(function (resolve, reject) {
                var urls = getUrlMapping();
                var promises = _.chain(urls)
                    .map(function (filenames, url) {
                        if (urlCache[url]) {
                            // Just save the cache data into vfs.
                            _.forEach(filenames, function (filename) {
                                vfs[filename] = urlCache[url];
                            });
                            return false; // No promise here, it was all cached.
                        } else {
                            // Not in the cache, get the font and save the data into vfs.
                            return loadFont(url).then(function (data) {
                                urlCache[url] = data;
                                _.forEach(filenames, function (filename) {
                                    vfs[filename] = data;
                                });
                            });
                        }
                    })
                    .filter(function (promise) {
                        return promise;
                    })
                    .value();
                $q.all(promises).then(function () {
                    resolve(vfs);
                });
            });
        };

        return {
            get: getVfs,
        };
    }
])

.factory('PdfCreate', [
    '$timeout',
    '$q',
    'gettextCatalog',
    'FileSaver',
    'PdfVfs',
    'Messaging',
    function ($timeout, $q, gettextCatalog, FileSaver, PdfVfs, Messaging) {
        var filenameMessageMap = {};
        var b64toBlob = function(b64Data) {
            var byteCharacters = atob(b64Data);
            var byteNumbers = new Array(byteCharacters.length);
            for (var i = 0; i < byteCharacters.length; i++) {
                byteNumbers[i] = byteCharacters.charCodeAt(i);
            }
            var byteArray = new Uint8Array(byteNumbers);
            var blob = new Blob([byteArray]);
            return blob;
        };
        var stateChange = function (state, filename, error) {
            var text, timeout;
            switch (state) {
                case 'info':
                    text = '<i class="fa fa-spinner fa-pulse fa-lg spacer-right"></i>' +
                        gettextCatalog.getString('Generating PDF file') + ' (' + filename + ') ...';
                    break;
                case 'success':
                    text = '<i class="fa fa-check fa-lg spacer-right"></i>' +
                        gettextCatalog.getString('PDF successfully generated.');
                    timeout = 3000;
                    break;
                case 'error':
                    text = '<i class="fa fa-exclamation-triangle fa-lg spacer-right"></i>' +
                        gettextCatalog.getString('Error while generating PDF file') +
                        ' (' + filename + '): <code>' + error + '</code>';
                    break;
            }
            $timeout(function () {
                filenameMessageMap[filename] = Messaging.createOrEditMessage(
                    filenameMessageMap[filename], text, state, {timeout: timeout});
            }, 1);
        };
        return {
            getBase64FromDocument: function (documentProvider) {
                return $q(function (resolve, reject) {
                    PdfVfs.get(documentProvider.getImageMap()).then(function (vfs) {
                        var pdfWorker = new Worker('/static/js/workers/pdf-worker.js');
                        pdfWorker.addEventListener('message', function (event) {
                            resolve(event.data);
                        });
                        pdfWorker.addEventListener('error', function (event) {
                            reject(event);
                        });
                        pdfWorker.postMessage(JSON.stringify({
                            pdfDocument: documentProvider.getDocument(),
                            vfs: vfs,
                        }));
                    });
                });
            },
            // Struckture of pdfDocuments: { filname1: doc, filename2: doc, ...}
            getBase64FromMultipleDocuments: function (pdfDocuments) {
                // concat all image sources together
                var imageMap = {};
                _.forEach(pdfDocuments, function (doc) {
                    _.forEach(doc.getImageMap(), function (data, path) {
                        if (!imageMap[path]) {
                            imageMap[path] = data;
                        }
                    });
                });
                return $q(function (resolve, reject) {
                    PdfVfs.get(imageMap).then(function (vfs) {
                        var pdfWorker = new Worker('/static/js/workers/pdf-worker.js');
                        var resultCount = 0;
                        var base64Map = {}; // Maps filename to base64
                        pdfWorker.addEventListener('message', function (event) {
                            resultCount++;
                            var data = JSON.parse(event.data);
                            base64Map[data.filename] = data.base64;
                            if (resultCount === _.keys(pdfDocuments).length) {
                                resolve(base64Map);
                            }
                        });
                        pdfWorker.addEventListener('error', function (event) {
                            reject(event);
                        });
                        _.forEach(pdfDocuments, function (doc, filename) {
                            pdfWorker.postMessage(JSON.stringify({
                                filename: filename,
                                pdfDocument: doc.getDocument(),
                                vfs: vfs,
                            }));
                        });
                    });
                });
            },
            download: function (documentProvider, filename) {
                stateChange('info', filename);

                this.getBase64FromDocument(documentProvider).then(function (data) {
                    var blob = b64toBlob(data);
                    stateChange('success', filename);
                    FileSaver.saveAs(blob, filename);
                }, function (error) {
                    stateChange('error', filename, error.message);
                });
            },
        };
    }
]);

}());

(function () {

'use strict';

// The core module for the OpenSlides projector
angular.module('OpenSlidesApp.core.projector', ['OpenSlidesApp.core'])

// Can be used to find out if the projector or the side is used
.constant('REALM', 'projector')

.run([
    '$http',
    'autoupdate',
    'DS',
    function ($http, autoupdate, DS) {
        autoupdate.newConnect();

        // If the connection aborts, we try to ping the server with whoami requests. If
        // the server is flushed, we clear the datastore, so the message 'this projector
        // cannot be shown' will be displayed. Otherwise establish the websocket connection.
        autoupdate.registerRetryConnectCallback(function () {
            return $http.get('/users/whoami').then(function (success) {
                if (success.data.user_id === null && !success.data.guest_enabled) {
                    DS.clear();
                } else {
                    autoupdate.newConnect();
                }
            });
        });
    }
])

// Provider to register slides in a .config() statement.
.provider('slides', [
    function() {
        var slidesMap = {};

        this.registerSlide = function(name, config) {
            slidesMap[name] = config;
            return this;
        };

        this.$get = function($templateRequest, $q) {
            var self = this;
            return {
                getElements: function(projector) {
                    var elements = [];
                    var factory = this;
                    _.forEach(projector.elements, function(element) {
                        if (element.name in slidesMap) {
                            element.template = slidesMap[element.name].template;
                            elements.push(element);
                        } else {
                            console.error("Unknown slide: " + element.name);
                        }
                    });
                    return elements;
                }
            };
        };
    }
])

.config([
    'slidesProvider',
    function(slidesProvider) {
        slidesProvider.registerSlide('core/clock', {
            template: 'static/templates/core/slide_clock.html',
        });

        slidesProvider.registerSlide('core/countdown', {
            template: 'static/templates/core/slide_countdown.html',
        });

        slidesProvider.registerSlide('core/projector-message', {
            template: 'static/templates/core/slide_message.html',
        });
    }
])

.controller('LanguageAndFontCtrl', [
    '$scope',
    'Languages',
    'Config',
    'Projector',
    'ProjectorID',
    'Fonts',
    function ($scope, Languages, Config, Projector, ProjectorID, Fonts) {
        // for the dynamic title
        $scope.projectorId = ProjectorID();
        $scope.$watch(function () {
            return Projector.lastModified($scope.projectorId);
        }, function () {
            var projector = Projector.get($scope.projectorId);
            if (projector) {
                $scope.projectorName = projector.name;
            }
        });

        $scope.$watch(function () {
            return Config.lastModified('projector_language');
        }, function () {
            var lang = Config.get('projector_language');
            if (!lang || lang.value == 'browser') {
                $scope.selectedLanguage = Languages.getBrowserLanguage();
            } else {
                $scope.selectedLanguage = lang.value;
            }
            Languages.setCurrentLanguage($scope.selectedLanguage);
        });

        $scope.$watch(function () {
            return Config.lastModified('font_regular') +
                Config.lastModified('font_italic') +
                Config.lastModified('font_bold') +
                Config.lastModified('font_bold_italic');
        }, function () {
            $scope.font = Fonts.getForCss('font_regular');
            $scope.font_medium = Fonts.getForCss('font_italic');
            $scope.font_condensed = Fonts.getForCss('font_bold');
            $scope.font_condensed_light = Fonts.getForCss('font_bold_italic');
        });
    }
])

// Projector Container Controller
.controller('ProjectorContainerCtrl', [
    '$scope',
    '$timeout',
    '$location',
    'gettext',
    'Projector',
    function($scope, $timeout, $location, gettext, Projector) {
        $scope.showError = true;

        // watch for changes in Projector
        $scope.$watch(function () {
            return Projector.lastModified($scope.projectorId);
        }, function () {
            var projector = Projector.get($scope.projectorId);
            if (projector) {
                $scope.showError = false;
                $scope.projectorWidth = projector.width;
                $scope.projectorHeight = projector.height;
                $scope.recalculateIframe();
            } else {
                $scope.showError = true;
                // delay displaying the error message, because with a slow internet
                // connection, the autoupdate with the projector may be delayed. We
                // de not want to irritate the user by showing this error to early.
                $scope.error = '';
                $timeout(function () {
                    if ($scope.showError) {
                        $scope.error = gettext('Can not open the projector.');
                    }
                }, 3000);
            }
        });

        // recalculate the actual Iframesize and scale
        $scope.recalculateIframe = function () {
            var scale_width = window.innerWidth / $scope.projectorWidth;
            var scale_height = window.innerHeight / $scope.projectorHeight;

            // Iframe has to be scaled down or saceUp is activated
            if (scale_width <= scale_height) {
                // width is the reference
                $scope.iframeWidth = window.innerWidth;
                $scope.scale = scale_width;
                $scope.iframeHeight = $scope.projectorHeight * scale_width;
            } else {
                // height is the reference
                $scope.iframeHeight = window.innerHeight;
                $scope.scale = scale_height;
                $scope.iframeWidth = $scope.projectorWidth * scale_height;
            }
        };

        // watch for changes in the windowsize
        $(window).on("resize.doResize", function () {
            $scope.$apply(function() {
                $scope.recalculateIframe();
            });
        });

        $scope.$on("$destroy",function (){
            $(window).off("resize.doResize");
        });
    }
])

.controller('ProjectorCtrl', [
    '$scope',
    '$location',
    '$timeout',
    'Projector',
    'slides',
    'Config',
    'ProjectorID',
    'Logos',
    function($scope, $location, $timeout, Projector, slides, Config, ProjectorID, Logos) {
        var projectorId = ProjectorID();

        $scope.broadcast = 0;

        var setElements = function (projector) {
            // Get all elements, that should be projected.
            var newElements = [];
            var enable_clock = Config.get('projector_enable_clock');
            enable_clock = enable_clock ? enable_clock.value : true;
            _.forEach(slides.getElements(projector), function (element) {
                if (!element.error) {
                    // Exclude the clock if it should be disabled.
                    if (enable_clock || element.name !== 'core/clock') {
                        newElements.push(element);
                    }
                } else {
                    console.error("Error for slide " + element.name + ": " + element.error);
                }
            });

            // Now we have to align $scope.elements to newElements:
            // We cannot just assign them, because the ng-repeat would reload every
            // element. This should be prevented (see #3259). To change $scope.elements:
            // 1) remove all elements from scope, that are not in newElements (compared by the uuid)
            // 2) Every new element in newElements, that is not in $scope.elements, get inserted there.
            // 3) If there is the same element in newElements and $scope.elements every changed property
            //    is copied from the new element to the scope element.

            $scope.elements = _.filter($scope.elements, function (element) {
                return _.some(newElements, function (newElement) {
                    return element.uuid === newElement.uuid;
                });
            });

            _.forEach(newElements, function (newElement) {
                var matchingElement = _.find($scope.elements, function (element) {
                    return element.uuid === newElement.uuid;
                });
                if (matchingElement) {
                    // copy all changed properties.
                    _.forEach(newElement, function (value, key) {
                        // key has own property and does not start with a '$'.
                        if (newElement.hasOwnProperty(key) && key.indexOf('$') != 0) {
                            if (typeof matchingElement[key] === 'undefined' || matchingElement[key] !== value) {
                                matchingElement[key] = value;
                            }
                        }
                    });
                } else {
                    $scope.elements.push(newElement);
                }
            });
        };

        $scope.scroll = 0;
        var setScroll = function (scroll) {
            $scope.scroll = -250 * scroll;
        };

        $scope.$watch(function () {
            return Projector.lastModified(projectorId);
        }, function () {
            $scope.projector = Projector.get(projectorId);
            if ($scope.projector) {
                if ($scope.broadcast === 0) {
                    setElements($scope.projector);
                    $scope.blank = $scope.projector.blank;
                }
                setScroll($scope.projector.scroll);
            } else {
                // Blank projector on error
                $scope.elements = [];
                $scope.projector = {
                    scale: 0,
                    blank: true
                };
                setScroll(0);
            }
        });

        $scope.$watch(function () {
            return Config.lastModified('projector_broadcast');
        }, function () {
            var bc = Config.get('projector_broadcast');
            if (bc) {
                if ($scope.broadcast != bc.value) {
                    $scope.broadcast = bc.value;
                    if ($scope.broadcastDeregister) {
                        // revert to original $scope.projector
                        $scope.broadcastDeregister();
                        $scope.broadcastDeregister = null;
                        setElements($scope.projector);
                        $scope.blank = $scope.projector.blank;
                    }
                }
                if ($scope.broadcast > 0) {
                    // get elements and blank from broadcast projector
                    $scope.broadcastDeregister = $scope.$watch(function () {
                        return Projector.lastModified($scope.broadcast);
                    }, function () {
                        if ($scope.broadcast > 0) {
                            var broadcast_projector = Projector.get($scope.broadcast);
                            if (broadcast_projector) {
                                setElements(broadcast_projector);
                                $scope.blank = broadcast_projector.blank;
                            }
                        }
                    });
                }
            }
        });

        $scope.$watch(function () {
            return Config.lastModified('projector_enable_clock');
        }, function () {
            setElements($scope.projector);
        });

        $scope.$on('$destroy', function() {
            if ($scope.broadcastDeregister) {
                $scope.broadcastDeregister();
                $scope.broadcastDeregister = null;
            }
        });
    }
])

.controller('SlideClockCtrl', [
    '$scope',
    '$interval',
    function($scope, $interval) {
        // Attention! Each object that is used here has to be dealt on server side.
        // Add it to the coresponding get_requirements method of the ProjectorElement
        // class.
        $scope.servertime = ( Date.now() / 1000 - $scope.serverOffset ) * 1000;
        var interval = $interval(function () {
            $scope.servertime = ( Date.now() / 1000 - $scope.serverOffset ) * 1000;
        }, 30000); // Update the clock every 30 seconds

        $scope.$on('$destroy', function() {
            if (interval) {
                $interval.cancel(interval);
            }
        });
    }
])

.controller('SlideCountdownCtrl', [
    '$scope',
    '$interval',
    'Countdown',
    function($scope, $interval, Countdown) {
        // Attention! Each object that is used here has to be dealt on server side.
        // Add it to the coresponding get_requirements method of the ProjectorElement
        // class.
        var id = $scope.element.id;
        var interval;
        var calculateCountdownTime = function (countdown) {
            countdown.seconds = Math.floor( $scope.countdown.countdown_time - Date.now() / 1000 + $scope.serverOffset );
        };
        $scope.$watch(function () {
            return Countdown.lastModified(id);
        }, function () {
            $scope.countdown = Countdown.get(id);
            if (interval) {
                $interval.cancel(interval);
            }
            if ($scope.countdown) {
                if ($scope.countdown.running) {
                    calculateCountdownTime($scope.countdown);
                    interval = $interval(function () { calculateCountdownTime($scope.countdown); }, 1000);
                } else {
                    $scope.countdown.seconds = $scope.countdown.countdown_time;
                }
            }
        });
        $scope.$on('$destroy', function() {
            // Cancel the interval if the controller is destroyed
            if (interval) {
                $interval.cancel(interval);
            }
        });
    }
])

.controller('SlideMessageCtrl', [
    '$scope',
    'ProjectorMessage',
    'Projector',
    'ProjectorID',
    'gettextCatalog',
    function($scope, ProjectorMessage, Projector, ProjectorID, gettextCatalog) {
        // Attention! Each object that is used here has to be dealt on server side.
        // Add it to the coresponding get_requirements method of the ProjectorElement
        // class.
        var id = $scope.element.id;

        if ($scope.element.identify) {
            var projector = Projector.get(ProjectorID());
            $scope.identifyMessage = gettextCatalog.getString('Projector') + ' ' + projector.id + ': ' + gettextCatalog.getString(projector.name);
        } else {
            $scope.message = ProjectorMessage.get(id);
            ProjectorMessage.bindOne(id, $scope, 'message');
        }
    }
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.core.remove-format-plugin', [
    'OpenSlidesApp.core',
])

/*
 * Plugin for the CKEditor that hooks into the removeformat plugin
 * which is a default plugin enabled by 'cleanup' in the config
 * toolbar.
 * We change the behavior of the removeformat command here:
 * It should not remove any tags and styles, but only the
 * 'DISALLOWED_STYLES'. Removeformat traverses through the DOM
 * and calles for every element the custom filter down below.
 * We change the element and return false, so the removeformat
 * plugin does not clean it up.
 */
.factory('OSRemoveFormatPlugin', [
    'Editor',
    'gettextCatalog',
    function (Editor, gettextCatalog) {
        var DISALLOWED_STYLES = ['color', 'background-color'];
        return {
            getPlugin: function () {
                return {
                    init: function (editor) {
                        editor.addRemoveFormatFilter(function (element) {
                            _.forEach(DISALLOWED_STYLES, function (style) {
                                element.removeStyle(style);
                            });
                            return false;
                        });
                    },
                };
            },
        };
    }
])

.run([
    'Editor',
    'OSRemoveFormatPlugin',
    function (Editor, OSRemoveFormatPlugin, gettext) {
        Editor.registerPlugin('OSRemoveFormat', OSRemoveFormatPlugin.getPlugin());
    }
]);

}());

(function () {

'use strict';

// The core module for the OpenSlides site
angular.module('OpenSlidesApp.core.site', [
    'OpenSlidesApp.core',
    'OpenSlidesApp.core.start',
    'OpenSlidesApp.core.csv',
    'OpenSlidesApp.core.remove-format-plugin',
    'OpenSlidesApp.poll.majority',
    'ui.router',
    'colorpicker.module',
    'formly',
    'formlyBootstrap',
    'localytics.directives',
    'ngDialog',
    'ngFileSaver',
    'ngMessages',
    'ckeditor',
    'luegg.directives',
    'xeditable',
    'rzModule',
])

// Can be used to find out if the projector or the side is used
.constant('REALM', 'site')

.factory('DateTimePickerTranslation', [
    'gettextCatalog',
    function (gettextCatalog) {
        return {
            getButtons: function () {
                return {
                    show: true,
                    now: {
                        show: true,
                        text: gettextCatalog.getString('now')
                    },
                    today: {
                        show: true,
                        text: gettextCatalog.getString('today')
                    },
                    clear: {
                        show: true,
                        text: gettextCatalog.getString('clear')
                    },
                    date: {
                        show: true,
                        text: gettextCatalog.getString('date')
                    },
                    time: {
                        show: true,
                        text: gettextCatalog.getString('time')
                    },
                    close: {
                        show: true,
                        text: gettextCatalog.getString('close')
                    }
                };
            }
        };
    }

])

// Provider to register entries for the main menu.
.provider('mainMenu', [
    function() {
        var mainMenuList = [];
        var scope;

        this.register = function(config) {
            mainMenuList.push(config);
        };

        this.$get = ['operator', function(operator) {
            return {
                registerScope: function (scope) {
                    this.scope = scope;
                },
                updateMainMenu: function () {
                    if (this.scope) {
                        this.scope.elements = this.getElements();
                    }
                },
                getElements: function() {
                    var elements = mainMenuList.filter(function (element) {
                        return typeof element.perm === "undefined" || operator.hasPerms(element.perm);
                    });

                    elements.sort(function (a, b) {
                        return a.weight - b.weight;
                    });
                    return elements;
                }
            };
        }];
    }
])

// Provider to register a searchable module/app.
.provider('Search', [
    function() {
        var searchModules = [];

        this.register = function(module) {
            searchModules.push(module);
        };

        this.$get = [
            function () {
                return {
                    getAll: function () {
                        return searchModules;
                    }
                };
            }
        ];
    }
])

.run([
    'editableOptions',
    'gettext',
    function (editableOptions, gettext) {
        editableOptions.theme = 'bs3';
        editableOptions.cancelButtonAriaLabel = gettext('Cancel');
        editableOptions.cancelButtonTitle = gettext('Cancel');
        editableOptions.clearButtonAriaLabel = gettext('Clear');
        editableOptions.clearButtonTitle = gettext('Clear');
        editableOptions.submitButtonAriaLabel = gettext('Submit');
        editableOptions.submitButtonTitle = gettext('Submit');
    }
])

.factory('WebpageTitle', [
    '$rootScope',
    function ($rootScope) {
        $rootScope.activeAppTitle = '';
        return {
            updateTitle: function (text) {
                $rootScope.activeAppTitle = text || '';
            },
        };
    }
])

// Watch for the basePerm on a stateChange and initialize the WebpageTitle factory
.run([
    '$rootScope',
    'operator',
    'WebpageTitle',
    function ($rootScope, operator, WebpageTitle) {
        $rootScope.$on('$stateChangeSuccess', function(event, toState) {
            WebpageTitle.updateTitle(toState.data ? toState.data.title : '');
            if (toState.data) {
                $rootScope.baseViewPermissionsGranted = toState.data.basePerm ?
                    operator.hasPerms(toState.data.basePerm) : true;
            } else {
                $rootScope.baseViewPermissionsGranted = true;
            }
            // Scroll to top on every state change
            $rootScope.gotoTop();
        });
    }
])

.config([
    'mainMenuProvider',
    'gettext',
    function (mainMenuProvider, gettext) {
        mainMenuProvider.register({
            'ui_sref': 'home',
            'img_class': 'home',
            'title': gettext('Home'),
            'weight': 100,
            'perm': 'core.can_see_frontpage',
        });

        mainMenuProvider.register({
            'ui_sref': 'config',
            'img_class': 'cog',
            'title': gettext('Settings'),
            'weight': 1000,
            'perm': 'core.can_manage_config',
        });
    }
])

.config([
    '$urlRouterProvider',
    '$locationProvider',
    function($urlRouterProvider, $locationProvider) {
        // define fallback url and html5Mode
        $urlRouterProvider.otherwise('/');
        $locationProvider.html5Mode(true);
    }
])

.config([
    '$httpProvider',
    function($httpProvider) {
        // Combine the django csrf system with the angular csrf system
        $httpProvider.defaults.xsrfCookieName = 'OpenSlidesCsrfToken';
        $httpProvider.defaults.xsrfHeaderName = 'X-CSRFToken';
    }
])


.config([
    '$stateProvider',
    '$urlMatcherFactoryProvider',
    function($stateProvider, $urlMatcherFactoryProvider) {
        // Make the trailing slash optional
        $urlMatcherFactoryProvider.strictMode(false);

        // Use stateProvider.decorator to give default values to our states
        $stateProvider.decorator('views', function(state, parent) {
            var result = {},
                views = parent(state);

            if (state.abstract || state.data && state.data.extern) {
                return views;
            }

            angular.forEach(views, function(config, name) {
                // Sets additional default values for templateUrl
                var templateUrl,
                    controller,
                    defaultControllers = {
                        create: 'CreateCtrl',
                        update: 'UpdateCtrl',
                        list: 'ListCtrl',
                        detail: 'DetailCtrl',
                    };

                // Split up state name
                // example: "motions.motion.detail.update" -> ['motions', 'motion', 'detail', 'update']
                var patterns = state.name.split('.');

                // set app and module name from state
                // - appName: patterns[0] (e.g. "motions")
                // - moduleNames: patterns without first element (e.g. ["motion", "detail", "update"])
                var appName = '';
                var moduleName = '';
                var moduleNames = [];
                if (patterns.length > 0) {
                    appName = patterns[0];
                    moduleNames = patterns.slice(1);
                }
                if (moduleNames.length > 0) {
                    // convert from camcelcase to dash notation
                    // example: ["motionBlock", "detail"] -> ["motion-block", "detail"]
                    for (var i = 0; i < moduleNames.length; i++) {
                        moduleNames[i] =  moduleNames[i].replace(/([a-z\d])([A-Z])/g, '$1-$2').toLowerCase();
                    }

                    // use special templateUrl for create and update view
                    // example: ["motion", "detail", "update"] -> "motion-form"
                    if (_.last(moduleNames).match(/(create|update)/)) {
                        moduleName = '/' + moduleNames[0] + '-form';
                    } else {
                        // convert modelNames array to url string
                        // example: ["motion-block", "detail"] -> "motion-block-detail"
                        moduleName = '/' + moduleNames.join('-');
                    }
                }
                templateUrl = 'static/templates/' + appName + moduleName + '.html';
                config.templateUrl = state.templateUrl || templateUrl;

                // controller
                if (patterns.length >= 3) {
                    controller = _.upperFirst(patterns[1]) + defaultControllers[_.last(patterns)];
                    config.controller = state.controller || controller;
                }
                result[name] = config;
            });
            return result;
        })

        .decorator('url', function(state, parent) {
            var defaultUrl;

            if (state.abstract) {
                defaultUrl = '';
            } else {
                var patterns = state.name.split('.'),
                    defaultUrls = {
                        create: '/new',
                        update: '/edit',
                        list: '',
                        // The id is expected to be an integer, if not, the url has to
                        // be defined manually
                        detail: '/{id:int}',
                    };

                defaultUrl = defaultUrls[_.last(patterns)];
            }

            state.url = state.url || defaultUrl;
            return parent(state);
        });
    }
])

.config([
    '$stateProvider',
    '$locationProvider',
    'gettext',
    function($stateProvider, $locationProvider, gettext) {
        // Core urls
        $stateProvider
            .state('home', {
                url: '/',
                templateUrl: 'static/templates/home.html',
                data: {
                    title: gettext('Home'),
                    basePerm: 'core.can_see_frontpage',
                },
            })
            .state('projector', {
                url: '/projector/{id:int}/',
                templateUrl: 'static/templates/projector-container.html',
                data: {extern: true},
                onEnter: function($window) {
                    $window.location.href = this.url;
                }
            })
            .state('real-projector', {
                url: '/real-projector/{id:int}/',
                templateUrl: 'static/templates/projector.html',
                data: {extern: true},
                onEnter: function($window) {
                    $window.location.href = this.url;
                }
            })
            .state('manage-projectors', {
                url: '/manage-projectors',
                templateUrl: 'static/templates/core/manage-projectors.html',
                controller: 'ManageProjectorsCtrl',
                data: {
                    title: gettext('Manage projectors'),
                    basePerm: 'core.can_manage_projector',
                },
            })
            .state('core', {
                url: '/core',
                abstract: true,
                template: "<ui-view/>",
            })

            // legal notice and version
            .state('legalnotice', {
                url: '/legalnotice',
                controller: 'LegalNoticeCtrl',
                data: {
                    title: gettext('Legal notice'),
                },
            })

            // privacy policy
            .state('privacypolicy', {
                url: '/privacypolicy',
                controller: 'PrivacyPolicyCtrl',
                data: {
                    title: gettext('Privacy policy'),
                },
            })

            //config
            .state('config', {
                url: '/config',
                controller: 'ConfigCtrl',
                data: {
                    title: gettext('Settings'),
                    basePerm: 'core.can_manage_config',
                },
            })

            // search
            .state('search', {
                url: '/search?q',
                controller: 'SearchCtrl',
                templateUrl: 'static/templates/search.html',
                data: {
                    title: gettext('Search'),
                },
            })

            // tag
            .state('core.tag', {
                url: '/tag',
                abstract: true,
                template: "<ui-view/>",
                data: {
                    title: gettext('Tags'),
                    basePerm: 'core.can_manage_tags',
                },
            })
            .state('core.tag.list', {})

            // Countdown
            .state('core.countdown', {
                url: '/countdown',
                abstract: true,
                template: "<ui-view/>",
                data: {
                    title: gettext('Countdown'),
                    basePerm: 'core.can_manage_projector',
                },
            })
            .state('core.countdown.detail', {
                resolve: {
                    countdownId: ['$stateParams', function($stateParams) {
                        return $stateParams.id;
                    }],
                }
            });

        $locationProvider.html5Mode(true);
    }
])

.factory('ProjectorMessageForm', [
    'Editor',
    'gettextCatalog',
    function (Editor, gettextCatalog) {
        return {
            getDialog: function (message) {
                return {
                    template: 'static/templates/core/projector-message-form.html',
                    controller: 'ProjectorMessageEditCtrl',
                    className: 'ngdialog-theme-default wide-form',
                    closeByEscape: false,
                    closeByDocument: false,
                    resolve: {
                        projectorMessageId: function () {
                            return message.id;
                        }
                    },
                };
            },
            getFormFields: function () {
                return [
                    {
                        key: 'message',
                        type: 'editor',
                        templateOptions: {
                            label: gettextCatalog.getString('Message'),
                        },
                        data: {
                            ckeditorOptions: Editor.getOptions()
                        }
                    },
                ];
            },
        };
    }
])

.factory('TagForm', [
    'gettextCatalog',
    function (gettextCatalog) {
        return {
            getDialog: function (tag) {
                return {
                    template: 'static/templates/core/tag-form.html',
                    controller: (tag) ? 'TagUpdateCtrl' : 'TagCreateCtrl',
                    className: 'ngdialog-theme-default wide-form',
                    closeByEscape: false,
                    closeByDocument: false,
                    resolve: {
                        tagId: function () {return tag ? tag.id : void 0;},
                    },
                };
            },
            getFormFields: function() {
                return [
                    {
                        key: 'name',
                        type: 'input',
                        templateOptions: {
                            label: gettextCatalog.getString('Name'),
                            required: true
                        }
                    },
                ];
            },
        };
    }
])

/* This factory handles the filtering of the OS-data-tables. It contains
 * all logic needed for the table header filtering. Things to configure:
 * - multiselectFilters: A dict associating the filter name to a list (empty per default). E.g.
 *       { tag: [],
 *         category: [], }
 * - booleanFilters: A dict containing a dict for every filter. The value property is a must.
 *   For displaying properties like displayName, choiceYes and choiceNo could be usefull. E.g.
 *      { isPresent: {
 *          value: undefined,
 *          displayName: gettext('Is present'), } }
 * - propertyList, propertyFunctionList, propertyDict: See function getObjectQueryString
 */
.factory('osTableFilter', [
    '$sessionStorage',
    function ($sessionStorage) {
        var createInstance = function (tableName) {
            var self = {
                multiselectFilters: {},
                booleanFilters: {},
                filterString: '',
            };
            var existsStorageEntry = function () {
                return $sessionStorage[tableName];
            };
            var storage = existsStorageEntry();
            if (storage) {
                self = storage;
            }

            self.existsStorageEntry = existsStorageEntry;
            self.save = function () {
                $sessionStorage[tableName] = self;
                self.changed();
            };
            self.areFiltersSet = function () {
                var areFiltersSet = _.find(self.multiselectFilters, function (filterList) {
                    return filterList.length > 0;
                });
                areFiltersSet = areFiltersSet || _.find(self.booleanFilters, function (filterDict) {
                    return filterDict.value !== undefined;
                });
                areFiltersSet = areFiltersSet || (self.filterString !== '');
                return areFiltersSet !== false;
            };
            self.reset = function (danger) {
                if (danger) {
                    return;
                }
                _.forEach(self.multiselectFilters, function (filterList, filter) {
                    self.multiselectFilters[filter] = [];
                });
                _.forEach(self.booleanFilters, function (filterDict, filter) {
                    self.booleanFilters[filter].value = undefined;
                });
                self.filterString = '';
                self.save();
            };
            self.operateMultiselectFilter = function (filter, id, danger) {
                if (!danger) {
                    if (_.indexOf(self.multiselectFilters[filter], id) > -1) {
                        // remove id
                        self.multiselectFilters[filter].splice(_.indexOf(self.multiselectFilters[filter], id), 1);
                    } else {
                        // add id
                        self.multiselectFilters[filter].push(id);
                    }
                    self.save();
                }
            };
            /* Three things are could be given to create the query string:
             * - propertyList: Just a list of object's properties like ['title', 'name']
             * - propertyFunktionList: A list of functions returning a property (e.g. [function(motion) {return motion.getTitle();}] for retrieving the motions title)
             * - propertyDict: A dict association properties that are lists to functions on how to handle them.
             *   E.g.: {'tags': function (tag) {return tag.name;}, }
             *         The list of tags will be mapped with this function to a list of strings (tag names).
             */
            self.getObjectQueryString = function (obj) {
                var stringList = [];
                _.forEach(self.propertyList, function (property) {
                    stringList.push(obj[property]);
                });
                _.forEach(self.propertyFunctionList, function (fn) {
                    stringList.push(fn(obj));
                });
                _.forEach(self.propertyDict, function (idFunction, property) {
                    stringList.push(_.map(obj[property], idFunction).join(' '));
                });
                return stringList.join(' ');
            };
            // Stub for callback
            self.changed = function () {};
            return self;
        };

        return {
            createInstance: createInstance
        };
    }
])

/* This factory takes care of the sorting of OS-data-tables. Things to configure:
 * - column: the default column which is the list sorted by (e.g.
 *   instance.column='title')
 */
.factory('osTableSort', [
    '$sessionStorage',
    function ($sessionStorage) {
        var createInstance = function (tableName) {
            var self = {
                column: '',
                reverse: false,
            };
            var storage = $sessionStorage[tableName];
            if (storage) {
                self = storage;
            }

            self.save = function () {
                $sessionStorage[tableName] = self;
            };
            self.toggle = function (column) {
                if (self.column === column) {
                    self.reverse = !self.reverse;
                }
                self.column = column;
                self.save();
            };
            return self;
        };

        return {
            createInstance: createInstance
        };
    }
])

/* Factory for pagination of the tables. Saves all settings (currentPage, ...)
 * to the session storage and recovers them when the table is reloaded.
 * You have to provide a 'tableName' where the settings are saved in the session
 * storage. Has to be unique for obvious reasons.
 * The 'itemsPerPage' is optional. If not given, it defaults to 25.
 */
.factory('osTablePagination', [
    '$rootScope',
    '$sessionStorage',
    function ($rootScope, $sessionStorage) {
        var createInstance = function (tableName, itemsPerPage) {
            // Defaults
            var self = {
                currentPage: 1,
                itemsPerPage: itemsPerPage || 25,
                limitBegin: 0,
            };

            // Check storage; maybe recover old state.
            var storage = $sessionStorage[tableName];
            if (storage) {
                self = storage;
            }

            self.save = function () {
                $sessionStorage[tableName] = self;
            };
            self.pageChanged = function () {
                self.limitBegin = (self.currentPage - 1) * self.itemsPerPage;
                self.save();
                $rootScope.gotoTop();
            };
            self.getPageCount = function (objs) {
                if (objs) {
                    return Math.ceil(objs.length/self.itemsPerPage);
                }
            };
            self.showNextPageArrow = function (objs) {
                if (objs) {
                    return self.currentPage != self.getPageCount(objs);
                }
            };
            self.showPrevPageArrow = function () {
                return self.currentPage != 1;
            };
            self.nextPage = function (objs) {
                if (objs && self.currentPage < self.getPageCount(objs)) {
                    self.currentPage++;
                    self.pageChanged();
                }
            };
            self.prevPage = function () {
                if (self.currentPage > 1) {
                    self.currentPage--;
                    self.pageChanged();
                }
            };
            return self;
        };

        return {
            createInstance: createInstance
        };
    }
])

/* This Factory could be used in any dialog, if the user should be warned, if another user
 * also has this dialog open. Use it like in this example in any dialog controller:
      var editingStoppedCallback = EditingWarning.editingStarted('editing_name' + item.id);
      $scope.$on('$destroy', editingStoppedCallback);
 */
.factory('EditingWarning', [
    'operator',
    'gettextCatalog',
    'Notify',
    'Messaging',
    function (operator, gettextCatalog, Notify, Messaging) {
        return {
            // This returns the callback function that the controller should call, if
            // the dialog got closed by the user. Provide a unique dialog name.
            editingStarted: function (dialogName) {
                // List of active editors
                var editorNames = [];
                var messagingId = dialogName + 'EditingWarning';
                // Add an editor (may come from open_request or open_response)
                var addActiveEditor = function (editorName) {
                    editorNames.push(editorName);
                    updateActiveEditors();
                };
                // Remove an editor, if he closes his dialog (dialog_closed)
                var removeActiveEditor = function (editorName) {
                    var firstIndex = _.indexOf(editorNames, editorName);
                    editorNames.splice(firstIndex, 1);
                    updateActiveEditors();
                };
                // Show a warning.
                var updateActiveEditors = function () {
                    if (editorNames.length === 0) {
                        Messaging.deleteMessage(messagingId);
                    } else {
                        // This block is only for getting the message string together...
                        var editorsWithoutAnonymous = _.filter(editorNames, function (name) {
                            return name;
                        });
                        var text = gettextCatalog.getString('Warning') + ': ';
                        if (editorsWithoutAnonymous.length === 0) { // Only anonymous
                            // Singular vs. plural
                            if (editorNames.length === 1) {
                                text += gettextCatalog.getString('One anonymous users is also editing this.');
                            } else {
                                text += editorNames.length + ' ' + gettextCatalog.getString('anonymous users are also editing this.');
                            }
                        } else {
                            // At least one named user. The max users to display is 5. Anonymous users doesn't get displayed
                            // by name, but the amount of users is shown.
                            text += _.slice(editorsWithoutAnonymous, 0, 5).join(', ');
                            if (editorsWithoutAnonymous.length > 5) {
                                // More than 5 users with names.
                                text += ', ... [+' + (editorNames.length - 5) + ']';
                            } else if (editorsWithoutAnonymous.length !== editorNames.length) {
                                // Less than 5 users, so the difference is calculated different.
                                text += ', ... [+' + (editorNames.length - editorsWithoutAnonymous.length) + ']';
                            }
                            // Singular vs. plural
                            if (editorNames.length === 1) {
                                text += ' ' + gettextCatalog.getString('is also editing this.');
                            } else {
                                text += ' ' + gettextCatalog.getString('are also editing this.');
                            }
                        }
                        Messaging.createOrEditMessage(messagingId, text, 'warning');
                    }
                };

                // The stucture of determinating which users are editing this dialog:
                // - send an open_query to every user with the name of this operator in the parameter. With
                //   this information all clients that listen to this request knows that this operator has
                //   opened the dialog.
                // - The clients, which have recieved the query send an answer (open_resonse) to this operator.
                // - The operator collects all resonses and fills the editornames list.
                // - If the dialog get closed, a dialog_closed is send. All recieven clients remove this
                //   operato from their editorNames list.
                var responseCallbackId = Notify.registerCallback(dialogName + '_open_response', function (notify) {
                    if (!notify.sendBySelf) {
                        addActiveEditor(notify.params.name);
                    }
                });
                var queryCallbackId = Notify.registerCallback(dialogName + '_open_query', function (notify) {
                    if (!notify.sendBySelf) {
                        addActiveEditor(notify.params.name);
                        if (notify.senderUserId) {
                            Notify.notify(dialogName + '_open_response', {
                                name: operator.user ? operator.user.short_name : '',
                            }, [notify.senderUserId]);
                        } else {
                            Notify.notify(dialogName + '_open_response', {
                                name: operator.user ? operator.user.short_name : '',
                            }, null, [notify.senderReplyChannelName]);
                        }
                    }
                });
                var closeCallbackId = Notify.registerCallback(dialogName + '_dialog_closed', function (notify) {
                    removeActiveEditor(notify.params.name);
                });
                // Send here the open_query to get the notify-chain started.
                Notify.notify(dialogName + '_open_query', {
                    name: operator.user ? operator.user.short_name : '',
                });
                // The function returned is to deregister the callbacks and send the dialog_closed notify, if
                // the dialog get closed.
                return function () {
                    Notify.deregisterCallbacks(responseCallbackId, queryCallbackId, closeCallbackId);
                    Messaging.deleteMessage(messagingId);
                    Notify.notify(dialogName + '_dialog_closed', {
                        name: operator.user ? operator.user.short_name : '',
                    });
                };
            },
        };
    }
])

/*
 * This filter filters all items in an array. If the filterArray is empty, the
 * array is passed. The filterArray contains numbers of the multiselect, e. g. [1, 3, 4].
 * Then, all items in the array are passed, if the item_id (get with id_function) matches
 * one of the ids in filterArray. id_function could also return a list of ids. Example:
 * Item 1 has two tags with ids [1, 4]. filterArray == [3, 4] --> match
 *
 * If -1 is in the array items without an id will not be filtered. This is for implementing
 * a filter option like: "All items without a category"
 */
.filter('MultiselectFilter', [
    function () {
        return function (array, filterArray, idFunction) {
            if (filterArray.length === 0) {
                return array;
            }
            var itemsWithoutProperty = _.indexOf(filterArray, -1) > -1;
            return Array.prototype.filter.call(array, function (item) {
                var id = idFunction(item);
                if (typeof id === 'number') {
                    id = [id];
                } else if (id === null || !id.length) {
                    return itemsWithoutProperty;
                }
                return _.intersection(id, filterArray).length > 0;
            });
        };
    }
])

.filter('osFilter', [
    function () {
        return function (array, string, getFilterString) {
            if (!string) {
                return array;
            }
            return Array.prototype.filter.call(array, function (item) {
                return getFilterString(item).toLowerCase().indexOf(string.toLowerCase()) > -1;
            });
        };
    }
])

// angular formly config options
.run([
    'formlyConfig',
    function (formlyConfig) {
        // NOTE: This next line is highly recommended. Otherwise Chrome's autocomplete
        // will appear over your options!
        formlyConfig.extras.removeChromeAutoComplete = true;

        // Configure custom types
        formlyConfig.setType({
            name: 'editor',
            extends: 'textarea',
            templateUrl: 'static/templates/core/editor.html',
        });
        formlyConfig.setType({
            name: 'password',
            extends: 'input',
            templateUrl: 'static/templates/core/password.html',
        });
        formlyConfig.setType({
            name: 'checkbox',
            templateUrl: 'static/templates/core/checkbox.html',
            overwriteOk: true,
        });
        formlyConfig.setType({
            name: 'checkbox-buttons',
            templateUrl: 'static/templates/core/checkbox-buttons.html',
            overwriteOk: true,
        });
        formlyConfig.setType({
            name: 'select-single',
            extends: 'select',
            templateUrl: 'static/templates/core/select-single.html'
        });
        formlyConfig.setType({
            name: 'select-multiple',
            extends: 'select',
            templateUrl: 'static/templates/core/select-multiple.html'
        });
        formlyConfig.setType({
            name: 'radio-buttons',
            templateUrl: 'static/templates/core/radio-buttons.html',
            wrapper: ['bootstrapHasError'],
            defaultOptions: {
                noFormControl: false
            }
        });
        formlyConfig.setType({
            name: 'file',
            extends: 'input',
            templateUrl: 'static/templates/core/file.html',
        });
    }
])

// html-tag os-form-field to generate generic from fields
// TODO: make it possible to use other fields then config fields
.directive('osFormField', [
    '$parse',
    'Config',
    'gettextCatalog',
    function($parse, Config, gettextCatalog) {
        var getHtmlType = function (type) {
            return {
                string: 'text',
                text: 'textarea',
                markupText: 'editor',
                integer: 'number',
                boolean: 'checkbox',
                choice: 'choice',
                comments: 'comments',
                colorpicker: 'colorpicker',
                datetimepicker: 'datetimepicker',
                majorityMethod: 'choice',
                translations: 'translations',
            }[type];
        };

        return {
            restrict: 'E',
            scope: true,
            templateUrl: 'static/templates/config-form-field.html',
            link: function ($scope, iElement, iAttrs, controller, transcludeFn) {
                var field = $parse(iAttrs.field)($scope);
                var config = Config.get(field.key);
                $scope.type = getHtmlType(field.input_type);
                if ($scope.type == 'choice') {
                    $scope.choices = field.choices;
                    $scope.value = config.value;
                } else {
                    $scope.value = gettextCatalog.getString(config.value);
                }
                $scope.label = field.label;
                $scope.key = 'field-' + field.key;
                $scope.help_text = field.help_text;
                $scope.default_value = field.default_value;
                $scope.reset = function () {
                    if ($scope.type == 'choice') {
                        $scope.value = $scope.default_value;
                    } else {
                        $scope.value = gettextCatalog.getString($scope.default_value);
                    }
                    $scope.save(field, $scope.value);
                };
            }
        };
    }
])

/* This directive provides a csv import template.
 * Papa Parse is used to parse the csv file. Accepted attributes:
 * * change:
 *   Callback if file changes. The one parameter is csv passing the parsed file
 * * config (optional):
 *   - accept: String with extensions: default '.csv .txt'
 *   - encodingOptions: List with encodings. Default ['UTF-8', 'ISO-8859-1']
 *   - parseConfig: a dict passed to PapaParse
 */
.directive('csvImport', [
    function () {
        return {
            restrict: 'E',
            templateUrl: 'static/templates/csv-import.html',
            scope: {
                change: '&',
                config: '=?',
            },
            controller: function ($scope, $element, $attrs, $location) {
                // set config if it is not given
                if (!$scope.config) {
                    $scope.config = {};
                }
                if (!$scope.config.parseConfig) {
                    $scope.config.parseConfig = {};
                }

                $scope.inputElement = angular.element($element[0].querySelector('#csvFileSelector'));

                // set accept and encoding
                $scope.accept = $scope.config.accept || '.csv';
                $scope.encodingOptions = $scope.config.encodingOptions || ['UTF-8'];
                $scope.encoding = $scope.encodingOptions[0];

                $scope.parse = function () {
                    var inputElement = $scope.inputElement[0];
                    if (!inputElement.files.length) {
                        $scope.change({csv: {data: {}}});
                    } else {
                        var parseConfig = _.defaults(_.clone($scope.config.parseConfig), {
                            delimiter: $scope.delimiter,
                            encoding: $scope.encoding,
                            header: false, // we do not want to have dicts in result
                            complete: function (csv) {
                                if (csv.data.length) {
                                    csv.meta.fields = csv.data[0];
                                }
                                else {
                                    csv.meta.fields = [];
                                }
                                csv.data = csv.data.splice(1); // do not interpret the header as data
                                $scope.$apply(function () {
                                    if (csv.meta.delimiter) {
                                        $scope.autodelimiter = csv.meta.delimiter;
                                    }
                                    $scope.change({csv: csv});
                                });
                            },
                            error: function () {
                                $scope.$apply(function () {
                                    $scope.change({csv: {data: {}}});
                                });
                            },
                        });

                        Papa.parse(inputElement.files[0], parseConfig);
                    }
                };

                $scope.clearFile = function () {
                    $scope.inputElement[0].value = '';
                    $scope.selectedFile = undefined;
                    $scope.parse();
                };

                $scope.inputElement.on('change', function () {
                    $scope.selectedFile = _.last($scope.inputElement[0].value.split('\\'));
                    $scope.parse();
                });
            },
        };
    }
])

.directive('messaging', [
    '$timeout',
    'Messaging',
    function ($timeout, Messaging) {
        return {
            restrict: 'E',
            templateUrl: 'static/templates/messaging.html',
            scope: {},
            controller: function ($scope, $element, $attrs, $location) {
                $scope.messages = {};

                var update = function () {
                    $scope.messages = Messaging.getMessages();
                };
                Messaging.registerMessageChangeCallback(update);

                $scope.close = function (id) {
                    Messaging.deleteMessage(id);
                };
            },
        };
    }
])

.controller('MainMenuCtrl', [
    '$scope',
    'mainMenu',
    function ($scope, mainMenu) {
        mainMenu.registerScope($scope);
        $scope.isMenuOpen = false;
        $scope.closeMenu = function () {
            $scope.isMenuOpen = false;
        };
    }
])

.controller('LanguageCtrl', [
    '$scope',
    'gettextCatalog',
    'Languages',
    'filterFilter',
    function ($scope, gettextCatalog, Languages, filterFilter) {
        $scope.languages = Languages.getLanguages();
        $scope.selectedLanguage = filterFilter($scope.languages, {selected: true});
        // controller to switch app language
        $scope.switchLanguage = function (lang) {
            $scope.languages = Languages.setCurrentLanguage(lang);
            $scope.selectedLanguage = filterFilter($scope.languages, {selected: true});
        };
    }
])

.controller('GotoTopCtrl', [
    '$scope',
    '$window',
    '$timeout',
    function ($scope, $window, $timeout) {
        $scope.show = false;
        angular.element($window).bind('scroll', function () {
            $timeout(function () {
                $scope.show = ($window.pageYOffset >= 150);
            });
        });
    }
])

.run([
    '$rootScope',
    '$window',
    function ($rootScope, $window) {
        $rootScope.gotoTop = function () {
            $window.scrollTo(0, 0);
        };
    }
])

// Prevent scrolling in number inputs. Instead of changing the number, the input
// is blurred and the window is scrolled. This is very important for our dialog
// forms, so a user didn't change a value, when he wants to scroll the form.
.run(function () {
    $('body').on('mousewheel', function (e) {
        if (e.target.nodeName === 'INPUT' && e.target.type === 'number') {
            $(e.target).blur();
        }
    });
})

// Projector Sidebar Controller
.controller('ProjectorSidebarCtrl', [
    '$scope',
    '$document',
    '$window',
    function ($scope, $document, $window) {
        $scope.isProjectorSidebar = false;
        $scope.showProjectorSidebar = function (show) {
            $scope.isProjectorSidebar = show;
        };

        // Sidebar scroll
        var marginTop = 20, // margin-top from #content
            marginBottom = 30, // 30px + 20px sidebar margin-bottom = 50px from footer
            sidebar;

        var sidebarScroll = function () {
            var sidebarHeight = sidebar.height(),
                sidebarOffset = sidebar.offset().top,
                sidebarMinOffset = $('#header').height() + $('#nav').height() + marginTop,
                documentHeight = $document.height(),
                windowHeight = $window.innerHeight,
                scrollTop = $window.pageYOffset;

            // First, check if there is a need to scroll: scroll if the sidebar is smaller then the content
            if (sidebarHeight < $('.col1').height()) {
                if ((scrollTop + marginTop + sidebarHeight) > (documentHeight - marginBottom)) {
                    // Stick to the bottom
                    var bottom = marginBottom + scrollTop + windowHeight - documentHeight;
                    sidebar.css({'position': 'fixed', 'top': '', 'bottom': bottom});
                } else if ((scrollTop + marginTop) > sidebarMinOffset) {
                    // scroll with the user
                    sidebar.css({'position': 'fixed', 'top': marginTop, 'bottom': ''});
                } else {
                    // Stick to the top
                    sidebar.css({'position': 'relative', 'top': 0, 'bottom': ''});
                }
            } else {
                // Stick to the top, if the sidebar is larger then the content
                sidebar.css({'position': 'relative', 'top': 0, 'bottom': ''});
            }
        };

        $scope.initSidebar = function () {
            sidebar = $('#sidebar');
            $scope.$watch(function () {
                return sidebar.height();
            }, sidebarScroll);
            angular.element($window).bind('scroll', sidebarScroll);
        };

    }
])

// Legal Notice Controller
.controller('LegalNoticeCtrl', [
    '$scope',
    '$http',
    function ($scope, $http) {
        $http.get('/core/version/').then(function (success) {
            $scope.core_version = success.data.openslides_version;
            $scope.core_license = success.data.openslides_license;
            $scope.core_url = success.data.openslides_url;
            $scope.plugins = success.data.plugins;
        });
    }
])

// Privacy Policy Controller
.controller('PrivacyPolicyCtrl', function () {})

// Config Controller
.controller('ConfigCtrl', [
    '$scope',
    '$timeout',
    'MajorityMethodChoices',
    'Config',
    'OpenSlidesConfigVariables',
    'gettextCatalog',
    'DateTimePickerTranslation',
    'Editor',
    function($scope, $timeout, MajorityMethodChoices, Config, OpenSlidesConfigVariables,
        gettextCatalog, DateTimePickerTranslation, Editor) {
        Config.bindAll({}, $scope, 'configs');
        $scope.configGroups = OpenSlidesConfigVariables;
        $scope.dateTimePickerTranslatedButtons = DateTimePickerTranslation.getButtons();

        $scope.ckeditorOptions = Editor.getOptions();
        $scope.ckeditorOptions.on.change = function (event) {
            // we could just retrieve the key, but we need the configOption object.
            var configOption_key = event.editor.element.$.id;

            // find configOption object
            var subgroups = _.flatMap($scope.configGroups, function (group) {
                return group.subgroups;
            });
            var items = _.flatMap(subgroups, function (subgroup) {
                return subgroup.items;
            });
            var configOption = _.find(items, function (_item) {
                return _item.key === configOption_key;
            });

            var editor = this;
            // The $timeout executes the given function in an angular context. Because
            // this is a standard JS event, all changes may not happen in the digist-cylce.
            // By using $timeout angular calls $apply for us that we do not have to care
            // about starting the digist-cycle.
            $timeout(function () {
                $scope.save(configOption, editor.getData());
            }, 1);
        };

        // save changed config value
        $scope.save = function(configOption, value) {
            Config.get(configOption.key).value = value;
            Config.save(configOption.key).then(function (success) {
                configOption.success = true;
                // fade out the success symbol after 2 seconds.
                $timeout(function () {
                    var element = $('#success-field-' + configOption.key);
                    element.fadeOut(800, function () {
                        configOption.success = void 0;
                    });
                }, 2000);
            }, function (error) {
                configOption.success = false;
                configOption.errorMessage = error.data.detail;
            });
        };

        // For comments input
        $scope.addComment = function (configOption, parent) {
            var maxId = _.max(_.keys(parent.value));
            if (maxId === undefined) {
                maxId = 1;
            } else {
                maxId = parseInt(maxId) + 1;
            }
            parent.value[maxId] = {
                name: gettextCatalog.getString('New'),
                public: false,
            };
            $scope.save(configOption, parent.value);
        };
        $scope.removeComment = function (configOption, parent, id) {
            parent.value[id] = null;
            $scope.save(configOption, parent.value);
        };

        // For custom translations input
        $scope.addTranslation = function (configOption, parent) {
            parent.value.push({
                original: gettextCatalog.getString('New'),
                translation: gettextCatalog.getString('New'),
            });
            $scope.save(configOption, parent.value);
        };
        $scope.removeTranslation = function (configOption, parent, index) {
            parent.value.splice(index, 1);
            $scope.save(configOption, parent.value);
        };

        // For majority method
        angular.forEach(
            _.filter($scope.configGroups, function (configGroup) {
                return configGroup.name === 'Motions' || configGroup.name === 'Elections';
            }),
            function (configGroup) {
                var configItem;
                _.forEach(configGroup.subgroups, function (subgroup) {
                    configItem = _.find(subgroup.items, ['input_type', 'majorityMethod']);
                    if (configItem !== undefined) {
                        // Break the forEach loop if we found something.
                        return false;
                    }
                });
                if (configItem !== undefined) {
                    configItem.choices = MajorityMethodChoices;
                }
            }
        );
    }
])

// Search Bar Controller
.controller('SearchBarCtrl', [
    '$scope',
    '$state',
    '$sanitize',
    function ($scope, $state, $sanitize) {
        $scope.search = function() {
            var query = _.escape($scope.querybar);
            $scope.querybar = '';
            $state.go('search', {q: query});
        };
    }
])

// Search Controller
.controller('SearchCtrl', [
    '$scope',
    '$filter',
    '$stateParams',
    'Search',
    'DS',
    'Motion',
    function ($scope, $filter, $stateParams, Search, DS, Motion) {
        $scope.searchresults = [];
        var searchModules = Search.getAll();

        // search function
        $scope.search = function() {
            $scope.results = [];
            var foundObjects = [];
            // search in rest properties of all defined searchModule
            // (does not found any related objects, e.g. speakers of items)
            _.forEach(searchModules, function(searchModule) {
                var result = {};
                result.verboseName = searchModule.verboseName;
                result.collectionName = searchModule.collectionName;
                result.urlDetailState = searchModule.urlDetailState;
                result.weight = searchModule.weight;
                result.checked = true;
                result.elements = $filter('filter')(DS.getAll(searchModule.collectionName), $scope.searchquery);
                $scope.results.push(result);
                _.forEach(result.elements, function(element) {
                    foundObjects.push(element);
                });
            });
            // search additionally in specific releations of all defined searchModules
            _.forEach(searchModules, function(searchModule) {
                _.forEach(DS.getAll(searchModule.collectionName), function(object) {
                    if (_.isFunction(object.hasSearchResult)) {
                        if (object.hasSearchResult(foundObjects, $scope.searchquery)) {
                            // releation found, check if object is not yet in search results
                            _.forEach($scope.results, function(result) {
                                if ((object.getResourceName() === result.collectionName) &&
                                        _.findIndex(result.elements, {'id': object.id}) === -1) {
                                    result.elements.push(object);
                                }
                            });
                        }
                    } else {
                        return false;
                    }
                });
            });
        };

        //get search string from parameters submitted from outside the scope
        if ($stateParams.q) {
            $scope.searchquery = $stateParams.q;
            $scope.search();
        }
    }
])

// Projector Control Controller
.controller('ProjectorControlCtrl', [
    '$scope',
    '$http',
    '$interval',
    '$state',
    '$q',
    '$filter',
    'Config',
    'Projector',
    'CurrentListOfSpeakersItem',
    'CurrentListOfSpeakersSlide',
    'ProjectionDefault',
    'ProjectorMessage',
    'Countdown',
    'gettextCatalog',
    'ngDialog',
    'ProjectorMessageForm',
    function($scope, $http, $interval, $state, $q, $filter, Config, Projector, CurrentListOfSpeakersItem,
        CurrentListOfSpeakersSlide, ProjectionDefault, ProjectorMessage, Countdown, gettextCatalog,
        ngDialog, ProjectorMessageForm) {
        ProjectorMessage.bindAll({}, $scope, 'messages');

        var intervals = [];
        var calculateCountdownTime = function (countdown) {
            countdown.seconds = Math.floor( countdown.countdown_time - Date.now() / 1000 + $scope.serverOffset );
        };
        var cancelIntervalTimers = function () {
            intervals.forEach(function (interval) {
                $interval.cancel(interval);
            });
        };
        $scope.$watch(function () {
            return Countdown.lastModified();
        }, function () {
            $scope.countdowns = Countdown.getAll();

            // stop ALL interval timer
            cancelIntervalTimers();
            $scope.countdowns.forEach(function (countdown) {
                if (countdown.running) {
                    calculateCountdownTime(countdown);
                    intervals.push($interval(function () { calculateCountdownTime(countdown); }, 1000));
                } else {
                    countdown.seconds = countdown.countdown_time;
                }
            });
        });
        $scope.$on('$destroy', function() {
            // Cancel all intervals if the controller is destroyed
            cancelIntervalTimers();
        });

        $scope.$watch(function () {
            return Projector.lastModified();
        }, function () {
            $scope.projectors = Projector.getAll();
            if (!$scope.active_projector) {
                $scope.active_projector = $filter('orderBy')($scope.projectors, 'id')[0];
            }
            $scope.setIframeSize($scope.active_projector);
            if ($scope.projectors.length === 1) {
                $scope.currentListOfSpeakersAsOverlay = true;
            }

            $scope.messageDefaultProjectorId = ProjectionDefault.filter({name: 'messages'})[0].projector_id;
            $scope.countdownDefaultProjectorId = ProjectionDefault.filter({name: 'countdowns'})[0].projector_id;
            $scope.listOfSpeakersDefaultProjectorId = ProjectionDefault.filter({name: 'agenda_current_list_of_speakers'})[0].projector_id;
        });
        // watch for changes in projector_broadcast and currentListOfSpeakersReference
        var last_broadcast;
        $scope.$watch(function () {
            return Config.lastModified();
        }, function () {
            var broadcast = Config.get('projector_broadcast').value;
            if (!last_broadcast || last_broadcast != broadcast) {
                last_broadcast = broadcast;
                $scope.broadcast = broadcast;
            }
            $scope.currentListOfSpeakersReference = $scope.config('projector_currentListOfSpeakers_reference');
        });

        $scope.changeProjector = function (projector) {
            $scope.active_projector = projector;
            $scope.setIframeSize(projector);
        };
        $scope.setIframeSize = function (projector) {
            $scope.scale = 256.0 / projector.width;
            $scope.iframeHeight = $scope.scale * projector.height;
        };

        $scope.editCurrentSlide = function (projector) {
            var data = projector.getFormOrStateForCurrentSlide();
            if (data) {
                if (data.form) {
                    ngDialog.open(data.form.getDialog({id: data.id}));
                } else {
                    $state.go(data.state, {id: data.id});
                }
            }
        };

        // *** countdown functions ***
        $scope.calculateCountdownTime = function (countdown) {
            countdown.seconds = Math.floor( countdown.countdown_time - Date.now() / 1000 + $scope.serverOffset );
        };
        $scope.editCountdown = function (countdown) {
            countdown.editFlag = false;
            countdown.description = countdown.new_description;
            Countdown.save(countdown);
            if (!countdown.running) {
                countdown.reset();
            }
        };
        $scope.addCountdown = function () {
            var default_time = parseInt($scope.config('projector_default_countdown'));
            var countdown = {
                description: '',
                default_time: default_time,
                countdown_time: default_time,
                running: false,
            };
            Countdown.create(countdown);
        };
        $scope.removeCountdown = function (countdown) {
            Countdown.destroy(countdown.id);
        };

        // *** message functions ***
        $scope.editMessage = function (message) {
            ngDialog.open(ProjectorMessageForm.getDialog(message));
        };
        $scope.addMessage = function () {
            var message = {message: ''};
            ProjectorMessage.create(message);
        };
        $scope.removeMessage = function (message) {
            ProjectorMessage.destroy(message.id);
        };

        /* Current list of speakers */
        $scope.currentListOfSpeakers = CurrentListOfSpeakersSlide;
        // Set the current overlay status
        if ($scope.currentListOfSpeakers.isProjected().length) {
            var isProjected = $scope.currentListOfSpeakers.isProjectedWithOverlayStatus();
            $scope.currentListOfSpeakersAsOverlay = isProjected[0].overlay;
        } else {
            $scope.currentListOfSpeakersAsOverlay = true;
        }
        // go to the list of speakers(management) of the currently displayed list of speakers reference slide
        $scope.goToListOfSpeakers = function() {
            var item = $scope.currentListOfSpeakersItem();
            if (item) {
                $state.go('agenda.item.detail', {id: item.id});
            }
        };
        $scope.currentListOfSpeakersItem = function () {
            return CurrentListOfSpeakersItem.getItem($scope.currentListOfSpeakersReference);
        };
        $scope.setOverlay = function (overlay) {
            $scope.currentListOfSpeakersAsOverlay = overlay;
            var isProjected = $scope.currentListOfSpeakers.isProjectedWithOverlayStatus();
            if (isProjected.length) {
                _.forEach(isProjected, function (mapping) {
                    if (mapping.overlay != overlay) { // change the overlay if it is different
                        $scope.currentListOfSpeakers.project(mapping.projectorId, overlay);
                    }
                });
            }
        };
    }
])

.controller('ProjectorMessageEditCtrl', [
    '$scope',
    'projectorMessageId',
    'ProjectorMessage',
    'ProjectorMessageForm',
    function ($scope, projectorMessageId, ProjectorMessage, ProjectorMessageForm) {
        $scope.formFields = ProjectorMessageForm.getFormFields();
        $scope.model = angular.copy(ProjectorMessage.get(projectorMessageId));

        $scope.save = function (message) {
            ProjectorMessage.inject(message);
            ProjectorMessage.save(message);
            $scope.closeThisDialog();
        };
    }
])

.controller('ManageProjectorsCtrl', [
    '$scope',
    '$http',
    '$timeout',
    'Projector',
    'ProjectionDefault',
    'Config',
    'ProjectorMessage',
    'ngDialog',
    function ($scope, $http, $timeout, Projector, ProjectionDefault, Config,
        ProjectorMessage, ngDialog) {
        ProjectionDefault.bindAll({}, $scope, 'projectiondefaults');

        /* Info on resolution calculating:
         * Internally the resolution is saved as (width, height) but the user has
         * an aspect ratio to choose and a width from 800 to 3840 (4K).*/
        $scope.aspectRatios = {
            '4:3': 4/3,
            '16:9': 16/9,
            '16:10': 16/10,
        };
        // when converting (x,y) -> (ratio, percentage) round issues may occur
        // (e.g. 800/600 != 4/3 with internal calculation issues). With this environment
        // is tested, if the calculated value is in the following interval:
        // [expected-environment; expected+environment]
        var RATIO_ENVIRONMENT = 0.05;

        // watch for changes in projector_broadcast
        // and projector_currentListOfSpeakers_reference
        var last_broadcast, last_clos;
        $scope.$watch(function () {
            return Config.lastModified();
        }, function () {
            var broadcast = $scope.config('projector_broadcast'),
            currentListOfSpeakers = $scope.config('projector_currentListOfSpeakers_reference');
            if (!last_broadcast || last_broadcast != broadcast) {
                last_broadcast = broadcast;
                $scope.broadcast = broadcast;
            }
            if (!last_clos || last_clos != currentListOfSpeakers) {
                last_clos = currentListOfSpeakers;
                $scope.currentListOfSpeakers = currentListOfSpeakers;
            }
        });

        // watch for changes in Projector, and recalc scale and iframeHeight
        var first_watch = true;
        $scope.resolutions = {};
        $scope.edit = [];
        $scope.sliders = {};
        $scope.$watch(function () {
            return Projector.lastModified();
        }, function () {
            $scope.projectors = Projector.getAll();
            $scope.projectors.forEach(function (projector) {
                projector.iframeScale = 256.0 / projector.width;
                projector.iframeHeight = projector.iframeScale * projector.height;
                if (first_watch) {
                    $scope.resolutions[projector.id] = {
                        width: projector.width,
                        height: projector.height
                    };
                    $scope.edit[projector.id] = false;
                    $scope.sliders[projector.id] = {
                        value: projector.width,
                        options: {
                            id: projector.id,
                            floor: 800,
                            ceil: 3840,
                            step: 10,
                            translate: function (value) {
                                return value + 'px';
                            },
                            onChange: function (v) {
                                $scope.calcResolution(projector);
                            },
                            onEnd: function (v) {
                                $scope.saveResolution(projector);
                            },
                            hideLimitLabels: true,
                        },
                    };
                    $scope.setAspectRatio(projector, $scope.getAspectRatio(projector));
                }
            });
            if ($scope.projectors.length) {
                first_watch = false;
            }
        });

        $scope.getAspectRatio = function (projector) {
            var ratio = projector.width/projector.height;
            var foundRatio = _.findKey($scope.aspectRatios, function (value) {
                return value >= (ratio-RATIO_ENVIRONMENT) && value <= (ratio+RATIO_ENVIRONMENT);
            });
            if (foundRatio === undefined) {
                return _.keys($scope.aspectRatios)[0];
            } else {
                return foundRatio;
            }
        };
        $scope.setAspectRatio = function (projector, aspectRatio) {
            $scope.resolutions[projector.id].aspectRatio = aspectRatio;
            $scope.resolutions[projector.id].aspectRatioNumber = $scope.aspectRatios[aspectRatio];
            $scope.calcResolution(projector);
        };
        $scope.calcResolution = function (projector) {
            var ratio = $scope.resolutions[projector.id].aspectRatioNumber;
            var width = $scope.sliders[projector.id].value;
            $scope.resolutions[projector.id].width = width;
            $scope.resolutions[projector.id].height = Math.round(width/ratio);
        };

        $scope.toggleEditMenu = function (projectorId) {
            $scope.edit[projectorId] = !$scope.edit[projectorId];
            $timeout(function () {
                $scope.$broadcast('rzSliderForceRender');
            });
        };

        // Set list of speakers reference
        $scope.setListOfSpeakers = function (projector) {
            Config.get('projector_currentListOfSpeakers_reference').value = projector.id;
            Config.save('projector_currentListOfSpeakers_reference');
        };

        // Projector functions
        $scope.setProjectionDefault = function (projector, projectiondefault) {
            if (projectiondefault.projector_id !== projector.id) {
                $http.post('/rest/core/projector/' + projector.id + '/set_projectiondefault/', projectiondefault.id);
            }
        };
        $scope.createProjector = function (name) {
            var projector = {
                name: name,
                config: {},
                scale: 0,
                scroll: 0,
                blank: false,
                projectiondefaults: [],
            };
            Projector.create(projector).then(function (projector) {
                $http.post('/rest/core/projector/' + projector.id + '/activate_elements/', [{
                    name: 'core/clock',
                    stable: true
                }]);
                $scope.resolutions[projector.id] = {
                    width: projector.width,
                    height: projector.height
                };
            });
        };
        $scope.deleteProjector = function (projector) {
            if (projector.id != 1) {
                Projector.destroy(projector.id);
            }
        };
        $scope.editCurrentSlide = function (projector) {
            var data = projector.getFormOrStateForCurrentSlide();
            if (data) {
                if (data.form) {
                    ngDialog.open(data.form.getDialog({id: data.id}));
                } else {
                    $state.go(data.state, {id: data.id});
                }
            }
        };
        $scope.editName = function (projector) {
            projector.config = projector.elements;
            Projector.save(projector);
        };
        $scope.saveResolution = function (projector) {
            $http.post(
                '/rest/core/projector/' + projector.id + '/set_resolution/',
                $scope.resolutions[projector.id]
            ).then(function (success) {
                $scope.resolutions[projector.id].error = null;
            }, function (error) {
                if (error.data) {
                    $scope.resolutions[projector.id].error = error.data.detail;
                } else {
                    $scope.resolutions[projector.id].error = null;
                }
            });
        };

        // Identify projectors
        $scope.identifyProjectors = function () {
            if ($scope.identifyPromise) {
                $timeout.cancel($scope.identifyPromise);
                $scope.removeIdentifierMessages();
            } else {
                // Create new Message
                var message = {
                    message: '',
                };
                ProjectorMessage.create(message).then(function(message){
                    $scope.projectors.forEach(function (projector) {
                        $http.post('/rest/core/projector/' + projector.id + '/activate_elements/', [{
                            name: 'core/projector-message',
                            stable: true,
                            id: message.id,
                            identify: true,
                        }]);
                    });
                    $scope.identifierMessage = message;
                });
                $scope.identifyPromise = $timeout($scope.removeIdentifierMessages, 3000);
            }
        };
        $scope.removeIdentifierMessages = function () {
            Projector.getAll().forEach(function (projector) {
                _.forEach(projector.elements, function (element, uuid) {
                    if (element.name === 'core/projector-message' && element.id === $scope.identifierMessage.id) {
                        $http.post('/rest/core/projector/' + projector.id + '/deactivate_elements/', [uuid]);
                    }
                });
            });
            ProjectorMessage.destroy($scope.identifierMessage.id);
            $scope.identifyPromise = null;
        };
    }
])

// Tag Controller
.controller('TagListCtrl', [
    '$scope',
    'Tag',
    'ngDialog',
    'TagForm',
    'gettext',
    'ErrorMessage',
    function($scope, Tag, ngDialog, TagForm, gettext, ErrorMessage) {
        Tag.bindAll({}, $scope, 'tags');
        $scope.alert = {};

        // setup table sorting
        $scope.sortColumn = 'name';
        $scope.reverse = false;
        // function to sort by clicked column
        $scope.toggleSort = function ( column ) {
            if ( $scope.sortColumn === column ) {
                $scope.reverse = !$scope.reverse;
            }
            $scope.sortColumn = column;
        };
        $scope.delete = function (tag) {
            Tag.destroy(tag.id).then(
                function(success) {
                    $scope.alert = {
                        type: 'success',
                        msg: gettext('The delete was successful.'),
                        show: true,
                    };
                }, function (error) {
                    $scope.alert = ErrorMessage.forAlert(error);
                }
            );
        };
        $scope.editOrCreate = function (tag) {
            ngDialog.open(TagForm.getDialog(tag));
        };
    }
])

.controller('TagCreateCtrl', [
    '$scope',
    'Tag',
    'TagForm',
    'ErrorMessage',
    function($scope, Tag, TagForm, ErrorMessage) {
        $scope.model = {};
        $scope.alert = {};
        $scope.formFields = TagForm.getFormFields();
        $scope.save = function (tag) {
            Tag.create(tag).then(
                function (success) {
                    $scope.closeThisDialog();
                },
                function (error) {
                    $scope.alert = ErrorMessage.forAlert(error);
                }
            );
        };
    }
])

.controller('TagUpdateCtrl', [
    '$scope',
    'Tag',
    'tagId',
    'TagForm',
    'ErrorMessage',
    function($scope, Tag, tagId, TagForm, ErrorMessage) {
        $scope.model = angular.copy(Tag.get(tagId));
        $scope.alert = {};
        $scope.formFields = TagForm.getFormFields();
        $scope.save = function (tag) {
            Tag.inject(tag);
            Tag.save(tag).then(function(success) {
                $scope.closeThisDialog();
            }, function (error) {
                // save error: revert all changes by restore
                // the original object
                Tag.refresh(tag);
                $scope.alert = ErrorMessage.forAlert(error);
            });
        };
    }
])

.controller('CountdownDetailCtrl', [
    '$scope',
    '$interval',
    'Countdown',
    'countdownId',
    function ($scope, $interval, Countdown, countdownId) {
        var interval;
        var calculateCountdownTime = function (countdown) {
            countdown.seconds = Math.floor( $scope.countdown.countdown_time - Date.now() / 1000 + $scope.serverOffset );
        };
        $scope.$watch(function () {
            return Countdown.lastModified(countdownId);
        }, function () {
            $scope.countdown = Countdown.get(countdownId);
            if (interval) {
                $interval.cancel(interval);
            }
            if ($scope.countdown) {
                if ($scope.countdown.running) {
                    calculateCountdownTime($scope.countdown);
                    interval = $interval(function () { calculateCountdownTime($scope.countdown); }, 1000);
                } else {
                    $scope.countdown.seconds = $scope.countdown.countdown_time;
                }
            }
        });
        $scope.$on('$destroy', function() {
            // Cancel the interval if the controller is destroyed
            if (interval) {
                $interval.cancel(interval);
            }
        });
    }
])

// counter of new (unread) chat messages
.value('NewChatMessages', [])

// ChatMessage Controller
.controller('ChatMessageCtrl', [
    '$scope',
    '$http',
    '$timeout',
    'ChatMessage',
    'NewChatMessages',
    function ($scope, $http, $timeout, ChatMessage, NewChatMessages) {
        ChatMessage.bindAll({}, $scope, 'chatmessages');
        $scope.unreadMessages = NewChatMessages.length;
        $scope.chatboxIsCollapsed = true;
        $scope.openChatbox = function () {
            $scope.chatboxIsCollapsed = !$scope.chatboxIsCollapsed;
            NewChatMessages = [];
            $scope.unreadMessages = NewChatMessages.length;
            $timeout(function () {
                angular.element('#messageInput').focus();
            }, 0);
        };
        $scope.sendMessage = function () {
            angular.element('#messageSendButton').addClass('disabled');
            angular.element('#messageInput').attr('disabled', '');
            $http.post(
                '/rest/core/chat-message/',
                {message: $scope.newMessage}
            ).then(function (success) {
                $scope.newMessage = '';
                angular.element('#messageSendButton').removeClass('disabled');
                angular.element('#messageInput').removeAttr('disabled');
                $timeout(function () {
                    angular.element('#messageInput').focus();
                }, 0);
            }, function (error) {
                angular.element('#messageSendButton').removeClass('disabled');
                angular.element('#messageInput').removeAttr('disabled');
            });
        };
        // increment unread messages counter for each new message
        $scope.$watch('chatmessages', function (newVal, oldVal) {
            // add new message id if there is really a new message which is not yet tracked
            if (oldVal.length > 0 && newVal.length > 0) {
                if ((oldVal[oldVal.length-1].id != newVal[newVal.length-1].id) &&
                    ($.inArray(newVal[newVal.length-1].id, NewChatMessages) == -1)) {
                    NewChatMessages.push(newVal[newVal.length-1].id);
                    $scope.unreadMessages = NewChatMessages.length;
                }
            } else if (newVal.length === 0) {
                NewChatMessages = [];
                $scope.unreadMessages = 0;
            }
        });

        $scope.clearChatHistory = function () {
            $http.post('/rest/core/chat-message/clear/');
        };
    }
])

// format time string for model ("s") and view format ("h:mm:ss" or "mm:ss")
.directive('minSecFormat', [
    'HumanTimeConverter',
    function (HumanTimeConverter) {
        return {
            require: 'ngModel',
            link: function(scope, element, attrs, ngModelController) {
                ngModelController.$parsers.push(function(data) {
                    //convert data from view format (mm:ss) to model format (s)
                    return HumanTimeConverter.humanTimeToSeconds(data, {seconds: true});
                });

                ngModelController.$formatters.push(function(data) {
                    //convert data from model format (s) to view format (mm:ss)
                    return HumanTimeConverter.secondsToHumanTime(data);
                });
            }
        };
    }
])

// format time string for model ("m") and view format ("h:mm" or "hh:mm")
.directive('hourMinFormat', [
    'HumanTimeConverter',
    function (HumanTimeConverter) {
        return {
            require: 'ngModel',
            link: function(scope, element, attrs, ngModelController) {
                ngModelController.$parsers.push(function(data) {
                    //convert data from view format (hh:mm) to model format (m)
                    return HumanTimeConverter.humanTimeToSeconds(data, {hours: true})/60;
                });

                ngModelController.$formatters.push(function(data) {
                    //convert data from model format (m) to view format (hh:mm)
                    return HumanTimeConverter.secondsToHumanTime(data*60,
                        { seconds: 'disabled',
                            hours: 'enabled' }
                    );
                });
            }
        };
    }
])

.directive('osFocusMe', [
    '$timeout',
    function ($timeout) {
        return {
            link: function (scope, element, attrs, model) {
                $timeout(function () {
                    element[0].focus();
                });
            }
        };
    }
])

.filter('toArray', function(){
    /*
     * Transforms an object to an array. Items of the array are the values of
     * the object elements.
     */
    return function(obj) {
        var result = [];
        angular.forEach(obj, function(val, key) {
            result.push(val);
        });
        return result;
    };
})

//Mark all core config strings for translation in Javascript
.config([
    'gettext',
    function (gettext) {
        gettext('Presentation and assembly system');
        gettext('Event name');
        gettext('<a href="http://www.openslides.org">OpenSlides</a> is a free ' +
                'web based presentation and assembly system for visualizing ' +
                'and controlling agenda, motions and elections of an ' +
                'assembly.');
        gettext('General');
        gettext('Event');
        gettext('Short description of event');
        gettext('Event date');
        gettext('Event location');
        gettext('Event organizer');
        gettext('Legal notice');
        gettext('Privacy policy');
        gettext('Front page title');
        gettext('Welcome to OpenSlides');
        gettext('Front page text');
        gettext('[Space for your welcome text.]');
        gettext('Allow access for anonymous guest users');
        gettext('Show this text on the login page');
        gettext('Separator used for all csv exports and examples');
        gettext('Page number alignment in PDF');
        gettext('Left');
        gettext('Center');
        gettext('Right');
        gettext('Standard font size in PDF');
        gettext('Show logo on projector');
        gettext('You can replace the logo by uploading an image and set it as ' +
                'the "Projector logo" in "files".');
        gettext('Projector');
        gettext('Projector language');
        gettext('Current browser language');
        gettext('Show title and description of event on projector');
        gettext('Display header and footer');
        gettext('Show the clock on projector');
        gettext('Background color of projector header and footer');
        gettext('Font color of projector header and footer');
        gettext('Font color of projector headline');
        gettext('Predefined seconds of new countdowns');
        gettext('Color for blanked projector');
        gettext('List of speakers overlay');
        gettext('Projector logo');
        gettext('Projector header image');
        gettext('PDF header logo');
        gettext('PDF footer logo');
        gettext('Web interface header logo');
        gettext('PDF ballot paper logo');
        gettext('Custom translations');

        // Mark the string 'Default projector' here, because it does not appear in the templates.
        gettext('Default projector');
    }
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.core.start', [])

.factory('OpenSlides', [
    '$http',
    '$rootScope',
    '$state',
    '$q',
    'DS',
    'autoupdate',
    'operator',
    'Group',
    'mainMenu',
    'ngDialog',
    'LoginDialog',
    function($http, $rootScope, $state, $q, DS, autoupdate, operator, Group, mainMenu, ngDialog, LoginDialog) {
        var OpenSlides = {
            bootup: function () {
                $rootScope.openslidesBootstrapDone = false;
                $http.get('/users/whoami/').then(function (success) {
                    $rootScope.guest_enabled = success.data.guest_enabled;
                    if (success.data.user_id === null && !success.data.guest_enabled) {
                        // Redirect to login dialog if user is not logged in.
                        $state.go('login', {guest_enabled: success.data.guest_enabled});
                    } else {
                        autoupdate.newConnect();
                        autoupdate.firstMessageDeferred.promise.then(function () {
                            operator.setUser(success.data.user_id, success.data.user);
                            $rootScope.operator = operator;
                            mainMenu.updateMainMenu();
                            $rootScope.openslidesBootstrapDone = true;
                        });
                    }
                });
            },
            shutdown: function () {
                // Close connection, clear the store and show the OS overlay.
                autoupdate.closeConnection();
                DS.clear();
                operator.setUser(null);
                $rootScope.openslidesBootstrapDone = false;
                $rootScope.operator = operator;
                // close all open dialogs (except the login dialog)
                _.forEach(ngDialog.getOpenDialogs(), function (id) {
                    if (id !== LoginDialog.id) {
                        ngDialog.close(id);
                    }
                });
            },
            reboot: function () {
                this.shutdown();
                this.bootup();
            },
        };

        // We need to 'ping' the server with a get request to whoami, because then we can decide,
        // if the server is down or respond with a 403 (this cannot be differentiated with websockets)
        autoupdate.registerRetryConnectCallback(function () {
            return $http.get('/users/whoami').then(function (success) {
                if (success.data.user_id === null && !success.data.guest_enabled) {
                    OpenSlides.shutdown();
                    // Redirect to login dialog if user is not logged in.
                    $state.go('login', {guest_enabled: success.data.guest_enabled});
                } else {
                    autoupdate.newConnect();
                }
            });
        });

        return OpenSlides;
    }
])

.run([
    'OpenSlides',
    function (OpenSlides) {
        OpenSlides.bootup();
    }
])

.run([
    '$rootScope',
    '$state',
    'operator',
    'User',
    'Group',
    'mainMenu',
    function ($rootScope, $state, operator, User, Group, mainMenu) {
        var permissionChangeCallback = function () {
            operator.reloadPerms();
            mainMenu.updateMainMenu();
            var stateData = $state.current.data;
            var basePerm = stateData ? stateData.basePerm : '';
            $rootScope.baseViewPermissionsGranted = basePerm ?
                operator.hasPerms(basePerm) : true;
        };

        $rootScope.$watch(function () {
            return Group.lastModified();
        }, function () {
            if (Group.getAll().length) {
                permissionChangeCallback();
            }
        });

        $rootScope.$watch(function () {
            return operator.user ? User.lastModified(operator.user.id) : true;
        }, function () {
            permissionChangeCallback();
        });
    }
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.mediafiles.forms', [
    'gettext',
    'ngFileUpload',
    'ui.router',
    //TODO: Add deps for operator, User
])

// Service for mediafile form
.factory('MediafileForm', [
    'gettextCatalog',
    'operator',
    'User',
    function (gettextCatalog, operator, User) {
        return {
            // ngDialog for mediafile form
            getDialog: function (mediafile) {
                var dialog = {
                    className: 'ngdialog-theme-default wide-form',
                    closeByEscape: false,
                    closeByDocument: false,
                };
                if (mediafile) {
                    dialog.template = 'static/templates/mediafiles/mediafile-form.html';
                    dialog.controller = 'MediafileUpdateCtrl';
                    dialog.resolve = {
                        mediafileId: function () {return mediafile ? mediafile.id : void 0;}
                    };
                } else {
                    dialog.template = 'static/templates/mediafiles/mediafile-upload-form.html';
                    dialog.controller = 'MediafileUploadCtrl';
                }
                return dialog;
            },
            getFormFields: function () {
                return [
                    {
                        key: 'title',
                        type: 'input',
                        templateOptions: {
                            label: gettextCatalog.getString('Title'),
                        },
                    },
                    {
                        key: 'hidden',
                        type: 'checkbox',
                        templateOptions: {
                            label: gettextCatalog.getString('Hidden'),
                        },
                        hide: !operator.hasPerms('mediafiles.can_see_hidden'),
                    },
                    {
                        key: 'uploader_id',
                        type: 'select-single',
                        templateOptions: {
                            label: gettextCatalog.getString('Uploaded by'),
                            options: User.getAll(),
                            ngOptions: 'option.id as option.full_name for option in to.options',
                            placeholder: gettextCatalog.getString('Select or search a participant ...')
                        },
                        hide: !operator.hasPerms('mediafiles.can_manage')
                    },
                ];

            }
        };
    }
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.mediafiles.image-plugin', [
    'OpenSlidesApp.mediafiles.resources',
    'gettext',
    'OpenSlidesApp.core',
])

.factory('ImageBrowserPlugin', [
    '$templateCache',
    'Mediafile',
    'gettextCatalog',
    'Editor',
    function ($templateCache, Mediafile, gettextCatalog, Editor) {
        return {
            getPlugin: function () {
                return {
                    init: function (editor) {
                        CKEDITOR.tools.imagebrowser = {};

                        // Initialize this dialog, if it is opened.
                        editor.on('dialogShow', function (event) {
                            var dialog = event.data;
                            if (dialog.getName() === 'imagebrowser-dialog') {
                                CKEDITOR.dialog.getCurrent().disableButton('ok');

                                // Load the main plugin template and paste it into the container
                                var template = $templateCache.get('static/templates/mediafiles/image-plugin.html');
                                if (!template) {
                                    throw 'Template for image plugin not found!';
                                }
                                $('#imageBrowserContainer').html(template);

                                // Load all images.
                                var images = '';
                                _.forEach(Mediafile.getAllImages(), function (image) {
                                        images += '<div class="image" onclick="CKEDITOR.tools.imagebrowser.selectImage(\'' +
                                            image.value + '\');" style="background-image:url(\'' +
                                            image.value + '\');" data-image="' + image.value + '"></div>';
                                });
                                $('#imageBrowser').html(images);

                                // Translate some strings. Angular tags are not available in CKEditor.
                                $('#scaleLabel').html(gettextCatalog.getString('Scale'));

                                // If the dialog was opened via double click, check the selected element. It
                                // may be an image, so preselect it.
                                var selectedElement = editor.getSelection().getStartElement();
                                if (selectedElement.is('img')) {
                                    // Check for given scale of this image.
                                    var styleAttr = $(selectedElement).attr('style');
                                    var scale;
                                    var scaleRegex = /width\s*:\s*(\d+)\s*%/g;
                                    var scaleMatch = scaleRegex.exec(styleAttr);
                                    if (scaleMatch) {
                                        scale = parseInt(scaleMatch[1]);
                                    }
                                    CKEDITOR.tools.imagebrowser.selectImage(
                                        selectedElement.getAttribute('src'), scale);
                                }
                                // Setup event listeners.
                                $('#image-scale').bind('keyup mouseup', function (event) {
                                    var scale = parseInt($('#image-scale').val());
                                    if (scale !== CKEDITOR.tools.imagebrowser.scale) {
                                        CKEDITOR.tools.imagebrowser.updateImageSize(scale);
                                    }
                                });
                            }
                        });
                        // React on double clicks in the textarea. If an image was selected, open this dialog.
                        editor.on('doubleclick', function (event) {
                            var element = event.data.element;
                            if (!element.isReadOnly()) {
                                if (element.is('img')) {
                                    event.data.dialog = 'imagebrowser-dialog';
                                    editor.getSelection().selectElement(element);
                                }
                            }
                        });
                        // Handler for selecting an image. It may be called by clicking on a thumbnail or by
                        // just giving the url. The scale is optional.
                        CKEDITOR.tools.imagebrowser.selectImage = function (url, scale) {
                            var browser = $('#imageBrowser');
                            _.forEach(browser.children(), function (child) { // check every available image
                                if (child.getAttribute('data-image') == url) { //match
                                    child.classList.add('selected');
                                    var image = $('#imagePreview img');
                                    // Setup an load event handler, so we can get the size of the image when loaded.
                                    image.on('load', function (event) {
                                        var w = event.target.naturalWidth;
                                        var h = event.target.naturalHeight;
                                        $('#originalSizeText').html(gettextCatalog.getString('Original size') +
                                            ': ' + w + ' &times; ' + h );
                                        $('#fullSizeContainer').width(w).height(h);
                                        if (scale !== undefined) {
                                            // Use custom scale.
                                            CKEDITOR.tools.imagebrowser.updateImageSize(scale);
                                        } else {
                                            CKEDITOR.tools.imagebrowser.updateImageSize(100);
                                        }
                                    });
                                    // Set the url of the main preview image.
                                    image.attr('src', url);
                                    $('#imagePreviewSection').removeClass('hidden');
                                    CKEDITOR.tools.imagebrowser.selected = url;
                                } else {
                                    // Wrong image, deselect it in the preview window.
                                    child.classList.remove('selected');
                                }
                            });
                        };
                        // Handler for updateing the image size.
                        CKEDITOR.tools.imagebrowser.updateImageSize = function (scale) {
                            if (isNaN(scale) || scale <= 0) {
                                CKEDITOR.dialog.getCurrent().disableButton('ok');
                            } else {
                                CKEDITOR.dialog.getCurrent().enableButton('ok');
                                CKEDITOR.tools.imagebrowser.scale = scale;
                                $('#imagePreview img').width(scale + '%');
                                $('#image-scale').val(scale);
                            }
                        };
                        // Insert the selected image into the textarea.
                        CKEDITOR.tools.imagebrowser.insertImage = function (url, scale) {
                            var editor = CKEDITOR.currentInstance;
                            var dialog = CKEDITOR.dialog.getCurrent();
                            var html = '<img src="' + url + '" data-cke-saved-src="' + url +
                                '" alt="' + url + '" style="width: ' + scale + '%;" />';
                            editor.config.allowedContent = true;
                            editor.insertHtml(html.trim());
                            dialog.hide();
                        };
                        editor.addCommand('imagebrowser-open', new CKEDITOR.dialogCommand('imagebrowser-dialog'));
                        // By naming the button 'image', it gets the same image as the original image button.
                        editor.ui.addButton('image', {
                            label: gettextCatalog.getString('Open image browser'),
                            command: 'imagebrowser-open',
                            toolbar: 'insert',
                        });
                    },
                };
            },
            getDialog: function () {
                return function (editor) {
                    return {
                        title: gettextCatalog.getString('Image browser'),
                        minWidth: 1000,
                        minHeight: 500,
                        contents: [
                            {
                                id: 'imagebrowser-tab1',
                                label: gettextCatalog.getString('Browse for images'),
                                elements: [
                                    {
                                        type: 'html',
                                        align: 'left',
                                        id: 'titleid',
                                        style: 'font-size: 20px; font-weight: bold;',
                                        html: gettextCatalog.getString('Browse for images'),
                                    }, {
                                        type: 'html',
                                        align: 'left',
                                        id: 'msg',
                                        style: '',
                                        html: '<div id="imageBrowserContainer"></div>'
                                    }
                                ],
                            },
                        ],
                        // insert image on OK.
                        onOk: function (event) {
                            var url = CKEDITOR.tools.imagebrowser.selected;
                            if (url) {
                                var scale = CKEDITOR.tools.imagebrowser.scale;
                                CKEDITOR.tools.imagebrowser.insertImage(url, scale);
                            }
                        },
                    };
                };
            },
        };
    }
])

.run([
    'Editor',
    'ImageBrowserPlugin',
    'gettext',
    function (Editor, ImageBrowserPlugin, gettext) {
        Editor.registerDialog('imagebrowser-dialog', ImageBrowserPlugin.getDialog());
        Editor.registerPlugin('imagebrowser', ImageBrowserPlugin.getPlugin());

        // mark all plugin strings
        gettext('Original size');
        gettext('Scale');
        gettext('Image browser');
        gettext('Browse for images');
    }
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.mediafiles.list', [
    'gettext',
    'ngDialog',
    'OpenSlidesApp.mediafiles.forms',
    'OpenSlidesApp.mediafiles.resources',
    //TODO: Add deps for operator, User, Projector, ProjectionDefault, osTableFilter, osTableSort,
])

.controller('MediafileListCtrl', [
    '$http',
    '$scope',
    'gettext',
    'ngDialog',
    'osTableFilter',
    'osTableSort',
    'osTablePagination',
    'ProjectionDefault',
    'Projector',
    'User',
    'Mediafile',
    'MediafileForm',
    'Logos',
    'Fonts',
    function ($http, $scope, gettext, ngDialog, osTableFilter, osTableSort, osTablePagination,
              ProjectionDefault, Projector, User, Mediafile, MediafileForm, Logos, Fonts) {
        $scope.$watch(function () {
            return Mediafile.lastModified();
        }, function () {
            $scope.mediafiles = _.orderBy(Mediafile.getAll(), ['title']);
        });
        User.bindAll({}, $scope, 'users');
        $scope.$watch(function() {
            return Projector.lastModified();
        }, function() {
            $scope.projectors = Projector.getAll();
            updatePresentedMediafiles();
            var projectiondefault = ProjectionDefault.filter({name: 'mediafiles'})[0];
            if (projectiondefault) {
                $scope.defaultProjectorId = projectiondefault.projector_id;
            }
        });

        function updatePresentedMediafiles () {
            $scope.presentedMediafiles = [];
            Projector.getAll().forEach(function (projector) {
                var projectorElements = _.map(projector.elements, function(element) { return element; });
                var mediaElements = _.filter(projectorElements, function (element) {
                    return element.name === 'mediafiles/mediafile';
                });
                mediaElements.forEach(function (element) {
                    $scope.presentedMediafiles.push(element);
                });
            });
            if ($scope.presentedMediafiles.length) {
                $scope.isMeta = false;
            } else {
                $scope.isMeta = true;
            }
        }

        updatePresentedMediafiles();

        // Filtering
        $scope.filter = osTableFilter.createInstance('MediafilesTableFilter');

        if (!$scope.filter.existsStorageEntry()) {
            $scope.filter.booleanFilters = {
                isHidden: {
                    value: undefined,
                    displayName: gettext('Hidden'),
                    choiceYes: gettext('Is hidden'),
                    choiceNo: gettext('Is not hidden'),
                    needExtraPermission: true,
                },
                isPdf: {
                    value: undefined,
                    displayName: gettext('Is PDF'),
                    choiceYes: gettext('Is PDF file'),
                    choiceNo: gettext('Is no PDF file'),
                },
            };
        }
        $scope.filter.propertyList = ['title_or_filename'];
        $scope.filter.propertyFunctionList = [
            function (mediafile) {return mediafile.uploader.get_short_name();},
            function (mediafile) {return mediafile.mediafile.type;},
            function (mediafile) {return mediafile.mediafile.name;},
        ];
        // Sorting
        $scope.sort = osTableSort.createInstance('MediafileTableSort');
        if (!$scope.sort.column) {
            $scope.sort.column = 'title_or_filename';
        }
        $scope.sortOptions = [
            {name: 'title_or_filename',
             display_name: gettext('Title')},
            {name: 'mediafile.type',
             display_name: gettext('Type')},
            {name: 'filesize',
             display_name: gettext('File size')},
            {name: 'timestamp',
             display_name: gettext('Upload time')},
            {name: 'uploader.get_short_name()',
             display_name: gettext('Uploaded by')},
        ];

        // pagination
        $scope.pagination = osTablePagination.createInstance('MediafileTablePagination');

        // open new/edit dialog
        $scope.openDialog = function (mediafile) {
            ngDialog.open(MediafileForm.getDialog(mediafile));
        };

        // *** select mode functions ***
        $scope.isSelectMode = false;
        // check all checkboxes
        $scope.checkAll = function () {
            $scope.selectedAll = !$scope.selectedAll;
            _.forEach($scope.mediafiles, function (mediafile) {
                mediafile.selected = $scope.selectedAll;
            });
        };
        // uncheck all checkboxes if SelectMode is closed
        $scope.uncheckAll = function () {
            if (!$scope.isSelectMode) {
                $scope.selectedAll = false;
                _.forEach($scope.mediafiles, function (mediafile) {
                    mediafile.selected = false;
                });
            }
        };
        // delete all selected mediafiles
        $scope.deleteMultiple = function () {
            angular.forEach($scope.mediafiles, function (mediafile) {
                if (mediafile.selected)
                    Mediafile.destroy(mediafile.id);
            });
            $scope.isSelectMode = false;
            $scope.uncheckAll();
        };
        // delete single mediafile
        $scope.delete = function (mediafile) {
            Mediafile.destroy(mediafile.id);
        };

        // ** PDF presentation functions **/
        // show document on projector
        $scope.showMediafile = function (projectorId, mediafile) {
            var isProjectedIds = mediafile.isProjected();
            _.forEach(isProjectedIds, function (id) {
                $http.post('/rest/core/projector/' + id + '/clear_elements/');
            });
            if (_.indexOf(isProjectedIds, projectorId) == -1) {
                var postUrl = '/rest/core/projector/' + projectorId + '/prune_elements/';
                var data = [{
                        name: 'mediafiles/mediafile',
                        id: mediafile.id,
                        numPages: mediafile.mediafile.pages,
                        page: 1,
                        scale: 'page-fit',
                        rotate: 0,
                        visible: true,
                        playing: false,
                        fullscreen: mediafile.is_pdf || mediafile.is_image
                }];
                $http.post(postUrl, data);
            }
        };

        var sendMediafileCommand = function (mediafile, data) {
            var updateData = _.extend({}, mediafile);
            _.extend(updateData, data);
            var postData = {};
            postData[mediafile.uuid] = updateData;

            // Find Projector where the mediafile is projected
            $scope.projectors.forEach(function (projector) {
                if (_.find(projector.elements, function (e) {return e.uuid == mediafile.uuid;})) {
                    $http.post('/rest/core/projector/' + projector.id + '/update_elements/', postData);
                }
            });
        };

        $scope.getTitle = function (mediafile) {
            return Mediafile.get(mediafile.id).title;
        };

        $scope.getType = function (presentedMediafile) {
            var mediafile = Mediafile.get(presentedMediafile.id);
            return mediafile.is_pdf ? 'pdf' : mediafile.is_image ? 'image' : 'video';
        };

        $scope.mediafileGoToPage = function (mediafile, page) {
            page = parseInt(page);
            if (page > 0 && page <= mediafile.numPages) {
                sendMediafileCommand(
                    mediafile,
                    {page: page}
                );
            }
        };
        $scope.mediafileZoomIn = function (mediafile) {
            var scale = 1;
            if (parseFloat(mediafile.scale)) {
                scale = mediafile.scale;
            }
            sendMediafileCommand(
                mediafile,
                {scale: scale + 0.2}
            );
        };
        $scope.mediafileFit = function (mediafile) {
            sendMediafileCommand(
                mediafile,
                {scale: 'page-fit'}
            );
        };
        $scope.mediafileZoomOut = function (mediafile) {
            var scale = 1;
            if (parseFloat(mediafile.scale)) {
                scale = mediafile.scale;
            }
            sendMediafileCommand(
                mediafile,
                {scale: scale - 0.2}
            );
        };
        $scope.mediafileChangePage = function (mediafile, pageNum) {
            sendMediafileCommand(
                mediafile,
                {pageToDisplay: pageNum}
            );
        };
        $scope.mediafileRotate = function (mediafile) {
            var rotation = mediafile.rotate;
            if (rotation === 270) {
                rotation = 0;
            } else {
                rotation = rotation + 90;
            }
            sendMediafileCommand(
                mediafile,
                {rotate: rotation}
            );
        };
        $scope.mediafileToggleFullscreen = function (mediafile) {
            sendMediafileCommand(
                mediafile,
                {fullscreen: !mediafile.fullscreen}
            );
        };
        $scope.mediafileTogglePlaying = function (mediafile) {
            sendMediafileCommand(
                mediafile,
                {playing: !mediafile.playing}
            );
        };

        /** Logos and fonts **/
        $scope.logos = Logos.getAll();
        $scope.fonts = Fonts.getAll();
        $scope.hasProjectorHeaderLogo = function (mediafile) {
            return _.some(mediafile.getLogos(), function (logo) {
                return logo.key === 'logo_projector_header';
            });
        };
    }
])

/*
 * Special filter only for mediafile list view.
 */
.filter('hiddenFilter', [
    '$filter',
    'operator',
    function ($filter, operator) {
        return function (array) {
            if (operator.hasPerms('mediafiles.can_see_hidden')) {
                return array;
            }
            return Array.prototype.filter.call(array, function (item) {
                return !item.hidden;
            });
        };
    }
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.mediafiles.projector', [
    'OpenSlidesApp.mediafiles.resources',
    //TODO: Add deps for slidesProvider
])

.config([
    'slidesProvider',
    function (slidesProvider) {
        slidesProvider.registerSlide('mediafiles/mediafile', {
            template: 'static/templates/mediafiles/slide_mediafile.html'
        });
    }
])

.controller('SlideMediafileCtrl', [
    '$scope',
    '$timeout',
    'Mediafile',
    function ($scope, $timeout, Mediafile) {
        // load mediafile object
        Mediafile.bindOne($scope.element.id, $scope, 'mediafile');

        $scope.showPdf = true;

        // Watch for page changes in the projector element. Adjust the page
        // in the canvas scope, so the viewer can change the size automatically.
        $scope.$watch(function () {
            return $scope.element.page;
        }, function () {
            var canvasScope = angular.element('#pdf-canvas').scope();
            if (canvasScope) {
                canvasScope.pageNum = $scope.element.page;
            }
        });

        // Watch for scale changes. If the scale is changed, reload the pdf
        // viewer by just disable and re-enable it.
        $scope.$watch(function () {
            return $scope.element.scale;
        }, function () {
            $scope.showPdf = false;
            $timeout(function () {
                $scope.showPdf = true;
            }, 1);
        });

        // Allow the elements to render properly
        setTimeout(function() {
            if ($scope.mediafile) {
                if ($scope.mediafile.is_pdf) {
                    $scope.pdfName = $scope.mediafile.title;
                    $scope.pdfUrl = $scope.mediafile.mediafileUrl;
                } else if ($scope.mediafile.is_video) {
                    var player = angular.element.find('#video-player')[0];
                    if ($scope.element.playing) {
                        player.play();
                    } else {
                        player.pause();
                    }
                }
            }
        }, 0);
    }
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.mediafiles.resources', [
    'gettext',
    'js-data',
    //TODO: Add deps for jsDataModel
])

.factory('Mediafile', [
    'DS',
    'gettext',
    'jsDataModel',
    'Logos',
    'Fonts',
    function (DS, gettext, jsDataModel, Logos, Fonts) {
        var name = 'mediafiles/mediafile';
        return DS.defineResource({
            name: name,
            useClass: jsDataModel,
            verboseName: gettext('Files'),
            verboseNamePlural: gettext('Files'),
            getAllImages: function () {
                var images = [];
                angular.forEach(this.getAll(), function(file) {
                    if (file.is_image) {
                        images.push({title: file.title, value: file.mediafileUrl});
                    }
                });
                return images;
            },
            methods: {
                getResourceName: function () {
                    return name;
                },
                // link name which is shown in search result
                getSearchResultName: function () {
                    return this.title;
                },
                // return true if a specific relation matches for given searchquery
                // (here: speakers)
                hasSearchResult: function (results) {
                    var mediafile = this;
                    // search for speakers (check if any user.id from already found users matches)
                    return _.some(results, function(result) {
                        if ((result.getResourceName() === "users/user") &&
                                (mediafile.uploader_id === result.id)) {
                            return true;
                        }
                    });
                },
                isUsedAsLogo: function () {
                    var mediafile = this;
                    return _.find(Logos.getAll(), function (logoPlaceholder) {
                        return logoPlaceholder.path === mediafile.mediafileUrl;
                    });
                },
                canBeUsedAsLogo: function () {
                    return this.is_image;
                },
                getLogos: function () {
                    var mediafile = this;
                    return _.filter(Logos.getAll(), function (logoPlaceholder) {
                        return logoPlaceholder.path === mediafile.mediafileUrl;
                    });
                },
                hasLogo: function (logo) {
                    var allUrls = _.map(this.getLogos(), function (logo) {
                       return logo.path;
                    });
                    return _.includes(allUrls, logo.path);
                },
                toggleLogo: function (logo) {
                    if (this.hasLogo(logo)) {
                        Logos.set(logo.key);
                    } else {
                        Logos.set(logo.key, this.mediafileUrl);
                    }
                },
                isUsedAsFont: function () {
                    var mediafile = this;
                    return _.find(Fonts.getAll(), function (font) {
                        return font.path === mediafile.mediafileUrl;
                    });
                },
                canBeUsedAsFont: function () {
                    return this.is_font;
                },
                getFonts: function () {
                    var mediafile = this;
                    return _.filter(Fonts.getAll(), function (font) {
                        return font.path === mediafile.mediafileUrl;
                    });
                },
                hasFont: function (font) {
                    var allUrls = _.map(this.getFonts(), function (font) {
                       return font.path;
                    });
                    return _.includes(allUrls, font.path);
                },
                toggleFont: function (font) {
                    if (this.hasFont(font)) {
                        Fonts.set(font.key);
                    } else {
                        Fonts.set(font.key, this.mediafileUrl);
                    }
                },
            },
            computed: {
                is_pdf: ['filetype', function (filetype) {
                    var PDF_FILE_TYPES = ['application/pdf'];
                    return _.includes(PDF_FILE_TYPES, filetype);
                }],
                is_image: ['filetype', function (filetype) {
                    var IMAGE_FILE_TYPES = ['image/png', 'image/jpeg', 'image/gif'];
                    return _.includes(IMAGE_FILE_TYPES, filetype);
                }],
                is_video: ['filetype', function (filetype) {
                    var VIDEO_FILE_TYPES = [ 'video/quicktime', 'video/mp4', 'video/webm',
                        'video/ogg', 'video/x-flv', 'application/x-mpegURL', 'video/MP2T',
                        'video/3gpp', 'video/x-msvideo', 'video/x-ms-wmv', 'video/x-matroska' ];
                    return _.includes(VIDEO_FILE_TYPES, filetype);
                }],
                is_presentable: ['is_pdf', 'is_image', 'is_video', function (is_pdf, is_image, is_video) {
                    return (is_pdf && !this.mediafile.encrypted) || is_image || is_video;
                }],
                is_font: [function () {
                    var FONT_FILE_EXTENSIONS = ['ttf', 'woff'];
                    var ext = _.last(this.mediafile.name.split('.'));
                    return _.includes(FONT_FILE_EXTENSIONS, ext);
                }],
                mediafileUrl: [function () {
                    return this.media_url_prefix + this.mediafile.name;
                }],
                filename: [function () {
                    var filename = this.mediafile.name;
                    return /\/(.+?)$/.exec(filename)[1];
                }],
                filetype: [function () {
                    return this.mediafile.type || gettext('undefined');
                }],
                title_or_filename: ['title', 'mediafile', function (title) {
                    return title || this.filename;
                }]
            },
            relations: {
                belongsTo: {
                    'users/user': {
                        localField: 'uploader',
                        localKey: 'uploader_id',
                    }
                }
            }
        });
    }
])

.run(['Mediafile', function (Mediafile) {}]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.mediafiles.site', [
    'OpenSlidesApp.mediafiles.list',
    'OpenSlidesApp.mediafiles.states',
    'OpenSlidesApp.mediafiles.update',
    'OpenSlidesApp.mediafiles.upload',
    'OpenSlidesApp.mediafiles.image-plugin',
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.mediafiles.states', [
    'gettext',
    'ui.router',
    //TODO: Add deps for mainMenuProvider
])

.config([
    'gettext',
    'mainMenuProvider',
    function (gettext, mainMenuProvider) {
        mainMenuProvider.register({
            'ui_sref': 'mediafiles.mediafile.list',
            'img_class': 'paperclip',
            'title': gettext('Files'),
            'weight': 600,
            'perm': 'mediafiles.can_see',
        });
    }
])

.config([
    'SearchProvider',
    'gettext',
    function (SearchProvider, gettext) {
        SearchProvider.register({
            'verboseName': gettext('Files'),
            'collectionName': 'mediafiles/mediafile',
            'urlDetailState': 'mediafiles.mediafile.detail',
            'weight': 600,
        });
    }
])

.config([
    'gettext',
    '$stateProvider',
    function (gettext, $stateProvider) {
        $stateProvider
        .state('mediafiles', {
            url: '/mediafiles',
            abstract: true,
            template: "<ui-view/>",
            data: {
                title: gettext('Files'),
                basePerm: 'mediafiles.can_see',
            },
        })
        .state('mediafiles.mediafile', {
            abstract: true,
            template: "<ui-view/>",
        })
        .state('mediafiles.mediafile.list', {});
    }
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.mediafiles.update', [
    'OpenSlidesApp.mediafiles.resources',
    //TODO: Add deps for operator, User
])

.controller('MediafileUpdateCtrl', [
    '$scope',
    'operator',
    'User',
    'Mediafile',
    'mediafileId',
    'MediafileForm',
    'ErrorMessage',
    function ($scope, operator, User, Mediafile, mediafileId, MediafileForm, ErrorMessage) {
        $scope.alert = {};
        $scope.formFields = MediafileForm.getFormFields();

        // set initial values for form model by create deep copy of motion object
        // so list/detail view is not updated while editing
        $scope.model = angular.copy(Mediafile.get(mediafileId));

        // save mediafile
        $scope.save = function (mediafile) {
            // reset title and uploader_id if empty
            if (!mediafile.title) {
                mediafile.title = mediafile.filename;
            }
            if (!mediafile.uploader_id) {
                mediafile.uploader_id = operator.user.id;
            }
            // inject the changed mediafile (copy) object back into DS store
            Mediafile.inject(mediafile);
            // save change mediafile object on server
            Mediafile.save(mediafile).then(
                function (success) {
                    $scope.closeThisDialog();
                },
                function (error) {
                    Mediafile.refresh(mediafile);
                    $scope.alert = ErrorMessage.forAlert(error);
                }
            );
        };
    }
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.mediafiles.upload', [
    'OpenSlidesApp.mediafiles.forms',
    'ngFileUpload',
])

.controller('MediafileUploadCtrl', [
    '$scope',
    '$q',
    'User',
    'Upload',
    'operator',
    'gettextCatalog',
    'ErrorMessage',
    function ($scope, $q, User, Upload, operator, gettextCatalog, ErrorMessage) {
        User.bindAll({}, $scope, 'users');
        $scope.alert = {};
        $scope.files = [];
        $scope.uploading = false;
        var idCounter = 0; // Used for uniqly identifing each file in $scope.files.

        // Convert bytes to human readable si units.
        var humanFileSize = function (bytes) {
            if(Math.abs(bytes) < 1000) {
                return bytes + ' B';
            }
            var units = ['kB','MB','GB','TB','PB','EB','ZB','YB'];
            var i = -1;
            do {
                bytes /= 1000;
                i++;
            } while(bytes >= 1000 && i < units.length - 1);

            return bytes.toFixed(1) + ' ' + units[i];
        };

        $scope.addFiles = function (files) {
            files = _.map(files, function (file) {
                idCounter += 1;
                // This is a client side representation used for the template
                return {
                    id: idCounter,
                    file: file,
                    title: file.name,
                    hidden: false,
                    uploader_id: operator.user.id,
                    name: file.name,
                    size: file.size,
                    humanSize: humanFileSize(file.size),
                    type: file.type,
                    progress: 0,
                };
            });
            // Add each file, that is not a duplicate to $scope.files
            _.forEach(files, function (file) {
                var duplicate = _.some($scope.files, function (_file) {
                    return file.name === _file.name &&
                        file.size === _file.size &&
                        file.type === _file.type;
                });
                if (!duplicate) {
                    $scope.files.push(file);
                }
            });
        };

        $scope.removeFile = function (id) {
            $scope.files = _.filter($scope.files, function (file) {
                return file.id !== id;
            });
        };

        // Add files via drag and drop
        $scope.$watch('dropFiles', function () {
            if ($scope.dropFiles) {
                $scope.addFiles($scope.dropFiles);
            }
        });

        // upload all files
        $scope.upload = function () {
            $scope.uploading = true;
            var promises = _.map($scope.files, function (file) {
                // clear error
                file.error = void 0;

                // Check, if all necessary fields are set.
                if (!file.title) {
                    file.title = file.file.name;
                }
                if (!file.uploader_id) {
                    file.uploader_id = operator.user.id;
                }

                return Upload.upload({
                    url: '/rest/mediafiles/mediafile/',
                    method: 'POST',
                    data: {
                        mediafile: file.file,
                        title: file.title,
                        uploader_id: file.uploader_id,
                        hidden: file.hidden
                    },
                }).then(
                    function (success) {
                        $scope.removeFile(file.id);
                    },
                    function (error) {
                        file.error = ErrorMessage.forAlert(error).msg;
                        return error;
                    },
                    function (progress) {
                        file.progress = parseInt(100.0 * progress.loaded / progress.total);
                    }
                );
            });

            $q.all(promises).then(function (success) {
                var errors = _.filter(success, function (entry) {
                    return entry;
                });

                if (errors.length) {
                    $scope.uploading = false;
                    var message = gettextCatalog.getString('Some files could not be uploaded');
                    $scope.alert = { type: 'danger', msg: message, show: true };
                } else {
                    $scope.close();
                }
            });
        };

        $scope.clear = function () {
            $scope.uploading = false;
            $scope.files = [];
        };

        $scope.close = function () {
            $scope.closeThisDialog();
        };
    }
])

.run([
    'gettext',
    function (gettext) {
        gettext('Some files could not be uploaded');
    }
]);

}());

(function () {

"use strict";

angular.module('OpenSlidesApp.motions', [
    'OpenSlidesApp.motions.motionBlock',
    'OpenSlidesApp.motions.lineNumbering',
    'OpenSlidesApp.motions.diff',
    'OpenSlidesApp.poll.majority',
    'OpenSlidesApp.users',
])

.factory('WorkflowState', [
    'DS',
    function (DS) {
        return DS.defineResource({
            name: 'motions/workflowstate',
            methods: {
                getNextStates: function () {
                    // TODO: Use filter with params with operator 'in'.
                    var states = [];
                    _.forEach(this.next_states_id, function (stateId) {
                        states.push(DS.get('motions/workflowstate', stateId));
                    });
                    return states;
                },
                getRecommendations: function () {
                    var params = {
                        where: {
                            'workflow_id': {
                                '==': this.workflow_id
                            },
                            'recommendation_label': {
                                '!=': null
                            }
                        }
                    };
                    return DS.filter('motions/workflowstate', params);
                }
            }
        });
    }
])

.factory('Workflow', [
    'DS',
    'WorkflowState',
    function (DS, WorkflowState) {
        return DS.defineResource({
            name: 'motions/workflow',
            relations: {
                hasMany: {
                    'motions/workflowstate': {
                        localField: 'states',
                        foreignKey: 'workflow_id',
                    }
                }
            }
        });
    }
])

.factory('MotionPoll', [
    'DS',
    'gettextCatalog',
    'Config',
    'MajorityMethods',
    function (DS, gettextCatalog, Config, MajorityMethods) {
        return DS.defineResource({
            name: 'motions/motion-poll',
            relations: {
                belongsTo: {
                    'motions/motion': {
                        localField: 'motion',
                        localKey: 'motion_id',
                    }
                }
            },
            methods: {
                // Returns percent base. Returns undefined if calculation is not possible in general.
                getPercentBase: function (config, type) {
                    var base;
                    switch (config) {
                        case 'CAST':
                            if (this.votescast <= 0 || this.votesinvalid < 0) {
                                // It would be OK to check only this.votescast < 0 because 0
                                // is checked again later but this is a little bit faster.
                                break;
                            }
                            base = this.votescast;
                            /* falls through */
                        case 'VALID':
                            if (this.votesvalid < 0) {
                                base = void 0;
                                break;
                            }
                            if (typeof base === 'undefined' && type !== 'votescast' && type !== 'votesinvalid') {
                                base = this.votesvalid;
                            }
                            /* falls through */
                        case 'YES_NO_ABSTAIN':
                            if (this.abstain < 0) {
                                base = void 0;
                                break;
                            }
                            if (typeof base === 'undefined' && type !== 'votescast' && type !== 'votesinvalid' && type !== 'votesvalid') {
                                base = this.yes + this.no + this.abstain;
                            }
                            /* falls through */
                        case 'YES_NO':
                            if (this.yes < 0 || this.no < 0 || this.abstain === -1 ) {
                                // It is not allowed to set 'Abstain' to 'majority' but exclude it from calculation.
                                // Setting 'Abstain' to 'undocumented' is possible, of course.
                                base = void 0;
                                break;
                            }
                            if (typeof base === 'undefined' && (type === 'yes' || type === 'no')) {
                                base = this.yes + this.no;
                            }
                    }
                    return base;
                },

                // Returns object with value and percent for this poll.
                getVote: function (vote, type) {
                    if (!this.has_votes) {
                        // Return undefined if this poll has no votes.
                        return;
                    }

                    // Initial values
                    var value = '',
                        percentStr = '',
                        percentNumber,
                        config = Config.get('motions_poll_100_percent_base').value;

                    // Check special values
                    switch (vote) {
                        case -1:
                            value = gettextCatalog.getString('majority');
                            break;
                        case -2:
                            value = gettextCatalog.getString('undocumented');
                            break;
                        default:
                            if (vote >= 0) {
                                value = vote;
                            } else {
                                value = 0;  // Vote was not defined. Set value to 0.
                            }
                    }

                    // Calculate percent value
                    var base = this.getPercentBase(config, type);
                    if (base) {
                        percentNumber = Math.round(vote * 100 / (base) * 100) / 100;
                        percentStr = '(' + percentNumber + ' %)';
                    }
                    return {
                        'value': value,
                        'percentStr': percentStr,
                        'percentNumber': percentNumber,
                        'display': value + ' ' + percentStr
                    };
                },

                // Returns 0 or positive integer if quorum is reached or surpassed.
                // Returns negativ integer if quorum is not reached.
                // Returns undefined if we can not calculate the quorum.
                isReached: function (method) {
                    if (!this.has_votes) {
                        // Return undefined if this poll has no votes.
                        return;
                    }

                    var isReached;
                    var config = Config.get('motions_poll_100_percent_base').value;
                    var base = this.getPercentBase(config, 'yes');
                    if (base) {
                        // Provide result only if base is not undefined and not 0.
                        isReached = MajorityMethods[method](this.yes, base);
                    }
                    return isReached;
                }
            }
        });
    }
])

.factory('MotionStateAndRecommendationParser', [
    'DS',
    'gettextCatalog',
    function (DS, gettextCatalog) {
        return {
            formatMotion: function (motion) {
                return '[motion:' + motion.id + ']';
            },
            parse: function (recommendation) {
                return recommendation.replace(/\[motion:(\d+)\]/g, function (match, id) {
                    var motion = DS.get('motions/motion', id);
                    if (motion) {
                        return motion.identifier ? motion.identifier : motion.getTitle();
                    } else {
                        return gettextCatalog.getString('<unknown motion>');
                    }
                });
            },
        };
    }
])


.factory('Motion', [
    'DS',
    '$http',
    'MotionPoll',
    'MotionStateAndRecommendationParser',
    'MotionChangeRecommendation',
    'MotionComment',
    'jsDataModel',
    'gettext',
    'gettextCatalog',
    'Config',
    'lineNumberingService',
    'diffService',
    'OpenSlidesSettings',
    'Projector',
    'ProjectHelper',
    'operator',
    function(DS, $http, MotionPoll, MotionStateAndRecommendationParser, MotionChangeRecommendation,
        MotionComment, jsDataModel, gettext, gettextCatalog, Config, lineNumberingService,
        diffService, OpenSlidesSettings, Projector, ProjectHelper, operator) {
        var name = 'motions/motion';
        return DS.defineResource({
            name: name,
            useClass: jsDataModel,
            verboseName: gettext('Motion'),
            verboseNamePlural: gettext('Motions'),
            validate: function (resource, data, callback) {
                MotionComment.populateFieldsReverse(data);
                callback(null, data);
            },
            computed: {
                isAmendment: function () {
                    return this.parent_id !== null;
                },
            },
            methods: {
                getResourceName: function () {
                    return name;
                },
                getVersion: function (versionId) {
                    versionId = versionId || this.active_version;
                    var index;
                    if (versionId == -1) {
                        index = this.versions.length - 1;
                    } else {
                        index = _.findIndex(this.versions, function (element) {
                            return element.id == versionId;
                        });
                    }
                    return this.versions[index] || {};
                },
                getTitle: function (versionId) {
                    return this.getVersion(versionId).title;
                },
                getAgendaTitle: function () {
                    var title = gettextCatalog.getString('Motion');
                    if (this.identifier) {
                        title += ' ' + this.identifier;
                    } else {
                        title += ' (' + this.getTitle() + ')';
                    }
                    return title;
                },
                getListOfSpeakersTitle: function () {
                    var title = gettextCatalog.getString('Motion');
                    if (this.identifier) {
                        title += ' ' + this.identifier;
                    } else {
                        title += ' (' + this.getTitle() + ')';
                    }
                    return title;
                },
                getTitleWithChanges: function (changeRecommendationMode, versionId) {
                    var titleChange = this.getTitleChangeRecommendation(versionId);
                    var title;
                    if (titleChange) {
                        if (changeRecommendationMode === "changed") {
                            title = titleChange.text;
                        } else if (changeRecommendationMode === 'agreed' && !titleChange.rejected) {
                            title = titleChange.text;
                        } else {
                            title = this.getTitle();
                        }
                    } else {
                        title = this.getTitle();
                    }
                    return title;
                },
                getSequentialNumber: function () {
                    var id = this.id + '';
                    var zeros = Math.max(0, OpenSlidesSettings.MOTION_IDENTIFIER_MIN_DIGITS - id.length);
                    for (var i = 0; i < zeros; i++) {
                        id = '0' + id;
                    }
                    return id;
                },
                getText: function (versionId) {
                    return this.getVersion(versionId).text;
                },
                getTextWithLineBreaks: function (versionId, highlight, callback) {
                    var lineLength = Config.get('motions_line_length').value,
                        html = this.getVersion(versionId).text;

                    return lineNumberingService.insertLineNumbers(html, lineLength, highlight, callback);
                },
                getTextBetweenChangeRecommendations: function (versionId, change1, change2, highlight) {
                    var line_from = (change1 ? change1.line_to : 1),
                        line_to = (change2 ? change2.line_from : null);

                    if (line_from > line_to) {
                        throw 'Invalid call of getTextBetweenChangeRecommendations: change1 needs to be before change2';
                    }
                    if (line_from == line_to) {
                        return '';
                    }

                    var lineLength = Config.get('motions_line_length').value,
                        html = lineNumberingService.insertLineNumbers(this.getVersion(versionId).text, lineLength),
                        data;

                    try {
                        data = diffService.extractRangeByLineNumbers(html, line_from, line_to);
                    } catch (e) {
                        // This only happens (as far as we know) when the motion text has been altered (shortened)
                        // without modifying the change recommendations accordingly.
                        // That's a pretty serious inconsistency that should not happen at all,
                        // we're just doing some basic damage control here.
                        var msg = 'Inconsistent data. A change recommendation is probably referring to a non-existant line number.';
                        return '<em style="color: red; font-weight: bold;">' + msg + '</em>';
                    }

                    // Add "merge-before"-css-class if the first line begins in the middle of a paragraph. Used for PDF.
                    html = diffService.addCSSClassToFirstTag(data.outerContextStart + data.innerContextStart, "merge-before") +
                        data.html + data.innerContextEnd + data.outerContextEnd;
                    html = lineNumberingService.insertLineNumbers(html, lineLength, highlight, null, line_from);

                    return html;
                },
                getTextRemainderAfterLastChangeRecommendation: function(versionId, changes, highlight) {
                    var maxLine = 0;
                    for (var i = 0; i < changes.length; i++) {
                        if (changes[i].line_to > maxLine) {
                            maxLine = changes[i].line_to;
                        }
                    }

                    var lineLength = Config.get('motions_line_length').value,
                        html = lineNumberingService.insertLineNumbers(this.getVersion(versionId).text, lineLength),
                        data;

                    try {
                        data = diffService.extractRangeByLineNumbers(html, maxLine, null);
                    } catch (e) {
                        // This only happens (as far as we know) when the motion text has been altered (shortened)
                        // without modifying the change recommendations accordingly.
                        // That's a pretty serious inconsistency that should not happen at all,
                        // we're just doing some basic damage control here.
                        var msg = 'Inconsistent data. A change recommendation is probably referring to a non-existant line number.';
                        return '<em style="color: red; font-weight: bold;">' + msg + '</em>';
                    }

                    if (data.html !== '') {
                        // Add "merge-before"-css-class if the first line begins in the middle of a paragraph. Used for PDF.
                        html = diffService.addCSSClassToFirstTag(data.outerContextStart + data.innerContextStart, "merge-before") +
                            data.html + data.innerContextEnd + data.outerContextEnd;
                        html = lineNumberingService.insertLineNumbers(html, lineLength, highlight, null, maxLine);
                    } else {
                        // Prevents empty lines at the end of the motion
                        html = '';
                    }
                    return html;
                },
                _getTextWithChangeRecommendations: function (versionId, highlight, lineBreaks, statusCompareCb) {
                    var lineLength = Config.get('motions_line_length').value,
                        html = this.getVersion(versionId).text,
                        changes = this.getTextChangeRecommendations(versionId, 'DESC');

                    for (var i = 0; i < changes.length; i++) {
                        var change = changes[i];
                        if (typeof statusCompareCb === 'undefined' || statusCompareCb(change.rejected)) {
                            html = lineNumberingService.insertLineNumbers(html, lineLength, null, null, 1);
                            html = diffService.replaceLines(html, change.text, change.line_from, change.line_to);
                        }
                    }

                    if (lineBreaks) {
                        html = lineNumberingService.insertLineNumbers(html, lineLength, highlight, null, 1);
                    }

                    return html;
                },
                getTextWithAllChangeRecommendations: function (versionId, highlight, lineBreaks) {
                    return this._getTextWithChangeRecommendations(versionId, highlight, lineBreaks, function() {
                        return true;
                    });
                },
                getTextWithoutRejectedChangeRecommendations: function (versionId, highlight, lineBreaks) {
                    return this._getTextWithChangeRecommendations(versionId, highlight, lineBreaks, function(rejected) {
                        return !rejected;
                    });
                },
                getTextByMode: function(mode, versionId, highlight, lineBreaks) {
                    /*
                     * @param mode ['original', 'diff', 'changed', 'agreed']
                     * @param versionId [if undefined, active_version will be used]
                     * @param highlight [the line number to highlight]
                     * @param lineBreaks [if line numbers / breaks should be included in the result]
                     */

                    lineBreaks = (lineBreaks === undefined ? true : lineBreaks);

                    var text;
                    switch (mode) {
                        case 'original':
                            if (lineBreaks) {
                                text = this.getTextWithLineBreaks(versionId, highlight);
                            } else {
                                text = this.getVersion(versionId).text;
                            }
                            break;
                        case 'diff':
                            var changes = this.getTextChangeRecommendations(versionId, 'ASC');
                            text = '';
                            for (var i = 0; i < changes.length; i++) {
                                text += this.getTextBetweenChangeRecommendations(versionId, (i === 0 ? null : changes[i - 1]), changes[i], highlight);
                                text += changes[i].getDiff(this, versionId, highlight);
                            }
                            text += this.getTextRemainderAfterLastChangeRecommendation(versionId, changes);

                            if (!lineBreaks) {
                                text = lineNumberingService.stripLineNumbers(text);
                            }
                            break;
                        case 'changed':
                            text = this.getTextWithAllChangeRecommendations(versionId, highlight, lineBreaks);
                            break;
                        case 'agreed':
                            text = this.getTextWithoutRejectedChangeRecommendations(versionId, highlight, lineBreaks);
                            break;
                    }
                    return text;
                },
                setTextStrippingLineBreaks: function (text) {
                    this.text = lineNumberingService.stripLineNumbers(text);
                },
                getReason: function (versionId) {
                    return this.getVersion(versionId).reason;
                },
                // full state name - optional with custom state name extension
                // depended by state and provided by a custom comment field
                getStateName: function () {
                    var name = '';
                    if (this.state) {
                        name = gettextCatalog.getString(this.state.name);
                        if (this.state.show_state_extension_field) {
                            // check motion comment fields for flag 'forState'
                            var commentFieldForStateId = MotionComment.getFieldIdForFlag('forState');
                            if (commentFieldForStateId > -1) {
                                name += ' ' + this.comments[commentFieldForStateId];
                            }
                        }
                    }
                    return MotionStateAndRecommendationParser.parse(name);
                },
                // full recommendation string - optional with custom recommendationextension
                // depended by state and provided by a custom comment field
                getRecommendationName: function () {
                    var recommendation = '';
                    if (Config.get('motions_recommendations_by').value !== '' && this.recommendation) {
                        recommendation = gettextCatalog.getString(this.recommendation.recommendation_label);
                        if (this.recommendation.show_recommendation_extension_field) {
                            // check motion comment fields for flag 'forRecommendation'
                            var commentFieldForRecommendationId = MotionComment.getFieldIdForFlag('forRecommendation');
                            if (commentFieldForRecommendationId > -1) {
                                recommendation += ' ' + this.comments[commentFieldForRecommendationId];
                            }
                        }
                    }
                    return MotionStateAndRecommendationParser.parse(recommendation);
                },
                // link name which is shown in search result
                getSearchResultName: function () {
                    return this.getTitle();
                },
                // return true if a specific relation matches for given searchquery
                // e.g. submitter, supporters or category
                hasSearchResult: function (results, searchquery) {
                    var motion = this;
                    // search for submitters and supporters (check if any user.id from already found users matches)
                    var foundSomething = _.some(results, function(result) {
                        if (result.getResourceName() === "users/user") {
                            if (_.some(motion.submitters, {'id': result.id})) {
                                return true;
                            } else if (_.some(motion.supporters, { 'id': result.id })) {
                                return true;
                            }
                        }
                    });
                    // search for category
                    if (!foundSomething && motion.category && motion.category.name.match(new RegExp(searchquery, 'i'))) {
                        foundSomething = true;
                    }

                    // search for change recommendation
                    if (!foundSomething) {
                        var recommendations = MotionChangeRecommendation.filter({
                            where: {motion_version_id: this.active_version}
                        });
                        foundSomething = _.some(recommendations, function(recommendation) {
                            if (recommendation.text.match(new RegExp(searchquery, 'i'))) {
                                return true;
                            }
                        });
                    }
                    return foundSomething;
                },
                getTextChangeRecommendations: function (versionId, order) {
                    /*
                     * Returns all change recommendations for this given version, sorted by line
                     * @param versionId
                     * @param order ['DESC' or 'ASC' (default)]
                     * @returns {*}
                     */
                    versionId = versionId || this.active_version;
                    order = order || 'ASC';
                    return MotionChangeRecommendation.filter({
                        where: {
                            motion_version_id: versionId
                        },
                        orderBy: [
                            ['line_from', order]
                        ]
                    }).filter(function(change) {
                        return change.isTextRecommendation();
                    });
                },
                getTitleChangeRecommendation: function (versionId) {
                    /**
                     * Returns the change recommendation affecting the title, or null
                     * @param versionId
                     * @returns MotionChangeRecommendation|null
                     */
                    versionId = versionId || this.active_version;
                    var changes = MotionChangeRecommendation.filter({
                        where: {
                            motion_version_id: versionId,
                            line_from: 0,
                            line_to: 0
                        }
                    });
                    return (changes.length > 0 ? changes[0] : null);
                },
                hasAmendments: function () {
                    return DS.filter('motions/motion', {parent_id: this.id}).length > 0;
                },
                isAllowed: function (action) {
                    /*
                     * Return true if the requested user is allowed to do the specific action.
                     * There are the following possible actions.
                     * - see
                     * - update
                     * - delete
                     * - create_poll
                     * - support
                     * - unsupport
                     * - change_state
                     * - reset_state
                     * - change_comments
                     * - change_recommendation
                     * - can_manage
                     * - can_see_amendments
                     * - can_create_amendments
                     *
                     *  NOTE: If you update this function please think about
                     *        server permissions, see motions/views.py.
                     */
                    switch (action) {
                        case 'see':
                            return (
                                operator.hasPerms('motions.can_see') &&
                                (
                                    !this.state.required_permission_to_see ||
                                    operator.hasPerms(this.state.required_permission_to_see) ||
                                    (operator.user in this.submitters)
                                )
                            );
                        case 'update':
                            return (
                                operator.hasPerms('motions.can_manage') ||
                                (
                                    (_.indexOf(this.submitters, operator.user) !== -1) &&
                                    this.state.allow_submitter_edit
                                )
                            );
                        case 'delete':
                            return (
                                operator.hasPerms('motions.can_manage') ||
                                (
                                    (_.indexOf(this.submitters, operator.user) !== -1) &&
                                    this.state.allow_submitter_edit
                                )
                            );
                        case 'create_poll':
                            return (
                                operator.hasPerms('motions.can_manage') &&
                                this.state &&
                                this.state.allow_create_poll
                            );
                        case 'support':
                            return (
                                operator.hasPerms('motions.can_support') &&
                                this.state.allow_support &&
                                Config.get('motions_min_supporters').value > 0 &&
                                (_.indexOf(this.submitters, operator.user) === -1) &&
                                (_.indexOf(this.supporters, operator.user) === -1)
                            );
                        case 'unsupport':
                            return this.state.allow_support && _.indexOf(this.supporters, operator.user) !== -1;
                        case 'change_state':
                            return operator.hasPerms('motions.can_manage');
                        case 'reset_state':
                            return operator.hasPerms('motions.can_manage');
                        case 'change_comments':
                            return operator.hasPerms('motions.can_manage_comments');
                        case 'change_recommendation':
                            return operator.hasPerms('motions.can_manage');
                        case 'can_manage':
                            return operator.hasPerms('motions.can_manage');
                        case 'can_see_amendments':
                            var result;
                            if (operator.hasPerms('motions.can_create')) {
                                result = Config.get('motions_amendments_enabled').value &&
                                    (this.hasAmendments() || this.isAllowed('can_create_amendment'));
                            } else if (operator.hasPerms('motions.can_see')) {
                                result = Config.get('motions_amendments_enabled').value && this.hasAmendments();
                            }
                            return result;
                        case 'can_create_amendment':
                            return (
                                operator.hasPerms('motions.can_create') &&
                                Config.get('motions_amendments_enabled').value &&
                                ( !this.isAmendment ||
                                  (this.isAmendment && OpenSlidesSettings.MOTIONS_ALLOW_AMENDMENTS_OF_AMENDMENTS))
                            );
                        default:
                            return false;
                    }
                },
                /* Overrides from jsDataModel factory.
                 * Also sets the projection mode if given; If not it projects in 'original' mode. */
                project: function (projectorId, mode) {
                    // if this object is already projected on projectorId, delete this element from this projector
                    var requestData = {
                        clear_ids: this.isProjected(),
                    };
                    // Was there a projector with the same id and mode as the given id and mode?
                    // If not, project the motion.
                    var wasProjectedBefore = _.some(this.isProjectedWithMode(), function (mapping) {
                        var value = (mapping.projectorId === projectorId);
                        if (mode) {
                            value = value && (mapping.mode === mode);
                        }
                        return value;
                    });
                    mode = mode || Config.get('motions_recommendation_text_mode').value;
                    if (!wasProjectedBefore) {
                        requestData.prune = {
                            id: projectorId,
                            element: {
                                name: name,
                                id: this.id,
                                mode: mode,
                            },
                        };
                    }
                    return ProjectHelper.project(requestData);
                },
                isProjected: function (mode) {
                    var self = this;
                    var predicate = function (element) {
                        var value = element.name === name &&
                            element.id === self.id;
                        if (mode) {
                            value = value && (element.mode === mode);
                        }
                        return value;
                    };
                    var projectorIds = [];
                    _.forEach(Projector.getAll(), function (projector) {
                        if (typeof _.findKey(projector.elements, predicate) === 'string') {
                            projectorIds.push(projector.id);
                        }
                    });
                    return projectorIds;
                },
                /* returns a list of mappings between projector id and mode:
                 * [ {projectorId: 2, mode: 'original'}, ... ] */
                isProjectedWithMode: function () {
                    var self = this;
                    var mapping = [];
                    _.forEach(Projector.getAll(), function (projector) {
                        _.forEach(projector.elements, function (element) {
                            if (element.name === name && element.id === self.id) {
                                mapping.push({
                                    projectorId: projector.id,
                                    mode: element.mode || 'original',
                                });
                            }
                        });
                    });
                    return mapping;
                },
                isRelatedProjected: function () {
                    // A motion related object is the list of speakers (through the agenda item)
                    if (this.agenda_item) {
                        return this.agenda_item.isListOfSpeakersProjected();
                    } else {
                        return [];
                    }
                },
            },
            relations: {
                belongsTo: {
                    'motions/category': {
                        localField: 'category',
                        localKey: 'category_id',
                    },
                    'motions/motion-block': {
                        localField: 'motionBlock',
                        localKey: 'motion_block_id',
                    },
                    'agenda/item': {
                        localKey: 'agenda_item_id',
                        localField: 'agenda_item',
                    }
                },
                hasMany: {
                    'core/tag': {
                        localField: 'tags',
                        localKeys: 'tags_id',
                    },
                    'mediafiles/mediafile': {
                        localField: 'attachments',
                        localKeys: 'attachments_id',
                    },
                    'users/user': [
                        {
                            localField: 'submitters',
                            localKeys: 'submitters_id',
                        },
                        {
                            localField: 'supporters',
                            localKeys: 'supporters_id',
                        }
                    ],
                    'motions/motion-poll': {
                        localField: 'polls',
                        foreignKey: 'motion_id',
                    }
                },
                hasOne: {
                    'motions/workflowstate': [
                        {
                            localField: 'state',
                            localKey: 'state_id',
                        },
                        {
                            localField: 'recommendation',
                            localKey: 'recommendation_id',
                        }
                    ]
                }
            }
        });
    }
])

// Service for generic comment fields
.factory('MotionComment', [
    '$filter',
    'Config',
    'operator',
    'Editor',
    function ($filter, Config, operator, Editor) {
        return {
            isSpecialCommentField: function (field) {
                if (field) {
                    return field.forState || field.forRecommendation;
                } else {
                    return false;
                }
            },
            getCommentsFields: function () {
                var fields = Config.get('motions_comments').value;
                return $filter('excludeDeletedAndForbiddenCommentsFields')(fields);
            },
            getNoSpecialCommentsFields: function () {
                var fields = this.getCommentsFields();
                return $filter('excludeSpecialCommentsFields')(fields);
            },
            getFormFields: function () {
                var fields = this.getNoSpecialCommentsFields();
                return _.map(fields, function (field, id) {
                        return {
                            key: 'comment_' + id,
                            type: 'editor',
                            templateOptions: {
                                label: field.name,
                            },
                            data: {
                                ckeditorOptions: Editor.getOptions()
                            },
                            hide: !operator.hasPerms("motions.can_manage_comments")
                        };
                    }
                );
            },
            populateFields: function (motion) {
                // Populate content of motion.comments to the single comment
                var fields = this.getCommentsFields();
                if (motion.comments) {
                    _.forEach(fields, function (field, id) {
                        motion['comment_' + id] = motion.comments[id];
                    });
                }
            },
            populateFieldsReverse: function (motion) {
                // Reverse equivalent to populateFields.
                var fields = this.getCommentsFields();
                motion.comments = {};
                _.forEach(fields, function (field, id) {
                    motion.comments[id] = motion['comment_' + id] || '';
                });
            },
            getFieldIdForFlag: function (flag) {
                var fields = this.getCommentsFields();
                return _.findKey(fields, [flag, true]);
            },
        };
    }
])

.filter('excludeSpecialCommentsFields', [
    'MotionComment',
    function (MotionComment) {
        return function (commentsFields) {
            var withoutSpecialCommentsFields = {};
            _.forEach(commentsFields, function (field, id) {
                if (!MotionComment.isSpecialCommentField(field)) {
                    withoutSpecialCommentsFields[id] = field;
                }
            });
            return withoutSpecialCommentsFields;
        };
    }
])

.filter('excludeDeletedAndForbiddenCommentsFields', [
    'MotionComment',
    'operator',
    function (MotionComment, operator) {
        return function (commentsFields) {
            var withoutDeletedAndForbiddenCommentsFields = {};
            _.forEach(commentsFields, function (field, id) {
                if (field && (field.public || operator.hasPerms('motions.can_see_comments'))) {
                    withoutDeletedAndForbiddenCommentsFields[id] = field;
                }
            });
            return withoutDeletedAndForbiddenCommentsFields;
        };
    }
])

.factory('Category', [
    'DS',
    function(DS) {
        return DS.defineResource({
            name: 'motions/category',
        });
    }
])

.factory('MotionChangeRecommendation', [
    'DS',
    'Config',
    'jsDataModel',
    'diffService',
    'lineNumberingService',
    'gettextCatalog',
    function (DS, Config, jsDataModel, diffService, lineNumberingService, gettextCatalog) {
        return DS.defineResource({
            name: 'motions/motion-change-recommendation',
            useClass: jsDataModel,
            methods: {
                saveStatus: function() {
                    this.DSSave();
                },
                isTitleRecommendation: function() {
                    return (this.line_from === 0 && this.line_to === 0);
                },
                isTextRecommendation: function() {
                    return (this.line_from !== 0 || this.line_to !== 0);
                },
                getDiff: function(motion, version, highlight) {
                    var lineLength = Config.get('motions_line_length').value,
                        html = lineNumberingService.insertLineNumbers(motion.getVersion(version).text, lineLength),
                        data, oldText;

                    try {
                        data = diffService.extractRangeByLineNumbers(html, this.line_from, this.line_to);
                        oldText = data.outerContextStart + data.innerContextStart +
                            data.html + data.innerContextEnd + data.outerContextEnd;
                    } catch (e) {
                        // This only happens (as far as we know) when the motion text has been altered (shortened)
                        // without modifying the change recommendations accordingly.
                        // That's a pretty serious inconsistency that should not happen at all,
                        // we're just doing some basic damage control here.
                        var msg = 'Inconsistent data. A change recommendation is probably referring to a non-existant line number.';
                        return '<em style="color: red; font-weight: bold;">' + msg + '</em>';
                    }
                    oldText = lineNumberingService.insertLineNumbers(oldText, lineLength, null, null, this.line_from);
                    var diff = diffService.diff(oldText, this.text);

                    // If an insertion makes the line longer than the line length limit, we need two line breaking runs:
                    // - First, for the official line numbers, ignoring insertions (that's been done some lines before)
                    // - Second, another one to prevent the displayed including insertions to exceed the page width
                    diff = lineNumberingService.insertLineBreaksWithoutNumbers(diff, lineLength, true);

                    if (highlight > 0) {
                        diff = lineNumberingService.highlightLine(diff, highlight);
                    }

                    var origBeginning = data.outerContextStart + data.innerContextStart;
                    if (diff.toLowerCase().indexOf(origBeginning.toLowerCase()) === 0) {
                        // Add "merge-before"-css-class if the first line begins in the middle of a paragraph. Used for PDF.
                        diff = diffService.addCSSClassToFirstTag(origBeginning, "merge-before") + diff.substring(origBeginning.length);
                    }

                    return diff;
                },
                getType: function(original_full_html) {
                    return this.type;
                },
                getTitle: function(original_full_html) {
                    var title;
                    if (this.line_to > (this.line_from + 1)) {
                        title = gettextCatalog.getString('%TYPE% from line %FROM% to %TO%');
                    } else {
                        title = gettextCatalog.getString('%TYPE% in line %FROM%');
                    }
                    switch (this.getType(original_full_html)) {
                        case diffService.TYPE_INSERTION:
                            title = title.replace('%TYPE%', gettextCatalog.getString('Insertion'));
                            break;
                        case diffService.TYPE_DELETION:
                            title = title.replace('%TYPE%', gettextCatalog.getString('Deletion'));
                            break;
                        case diffService.TYPE_REPLACEMENT:
                            title = title.replace('%TYPE%', gettextCatalog.getString('Replacement'));
                            break;
                        case diffService.TYPE_OTHER:
                            title = title.replace('%TYPE%', this.other_description);
                            break;
                    }
                    title = title.replace('%FROM%', this.line_from).replace('%TO%', (this.line_to - 1));
                    return title;
                }
            }
        });
    }
])

.run([
    'Motion',
    'Category',
    'Workflow',
    'MotionChangeRecommendation',
    function(Motion, Category, Workflow, MotionChangeRecommendation) {}
])


// Mark all motion workflow state strings for translation in JavaScript.
// (see motions/signals.py)
.config([
    'gettext',
    function (gettext) {
        // workflow 1
        gettext('Simple Workflow');
        gettext('submitted');
        gettext('accepted');
        gettext('Accept');
        gettext('Acceptance');
        gettext('rejected');
        gettext('Reject');
        gettext('Rejection');
        gettext('not decided');
        gettext('Do not decide');
        gettext('No decision');
        // workflow 2
        gettext('Complex Workflow');
        gettext('published');
        gettext('permitted');
        gettext('Permit');
        gettext('Permission');
        gettext('accepted');
        gettext('Accept');
        gettext('Acceptance');
        gettext('rejected');
        gettext('Reject');
        gettext('Rejection');
        gettext('withdrawed');
        gettext('Withdraw');
        gettext('adjourned');
        gettext('Adjourn');
        gettext('Adjournment');
        gettext('not concerned');
        gettext('Do not concern');
        gettext('No concernment');
        gettext('refered to committee');
        gettext('Refer to committee');
        gettext('Referral to committee');
        gettext('needs review');
        gettext('Needs review');
        gettext('rejected (not authorized)');
        gettext('Reject (not authorized)');
        gettext('Rejection (not authorized)');
    }
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.motions.csv', [])

.factory('MotionCsvExport', [
    'gettextCatalog',
    'Config',
    'CsvDownload',
    'lineNumberingService',
    function (gettextCatalog, Config, CsvDownload, lineNumberingService) {
        var makeHeaderline = function (params) {
            var headerline = ['Identifier', 'Title'];
            if (params.include.text) {
                headerline.push('Text');
            }
            if (params.include.reason) {
                headerline.push('Reason');
            }
            if (params.include.submitters) {
                headerline.push('Submitter');
            }
            headerline.push('Category');
            if (params.include.origin) {
                headerline.push('Origin');
            }
            if (params.include.motionBlock) {
                headerline.push('Motion block');
            }
            return _.map(headerline, function (entry) {
                return gettextCatalog.getString(entry);
            });
        };
        return {
            export: function (motions, params) {
                if (!params) {
                    params = {};
                }
                _.defaults(params, {
                    filename: gettextCatalog.getString('motions') + '.csv',
                    changeRecommendationMode: Config.get('motions_recommendation_text_mode').value,
                    include: {
                        text: true,
                        reason: true,
                        submitters: true,
                        origin: true,
                        motionBlock: true,
                        state: true,
                        recommendation: true,
                    },
                });
                if (!_.includes(['original', 'changed', 'agreed'], params.changeRecommendationMode)) {
                    params.changeRecommendationMode = 'original';
                }

                var csvRows = [
                    makeHeaderline(params)
                ];
                _.forEach(motions, function (motion) {
                    var text = motion.getTextByMode(params.changeRecommendationMode, null, null, false);
                    var row = [];
                    // Identifier and title
                    row.push('"' + motion.identifier !== null ? motion.identifier : '' + '"');
                    row.push('"' + motion.getTitle() + '"');

                    // Text
                    if (params.include.text) {
                        row.push('"' + text + '"');
                    }

                    // Reason
                    if (params.include.reason) {
                        row.push('"' + motion.getReason() + '"');
                    }

                    // Submitters
                    if (params.include.submitters) {
                        var submitters = [];
                        angular.forEach(motion.submitters, function(user) {
                            var user_short_name = [user.title, user.first_name, user.last_name].join(' ').trim();
                            submitters.push(user_short_name);
                        });
                        row.push('"' + submitters.join('; ') + '"');
                    }

                    // Category
                    var category = motion.category ? motion.category.name : '';
                    row.push('"' + category + '"');

                    // Origin
                    if (params.include.origin) {
                        row.push('"' + motion.origin + '"');
                    }

                    // Motion block
                    if (params.include.motionBlock) {
                        var blockTitle = motion.motionBlock ? motion.motionBlock.title : '';
                        row.push('"' + blockTitle + '"');
                    }

                    csvRows.push(row);
                });
                CsvDownload(csvRows, params.filename);
            },
            downloadExample: function () {
                var csvRows = [makeHeaderline({ include: {
                        text: true,
                        reason: true,
                        submitters: true,
                        origin: true,
                        motionBlock: true,
                        state: true,
                        recommendation: true,
                    }}),
                    // example entries
                    ['A1', 'Title 1', 'Text 1', 'Reason 1', 'Submitter A', 'Category A', 'Last Year Conference A', 'Block A'],
                    ['B1', 'Title 2', 'Text 2', 'Reason 2', 'Submitter B', 'Category B', '', 'Block A'],
                    [''  , 'Title 3', 'Text 3', '', '', '', '', ''],
                ];
                CsvDownload(csvRows, gettextCatalog.getString('motions-example') + '.csv');
            },
        };
    }
]);

}());

(function () {

"use strict";

angular.module('OpenSlidesApp.motions.diff', ['OpenSlidesApp.motions.lineNumbering'])

.service('diffService', [
    'lineNumberingService',
    '$cacheFactory',
    function (lineNumberingService, $cacheFactory) {
        var ELEMENT_NODE = 1,
            TEXT_NODE = 3,
            DOCUMENT_FRAGMENT_NODE = 11;

        var diffCache = $cacheFactory('diff.service');

        this.TYPE_REPLACEMENT = 0;
        this.TYPE_INSERTION = 1;
        this.TYPE_DELETION = 2;
        this.TYPE_OTHER = 3;

        this.getLineNumberNode = function(fragment, lineNumber) {
            return fragment.querySelector('os-linebreak.os-line-number.line-number-' + lineNumber);
        };

        this._getNodeContextTrace = function(node) {
            var context = [],
                currNode = node;
            while (currNode) {
                context.unshift(currNode);
                currNode = currNode.parentNode;
            }
            return context;
        };

        this._isFirstNonemptyChild = function(node, child) {
            for (var i = 0; i < node.childNodes.length; i++) {
                if (node.childNodes[i] === child) {
                    return true;
                }
                if (node.childNodes[i].nodeType !== TEXT_NODE || node.childNodes[i].nodeValue.match(/\S/)) {
                    return false;
                }
            }
            return false;
        };

        // Adds elements like <OS-LINEBREAK class="os-line-number line-number-23" data-line-number="23"/>
        this._insertInternalLineMarkers = function(fragment) {
            if (fragment.querySelectorAll('OS-LINEBREAK').length > 0) {
                // Prevent duplicate calls
                return;
            }
            var lineNumbers = fragment.querySelectorAll('span.os-line-number'),
                lineMarker, maxLineNumber;

            for (var i = 0; i < lineNumbers.length; i++) {
                var insertBefore = lineNumbers[i];
                while (insertBefore.parentNode.nodeType !== DOCUMENT_FRAGMENT_NODE &&
                       this._isFirstNonemptyChild(insertBefore.parentNode, insertBefore)) {
                    insertBefore = insertBefore.parentNode;
                }
                lineMarker = document.createElement('OS-LINEBREAK');
                lineMarker.setAttribute('data-line-number', lineNumbers[i].getAttribute('data-line-number'));
                lineMarker.setAttribute('class', lineNumbers[i].getAttribute('class'));
                insertBefore.parentNode.insertBefore(lineMarker, insertBefore);
                maxLineNumber = lineNumbers[i].getAttribute('data-line-number');
            }

            // Add one more "fake" line number at the end and beginning, so we can select the last line as well
            lineMarker = document.createElement('OS-LINEBREAK');
            lineMarker.setAttribute('data-line-number', (parseInt(maxLineNumber) + 1));
            lineMarker.setAttribute('class', 'os-line-number line-number-' + (parseInt(maxLineNumber) + 1));
            fragment.appendChild(lineMarker);

            lineMarker = document.createElement('OS-LINEBREAK');
            lineMarker.setAttribute('data-line-number', '0');
            lineMarker.setAttribute('class', 'os-line-number line-number-0');
            fragment.insertBefore(lineMarker, fragment.firstChild);
        };

        // @TODO Check if this is actually necessary
        this._insertInternalLiNumbers = function(fragment) {
            if (fragment.querySelectorAll('LI[os-li-number]').length > 0) {
                // Prevent duplicate calls
                return;
            }
            var ols = fragment.querySelectorAll('OL');
            for (var i = 0; i < ols.length; i++) {
                var ol = ols[i],
                    liNo = 0;
                for (var j = 0; j < ol.childNodes.length; j++) {
                    if (ol.childNodes[j].nodeName == 'LI') {
                        liNo++;
                        ol.childNodes[j].setAttribute('os-li-number', liNo);
                    }
                }
            }
        };

        this._addStartToOlIfNecessary = function(node) {
            var firstLiNo = null;
            for (var i = 0; i < node.childNodes.length && firstLiNo === null; i++) {
                if (node.childNode[i].nodeName == 'LI') {
                    var lineNo = node.childNode[i].getAttribute('ol-li-number');
                    if (lineNo) {
                        firstLiNo = parseInt(lineNo);
                    }
                }
            }
            if (firstLiNo > 1) {
                node.setAttribute('start', firstLiNo);
            }
        };

        this._isWithinNthLIOfOL = function(olNode, descendantNode) {
            var nthLIOfOL = null;
            while (descendantNode.parentNode) {
                if (descendantNode.parentNode === olNode) {
                    var lisBeforeOl = 0,
                        foundMe = false;
                    for (var i = 0; i < olNode.childNodes.length && !foundMe; i++) {
                        if (olNode.childNodes[i] === descendantNode) {
                            foundMe = true;
                        } else if (olNode.childNodes[i].nodeName === 'LI') {
                            lisBeforeOl++;
                        }
                    }
                    nthLIOfOL = lisBeforeOl + 1;
                }
                descendantNode = descendantNode.parentNode;
            }
            return nthLIOfOL;
        };

       /*
        * Returns an array with the following values:
        * 0: the most specific DOM-node that contains both line numbers
        * 1: the context of node1 (an array of dom-elements; 0 is the document fragment)
        * 2: the context of node2 (an array of dom-elements; 0 is the document fragment)
        * 3: the index of [0] in the two arrays
        */
        this._getCommonAncestor = function(node1, node2) {
            var trace1 = this._getNodeContextTrace(node1),
                trace2 = this._getNodeContextTrace(node2),
                commonAncestor = null,
                commonIndex = null,
                childTrace1 = [],
                childTrace2 = [];

            for (var i = 0; i < trace1.length && i < trace2.length; i++) {
                if (trace1[i] == trace2[i]) {
                    commonAncestor = trace1[i];
                    commonIndex = i;
                }
            }
            for (i = commonIndex + 1; i < trace1.length; i++) {
                childTrace1.push(trace1[i]);
            }
            for (i = commonIndex + 1; i < trace2.length; i++) {
                childTrace2.push(trace2[i]);
            }
            return {
                'commonAncestor': commonAncestor,
                'trace1' : childTrace1,
                'trace2' : childTrace2,
                'index': commonIndex
            };
        };

        this._serializeTag = function(node) {
            if (node.nodeType == DOCUMENT_FRAGMENT_NODE) {
                // Fragments are only placeholders and do not have an HTML representation
                return '';
            }
            var html = '<' + node.nodeName;
            for (var i = 0; i < node.attributes.length; i++) {
                var attr = node.attributes[i];
                if (attr.name != 'os-li-number') {
                    html += ' ' + attr.name + '="' + attr.value + '"';
                }
            }
            html += '>';
            return html;
        };

        this._serializeDom = function(node, stripLineNumbers) {
            if (node.nodeType === TEXT_NODE) {
                return node.nodeValue.replace(/</g, "&lt;").replace(/>/g, "&gt;");
            }
            if (stripLineNumbers && (
                lineNumberingService._isOsLineNumberNode(node) || lineNumberingService._isOsLineBreakNode(node))) {
                return '';
            }
            if (node.nodeName === 'OS-LINEBREAK') {
                return '';
            }
            if (node.nodeName === 'BR') {
                var br = '<BR';
                for (i = 0; i < node.attributes.length; i++) {
                    var attr = node.attributes[i];
                    br += " " + attr.name + "=\"" + attr.value + "\"";
                }
                return br + '>';
            }

            var html = this._serializeTag(node);
            for (var i = 0; i < node.childNodes.length; i++) {
                if (node.childNodes[i].nodeType === TEXT_NODE) {
                    html += node.childNodes[i].nodeValue.replace(/&/g, "&amp;").replace(/</g, "&lt;").replace(/>/g, "&gt;");
                } else if (!stripLineNumbers || (!lineNumberingService._isOsLineNumberNode(node.childNodes[i]) && !lineNumberingService._isOsLineBreakNode(node.childNodes[i]))) {
                    html += this._serializeDom(node.childNodes[i], stripLineNumbers);
                }
            }
            if (node.nodeType !== DOCUMENT_FRAGMENT_NODE) {
                html += '</' + node.nodeName + '>';
            }

            return html;
        };

        /**
         * Implementation hint: the first element of "toChildTrace" array needs to be a child element of "node"
         */
        this._serializePartialDomToChild = function(node, toChildTrace, stripLineNumbers) {
            if (lineNumberingService._isOsLineNumberNode(node) || lineNumberingService._isOsLineBreakNode(node)) {
                return '';
            }
            if (node.nodeName == 'OS-LINEBREAK') {
                return '';
            }

            var html = this._serializeTag(node);

            for (var i = 0, found = false; i < node.childNodes.length && !found; i++) {
                if (node.childNodes[i] == toChildTrace[0]) {
                    found = true;
                    var remainingTrace = toChildTrace;
                    remainingTrace.shift();
                    if (!lineNumberingService._isOsLineNumberNode(node.childNodes[i])) {
                        html += this._serializePartialDomToChild(node.childNodes[i], remainingTrace, stripLineNumbers);
                    }
                } else if (node.childNodes[i].nodeType == TEXT_NODE) {
                    html += node.childNodes[i].nodeValue;
                } else {
                    if (!stripLineNumbers || (!lineNumberingService._isOsLineNumberNode(node.childNodes[i]) &&
                      !lineNumberingService._isOsLineBreakNode(node.childNodes[i]))) {
                        html += this._serializeDom(node.childNodes[i], stripLineNumbers);
                    }
                }
            }
            if (!found) {
                console.trace();
                throw "Inconsistency or invalid call of this function detected (to)";
            }
            return html;
        };

        /**
         * Implementation hint: the first element of "toChildTrace" array needs to be a child element of "node"
         */
        this._serializePartialDomFromChild = function(node, fromChildTrace, stripLineNumbers) {
            if (lineNumberingService._isOsLineNumberNode(node) || lineNumberingService._isOsLineBreakNode(node)) {
                return '';
            }
            if (node.nodeName == 'OS-LINEBREAK') {
                return '';
            }

            var html = '';
            for (var i = 0, found = false; i < node.childNodes.length; i++) {
                if (node.childNodes[i] == fromChildTrace[0]) {
                    found = true;
                    var remainingTrace = fromChildTrace;
                    remainingTrace.shift();
                    if (!lineNumberingService._isOsLineNumberNode(node.childNodes[i])) {
                        html += this._serializePartialDomFromChild(node.childNodes[i], remainingTrace, stripLineNumbers);
                    }
                } else if (found) {
                    if (node.childNodes[i].nodeType == TEXT_NODE) {
                        html += node.childNodes[i].nodeValue;
                    } else {
                        if (!stripLineNumbers || (!lineNumberingService._isOsLineNumberNode(node.childNodes[i]) &&
                          !lineNumberingService._isOsLineBreakNode(node.childNodes[i]))) {
                            html += this._serializeDom(node.childNodes[i], stripLineNumbers);
                        }
                    }
                }
            }
            if (!found) {
                console.trace();
                throw "Inconsistency or invalid call of this function detected (from)";
            }
            if (node.nodeType != DOCUMENT_FRAGMENT_NODE) {
                html += '</' + node.nodeName + '>';
            }
            return html;
        };

        this.htmlToFragment = function(html) {
            var fragment = document.createDocumentFragment(),
                div = document.createElement('DIV');
            div.innerHTML = html;
            while (div.childElementCount) {
                var child = div.childNodes[0];
                div.removeChild(child);
                fragment.appendChild(child);
            }
            return fragment;
        };

        /**
         * When a <li> with a os-split-before-class (set by extractRangeByLineNumbers) is edited when creating a
         * change recommendation and is split again in CKEditor, the second list items also gets that class.
         * This is not correct however, as the second one actually is a new list item. So we need to remove it again.
         *
         * @param {string} html
         * @returns {string}
         */
        this.removeDuplicateClassesInsertedByCkeditor = function(html) {
            var fragment = this.htmlToFragment(html);
            var items = fragment.querySelectorAll('li.os-split-before');
            for (var i = 0; i < items.length; i++) {
                if (!this._isFirstNonemptyChild(items[i].parentNode, items[i])) {
                    this.removeCSSClass(items[i], 'os-split-before');
                }
            }
            return this._serializeDom(fragment, false);
        };

        /**
         * Returns the HTML snippet between two given line numbers.
         *
         * Hint:
         * - The last line (toLine) is not included anymore, as the number refers to the line breaking element
         * - if toLine === null, then everything from fromLine to the end of the fragment is returned
         *
         * In addition to the HTML snippet, additional information is provided regarding the most specific DOM element
         * that contains the whole section specified by the line numbers (like a P-element if only one paragraph is selected
         * or the most outer DIV, if multiple sections selected).
         *
         * This additional information is meant to render the snippet correctly without producing broken HTML
         *
         * The return object has the following fields:
         * - html: The HTML between the two line numbers.
         *         Line numbers and automatically set line breaks are stripped.
         *         All HTML tags are converted to uppercase
         *         (e.g. Line 2</LI><LI>Line3</LI><LI>Line 4 <br>)
         * - ancestor: the most specific DOM element that contains the HTML snippet (e.g. a UL, if several LIs are selected)
         * - outerContextStart: An HTML string that opens all necessary tags to get the browser into the rendering mode
         *                      of the ancestor element (e.g. <DIV><UL> in the case of the multiple LIs)
         * - outerContectEnd:   An HTML string that closes all necessary tags from the ancestor element (e.g. </UL></DIV>
         * - innerContextStart: A string that opens all necessary tags between the ancestor
         *                      and the beginning of the selection (e.g. <LI>)
         * - innerContextEnd:   A string that closes all tags after the end of the selection to the ancestor (e.g. </LI>)
         * - previousHtml:      The HTML before the selected area begins (including line numbers)
         * - previousHtmlEndSnippet: A HTML snippet that closes all open tags from previousHtml
         * - followingHtml:     The HTML after the selected area
         * - followingHtmlStartSnippet: A HTML snippet that opens all HTML tags necessary to render "followingHtml"
         *
         *
         * In some cases, the returned HTML tags receive additional CSS classes, providing information both for
         * rendering it and for merging it again correctly.
         * - os-split-*:        These classes are set for all HTML Tags that have been split into two by this process,
         *                      e.g. if the fromLine- or toLine-line-break was somewhere in the middle of this tag.
         *                      If a tag is split, the first one receives "os-split-after", and the second one "os-split-before".
         * For example, for the following string <p>Line 1<br>Line 2<br>Line 3</p>:
         * - extracting line 1 to 2 results in <p class="os-split-after">Line 1</p>
         * - extracting line 2 to 3 results in <p class="os-split-after os-split-before">Line 2</p>
         * - extracting line 3 to null/4 results in <p class="os-split-before">Line 3</p>
         */
        this.extractRangeByLineNumbers = function(htmlIn, fromLine, toLine) {
            if (typeof(htmlIn) !== 'string') {
                throw 'Invalid call - extractRangeByLineNumbers expects a string as first argument';
            }

            var cacheKey = fromLine + "-" + toLine + "-" + lineNumberingService.djb2hash(htmlIn),
                cached = diffCache.get(cacheKey);

            if (!angular.isUndefined(cached)) {
                return cached;
            }

            var fragment = this.htmlToFragment(htmlIn);

            this._insertInternalLineMarkers(fragment);
            this._insertInternalLiNumbers(fragment);
            if (toLine === null) {
                var internalLineMarkers = fragment.querySelectorAll('OS-LINEBREAK');
                toLine = parseInt(internalLineMarkers[internalLineMarkers.length - 1].getAttribute("data-line-number"));
            }

            var fromLineNode = this.getLineNumberNode(fragment, fromLine),
                toLineNode = (toLine ? this.getLineNumberNode(fragment, toLine) : null),
                ancestorData = this._getCommonAncestor(fromLineNode, toLineNode);

            var fromChildTraceRel = ancestorData.trace1,
                fromChildTraceAbs = this._getNodeContextTrace(fromLineNode),
                toChildTraceRel = ancestorData.trace2,
                toChildTraceAbs = this._getNodeContextTrace(toLineNode),
                ancestor = ancestorData.commonAncestor,
                htmlOut = '',
                outerContextStart = '',
                outerContextEnd = '',
                innerContextStart = '',
                innerContextEnd = '',
                previousHtmlEndSnippet = '',
                followingHtmlStartSnippet = '',
                fakeOl, offset;


            fromChildTraceAbs.shift();
            var previousHtml = this._serializePartialDomToChild(fragment, fromChildTraceAbs, false);
            toChildTraceAbs.shift();
            var followingHtml = this._serializePartialDomFromChild(fragment, toChildTraceAbs, false);

            var currNode = fromLineNode,
                isSplit = false;
            while (currNode.parentNode) {
                if (!this._isFirstNonemptyChild(currNode.parentNode, currNode)) {
                    isSplit = true;
                }
                if (isSplit) {
                    this.addCSSClass(currNode.parentNode, 'os-split-before');
                }
                if (currNode.nodeName !== 'OS-LINEBREAK') {
                    previousHtmlEndSnippet += '</' + currNode.nodeName + '>';
                }
                currNode = currNode.parentNode;
            }

            currNode = toLineNode;
            isSplit = false;
            while (currNode.parentNode) {
                if (!this._isFirstNonemptyChild(currNode.parentNode, currNode)) {
                    isSplit = true;
                }
                if (isSplit) {
                    this.addCSSClass(currNode.parentNode, 'os-split-after');
                }
                if (currNode.parentNode.nodeName === 'OL') {
                    fakeOl = currNode.parentNode.cloneNode(false);
                    offset = (currNode.parentNode.getAttribute("start") ? parseInt(currNode.parentNode.getAttribute("start")) - 1 : 0);
                    fakeOl.setAttribute('start', (this._isWithinNthLIOfOL(currNode.parentNode, toLineNode) + offset).toString());
                    followingHtmlStartSnippet = this._serializeTag(fakeOl) + followingHtmlStartSnippet;
                } else {
                    followingHtmlStartSnippet = this._serializeTag(currNode.parentNode) + followingHtmlStartSnippet;
                }
                currNode = currNode.parentNode;
            }

            var found = false;
            isSplit = false;
            for (var i = 0; i < fromChildTraceRel.length && !found; i++) {
                if (fromChildTraceRel[i].nodeName === 'OS-LINEBREAK') {
                    found = true;
                } else {
                    if (!this._isFirstNonemptyChild(fromChildTraceRel[i], fromChildTraceRel[i + 1])) {
                        isSplit = true;
                    }
                    if (fromChildTraceRel[i].nodeName === 'OL') {
                        fakeOl = fromChildTraceRel[i].cloneNode(false);
                        offset = (fromChildTraceRel[i].getAttribute("start") ? parseInt(fromChildTraceRel[i].getAttribute("start")) - 1 : 0);
                        fakeOl.setAttribute('start', (offset + this._isWithinNthLIOfOL(fromChildTraceRel[i], fromLineNode)).toString());
                        innerContextStart += this._serializeTag(fakeOl);
                    } else {
                        if (i < (fromChildTraceRel.length - 1) && isSplit) {
                            this.addCSSClass(fromChildTraceRel[i], 'os-split-before');
                        }
                        innerContextStart += this._serializeTag(fromChildTraceRel[i]);
                    }
                }
            }
            found = false;
            for (i = 0; i < toChildTraceRel.length && !found; i++) {
                if (toChildTraceRel[i].nodeName === 'OS-LINEBREAK') {
                    found = true;
                } else {
                    innerContextEnd = '</' + toChildTraceRel[i].nodeName + '>' + innerContextEnd;
                }
            }

            found = false;
            for (i = 0; i < ancestor.childNodes.length; i++) {
                if (ancestor.childNodes[i] === fromChildTraceRel[0]) {
                    found = true;
                    fromChildTraceRel.shift();
                    htmlOut += this._serializePartialDomFromChild(ancestor.childNodes[i], fromChildTraceRel, true);
                } else if (ancestor.childNodes[i] === toChildTraceRel[0]) {
                    found = false;
                    toChildTraceRel.shift();
                    htmlOut += this._serializePartialDomToChild(ancestor.childNodes[i], toChildTraceRel, true);
                } else if (found === true) {
                    htmlOut += this._serializeDom(ancestor.childNodes[i], true);
                }
            }

            currNode = ancestor;
            while (currNode.parentNode) {
                if (currNode.nodeName === 'OL') {
                    fakeOl = currNode.cloneNode(false);
                    offset = (currNode.getAttribute("start") ? parseInt(currNode.getAttribute("start")) - 1 : 0);
                    fakeOl.setAttribute('start', (this._isWithinNthLIOfOL(currNode, fromLineNode) + offset).toString());
                    outerContextStart = this._serializeTag(fakeOl) + outerContextStart;
                } else {
                    outerContextStart = this._serializeTag(currNode) + outerContextStart;
                }
                outerContextEnd += '</' + currNode.nodeName + '>';
                currNode = currNode.parentNode;
            }

            var ret = {
                'html': htmlOut,
                'ancestor': ancestor,
                'outerContextStart': outerContextStart,
                'outerContextEnd': outerContextEnd,
                'innerContextStart': innerContextStart,
                'innerContextEnd': innerContextEnd,
                'previousHtml': previousHtml,
                'previousHtmlEndSnippet': previousHtmlEndSnippet,
                'followingHtml': followingHtml,
                'followingHtmlStartSnippet': followingHtmlStartSnippet
            };

            diffCache.put(cacheKey, ret);
            return ret;
        };

        /*
         * This is a workardoun to prevent the last word of the inserted text from accidently being merged with the
         * first word of the following line.
         *
         * This happens as trailing spaces in the change recommendation's text are frequently stripped,
         * which is pretty nasty if the original text goes on after the affected line. So we insert a space
         * if the original line ends with one.
         */
        this._insertDanglingSpace = function(element) {
            if (element.childNodes.length > 0) {
                var lastChild = element.childNodes[element.childNodes.length - 1];
                if (lastChild.nodeType == TEXT_NODE && !lastChild.nodeValue.match(/[\S]/) && element.childNodes.length > 1) {
                    // If the text node only contains whitespaces, chances are high it's just space between block elmeents,
                    // like a line break between </LI> and </UL>
                    lastChild = element.childNodes[element.childNodes.length - 2];
                }
                if (lastChild.nodeType == TEXT_NODE) {
                    if (lastChild.nodeValue === '' || lastChild.nodeValue.substr(-1) != ' ') {
                        lastChild.nodeValue += ' ';
                    }
                } else {
                    this._insertDanglingSpace(lastChild);
                }
            }
        };

        /*
         * This functions merges to arrays of nodes. The last element of nodes1 and the first element of nodes2
         * are merged, if they are of the same type.
         *
         * This is done recursively until a TEMPLATE-Tag is is found, which was inserted in this.replaceLines.
         * Using a TEMPLATE-Tag is a rather dirty hack, as it is allowed inside of any other element, including <ul>.
         *
         */
        this._replaceLinesMergeNodeArrays = function(nodes1, nodes2) {
            if (nodes1.length === 0) {
                return nodes2;
            }
            if (nodes2.length === 0) {
                return nodes1;
            }

            var out = [];
            for (var i = 0; i < nodes1.length - 1; i++) {
                out.push(nodes1[i]);
            }

            var lastNode = nodes1[nodes1.length - 1],
                firstNode = nodes2[0];
            if (lastNode.nodeType === TEXT_NODE && firstNode.nodeType === TEXT_NODE) {
                var newTextNode = lastNode.ownerDocument.createTextNode(lastNode.nodeValue + firstNode.nodeValue);
                out.push(newTextNode);
            } else if (lastNode.nodeName === firstNode.nodeName) {
                var newNode = lastNode.ownerDocument.createElement(lastNode.nodeName);
                for (i = 0; i < lastNode.attributes.length; i++) {
                    var attr = lastNode.attributes[i];
                    newNode.setAttribute(attr.name, attr.value);
                }

                // Remove #text nodes inside of List elements (OL/UL), as they are confusing
                var lastChildren, firstChildren;
                if (lastNode.nodeName === 'OL' || lastNode.nodeName === 'UL') {
                    lastChildren = [];
                    firstChildren = [];
                    for (i = 0; i < firstNode.childNodes.length; i++) {
                        if (firstNode.childNodes[i].nodeType === ELEMENT_NODE) {
                            firstChildren.push(firstNode.childNodes[i]);
                        }
                    }
                    for (i = 0; i < lastNode.childNodes.length; i++) {
                        if (lastNode.childNodes[i].nodeType === ELEMENT_NODE) {
                            lastChildren.push(lastNode.childNodes[i]);
                        }
                    }
                } else {
                    lastChildren = lastNode.childNodes;
                    firstChildren = firstNode.childNodes;
                }

                var children = this._replaceLinesMergeNodeArrays(lastChildren, firstChildren);
                for (i = 0; i < children.length; i++) {
                    newNode.appendChild(children[i]);
                }

                out.push(newNode);
            } else {
                if (lastNode.nodeName !== 'TEMPLATE') {
                    out.push(lastNode);
                }
                if (firstNode.nodeName !== 'TEMPLATE') {
                    out.push(firstNode);
                }
            }

            for (i = 1; i < nodes2.length; i++) {
                out.push(nodes2[i]);
            }

            return out;
        };

        /**
         *
         * @param {string} html
         * @returns {string}
         * @private
         */
        this._normalizeHtmlForDiff = function (html) {
            // Convert all HTML tags to uppercase, but leave the values of attributes unchanged
            // All attributes and CSS class names  are sorted alphabetically
            // If an attribute is empty, it is removed
            html = html.replace(/<(\/?[a-z]*)( [^>]*)?>/ig, function (html, tag, attributes) {
                var tagNormalized = tag.toUpperCase();
                if (attributes === undefined) {
                    attributes = "";
                }
                var attributesList = [],
                    attributesMatcher = /( [^"'=]*)(= *((["'])(.*?)\4))?/gi,
                    match;
                do {
                    match = attributesMatcher.exec(attributes);
                    if (match) {
                        var attrNormalized = match[1].toUpperCase(),
                            attrValue = match[5];
                        if (match[2] !== undefined) {
                            if (attrNormalized === ' CLASS') {
                                attrValue = attrValue.split(' ').sort().join(' ').replace(/^\s+/, '').replace(/\s+$/, '');
                            }
                            attrNormalized += "=" + match[4] + attrValue + match[4];
                        }
                        if (attrValue !== '') {
                            attributesList.push(attrNormalized);
                        }
                    }
                } while (match);
                attributes = attributesList.sort().join('');
                return "<" + tagNormalized + attributes + ">";
            });

            var entities = {
                '&nbsp;': ' ',
                '&ndash;': '-',
                '&auml;': 'ä',
                '&ouml;': 'ö',
                '&uuml;': 'ü',
                '&Auml;': 'Ä',
                '&Ouml;': 'Ö',
                '&Uuml;': 'Ü',
                '&szlig;': 'ß'
            };

            html = html.replace(/\s+<\/P>/gi, '</P>').replace(/\s+<\/DIV>/gi, '</DIV>').replace(/\s+<\/LI>/gi, '</LI>');
            html = html.replace(/\s+<LI>/gi, '<LI>').replace(/<\/LI>\s+/gi, '</LI>');
            html = html.replace(/\u00A0/g, ' ');
            html = html.replace(/\u2013/g, '-');
            for (var ent in entities) {
                html = html.replace(new RegExp(ent, 'g'), entities[ent]);
            }

            // Newline characters: after closing block-level-elements, but not after BR (which is inline)
            html = html.replace(/(<br *\/?>)\n/gi, "$1");
            html = html.replace(/[ \n\t]+/gi, ' ');
            html = html.replace(/(<\/(div|p|ul|li|blockquote>)>) /gi, "$1\n");

            return html;
        };

        /**
         * @param {string} htmlOld
         * @param {string} htmlNew
         * @returns {number}
         */
        this.detectReplacementType = function (htmlOld, htmlNew) {
            htmlOld = this._normalizeHtmlForDiff(htmlOld);
            htmlNew = this._normalizeHtmlForDiff(htmlNew);

            if (htmlOld == htmlNew) {
                return this.TYPE_REPLACEMENT;
            }

            var i, foundDiff;
            for (i = 0, foundDiff = false; i < htmlOld.length && i < htmlNew.length && foundDiff === false; i++) {
                if (htmlOld[i] != htmlNew[i]) {
                    foundDiff = true;
                }
            }

            var remainderOld = htmlOld.substr(i - 1),
                remainderNew = htmlNew.substr(i - 1),
                type = this.TYPE_REPLACEMENT;

            if (remainderOld.length > remainderNew.length) {
                if (remainderOld.substr(remainderOld.length - remainderNew.length) == remainderNew) {
                    type = this.TYPE_DELETION;
                }
            } else if (remainderOld.length < remainderNew.length) {
                if (remainderNew.substr(remainderNew.length - remainderOld.length) == remainderOld) {
                    type = this.TYPE_INSERTION;
                }
            }

            return type;
        };

        /**
         * @param {string} oldHtml
         * @param {string} newHTML
         * @param {number} fromLine
         * @param {number} toLine
         */
        this.replaceLines = function (oldHtml, newHTML, fromLine, toLine) {
            var data = this.extractRangeByLineNumbers(oldHtml, fromLine, toLine),
                previousHtml = data.previousHtml + '<TEMPLATE></TEMPLATE>' + data.previousHtmlEndSnippet,
                previousFragment = this.htmlToFragment(previousHtml),
                followingHtml = data.followingHtmlStartSnippet + '<TEMPLATE></TEMPLATE>' + data.followingHtml,
                followingFragment = this.htmlToFragment(followingHtml),
                newFragment = this.htmlToFragment(newHTML);

            if (data.html.length > 0 && data.html.substr(-1) === ' ') {
                this._insertDanglingSpace(newFragment);
            }

            var merged = this._replaceLinesMergeNodeArrays(previousFragment.childNodes, newFragment.childNodes);
            merged = this._replaceLinesMergeNodeArrays(merged, followingFragment.childNodes);

            var mergedFragment = document.createDocumentFragment();
            for (var i = 0; i < merged.length; i++) {
                mergedFragment.appendChild(merged[i]);
            }

            var forgottenTemplates = mergedFragment.querySelectorAll("TEMPLATE");
            for (i = 0; i < forgottenTemplates.length; i++) {
                var el = forgottenTemplates[i];
                el.parentNode.removeChild(el);
            }

            var forgottenSplitClasses = mergedFragment.querySelectorAll(".os-split-before, .os-split-after");
            for (i = 0; i < forgottenSplitClasses.length; i++) {
                this.removeCSSClass(forgottenSplitClasses[i], 'os-split-before');
                this.removeCSSClass(forgottenSplitClasses[i], 'os-split-after');
            }

            return this._serializeDom(mergedFragment, true);
        };

        this.addCSSClass = function (node, className) {
            if (node.nodeType !== ELEMENT_NODE) {
                return;
            }
            var classes = node.getAttribute('class');
            classes = (classes ? classes.split(' ') : []);
            if (classes.indexOf(className) === -1) {
                classes.push(className);
            }
            node.setAttribute('class', classes.join(' '));
        };

        this.removeCSSClass = function (node, className) {
            if (node.nodeType !== ELEMENT_NODE) {
                return;
            }
            var classes = node.getAttribute('class'),
                newClasses = [];
            classes = (classes ? classes.split(' ') : []);
            for (var i = 0; i < classes.length; i++) {
                if (classes[i] !== className) {
                    newClasses.push(classes[i]);
                }
            }
            if (newClasses.length === 0) {
                node.removeAttribute('class');
            } else {
                node.setAttribute('class', newClasses.join(' '));
            }
        };

        this.addDiffMarkup = function (originalHTML, newHTML, fromLine, toLine, diffFormatterCb) {
            var data = this.extractRangeByLineNumbers(originalHTML, fromLine, toLine),
                previousHtml = data.previousHtml + '<TEMPLATE></TEMPLATE>' + data.previousHtmlEndSnippet,
                previousFragment = this.htmlToFragment(previousHtml),
                followingHtml = data.followingHtmlStartSnippet + '<TEMPLATE></TEMPLATE>' + data.followingHtml,
                followingFragment = this.htmlToFragment(followingHtml),
                newFragment = this.htmlToFragment(newHTML),
                oldHTML = data.outerContextStart + data.innerContextStart + data.html +
                    data.innerContextEnd + data.outerContextEnd,
                oldFragment = this.htmlToFragment(oldHTML),
                el;

            var diffFragment = diffFormatterCb(oldFragment, newFragment);

            var mergedFragment = document.createDocumentFragment();
            while (previousFragment.firstChild) {
                el = previousFragment.firstChild;
                previousFragment.removeChild(el);
                mergedFragment.appendChild(el);
            }
            while (diffFragment.firstChild) {
                el = diffFragment.firstChild;
                diffFragment.removeChild(el);
                mergedFragment.appendChild(el);
            }
            while (followingFragment.firstChild) {
                el = followingFragment.firstChild;
                followingFragment.removeChild(el);
                mergedFragment.appendChild(el);
            }

            var forgottenTemplates = mergedFragment.querySelectorAll("TEMPLATE");
            for (var i = 0; i < forgottenTemplates.length; i++) {
                el = forgottenTemplates[i];
                el.parentNode.removeChild(el);
            }

            return this._serializeDom(mergedFragment, true);
        };

        /**
         * Adapted from http://ejohn.org/projects/javascript-diff-algorithm/
         * by John Resig, MIT License
         * @param {array} oldArr
         * @param {array} newArr
         * @returns {object}
         */
        this._diff = function (oldArr, newArr) {
            var ns = {},
                os = {},
                i;

            for (i = 0; i < newArr.length; i++) {
                if (ns[newArr[i]] === undefined)
                    ns[newArr[i]] = {rows: [], o: null};
                ns[newArr[i]].rows.push(i);
            }

            for (i = 0; i < oldArr.length; i++) {
                if (os[oldArr[i]] === undefined)
                    os[oldArr[i]] = {rows: [], n: null};
                os[oldArr[i]].rows.push(i);
            }

            for (i in ns) {
                if (ns[i].rows.length == 1 && typeof(os[i]) != "undefined" && os[i].rows.length == 1) {
                    newArr[ns[i].rows[0]] = {text: newArr[ns[i].rows[0]], row: os[i].rows[0]};
                    oldArr[os[i].rows[0]] = {text: oldArr[os[i].rows[0]], row: ns[i].rows[0]};
                }
            }

            for (i = 0; i < newArr.length - 1; i++) {
                if (newArr[i].text !== null && newArr[i + 1].text === undefined && newArr[i].row + 1 < oldArr.length &&
                    oldArr[newArr[i].row + 1].text === undefined && newArr[i + 1] == oldArr[newArr[i].row + 1]) {
                    newArr[i + 1] = {text: newArr[i + 1], row: newArr[i].row + 1};
                    oldArr[newArr[i].row + 1] = {text: oldArr[newArr[i].row + 1], row: i + 1};
                }
            }

            for (i = newArr.length - 1; i > 0; i--) {
                if (newArr[i].text !== null && newArr[i - 1].text === undefined && newArr[i].row > 0 &&
                    oldArr[newArr[i].row - 1].text === undefined && newArr[i - 1] == oldArr[newArr[i].row - 1]) {
                    newArr[i - 1] = {text: newArr[i - 1], row: newArr[i].row - 1};
                    oldArr[newArr[i].row - 1] = {text: oldArr[newArr[i].row - 1], row: i - 1};
                }
            }

            return {o: oldArr, n: newArr};
        };

        this._tokenizeHtml = function (str) {
            var splitArrayEntriesEmbedSeparator = function (arr, by, prepend) {
                var newArr = [];
                for (var i = 0; i < arr.length; i++) {
                    if (arr[i][0] === '<' && (by === " " || by === "\n")) {
                        // Don't split HTML tags
                        newArr.push(arr[i]);
                        continue;
                    }

                    var parts = arr[i].split(by);
                    if (parts.length === 1) {
                        newArr.push(arr[i]);
                    } else {
                        var j;
                        if (prepend) {
                            if (parts[0] !== '') {
                                newArr.push(parts[0]);
                            }
                            for (j = 1; j < parts.length; j++) {
                                newArr.push(by + parts[j]);
                            }
                        } else {
                            for (j = 0; j < parts.length - 1; j++) {
                                newArr.push(parts[j] + by);
                            }
                            if (parts[parts.length - 1] !== '') {
                                newArr.push(parts[parts.length - 1]);
                            }
                        }
                    }
                }
                return newArr;
            };
            var splitArrayEntriesSplitSeparator = function (arr, by) {
                var newArr = [];
                for (var i = 0; i < arr.length; i++) {
                    if (arr[i][0] === '<') {
                        newArr.push(arr[i]);
                        continue;
                    }
                    var parts = arr[i].split(by);
                    for (var j = 0; j < parts.length; j++) {
                        if (j > 0) {
                            newArr.push(by);
                        }
                        newArr.push(parts[j]);
                    }
                }
                return newArr;
            };
            var arr = splitArrayEntriesEmbedSeparator([str], '<', true);
            arr = splitArrayEntriesEmbedSeparator(arr, '>', false);
            arr = splitArrayEntriesSplitSeparator(arr, " ");
            arr = splitArrayEntriesSplitSeparator(arr, ".");
            arr = splitArrayEntriesSplitSeparator(arr, ",");
            arr = splitArrayEntriesSplitSeparator(arr, "!");
            arr = splitArrayEntriesSplitSeparator(arr, "-");
            arr = splitArrayEntriesEmbedSeparator(arr, "\n", false);

            var arrWithoutEmptes = [];
            for (var i = 0; i < arr.length; i++) {
                if (arr[i] !== '') {
                    arrWithoutEmptes.push(arr[i]);
                }
            }

            return arrWithoutEmptes;
        };

        /**
         * @param {string} oldStr
         * @param {string} newStr
         * @returns {string}
         */
        this._diffString = function (oldStr, newStr) {
            oldStr = this._normalizeHtmlForDiff(oldStr.replace(/\s+$/, '').replace(/^\s+/, ''));
            newStr = this._normalizeHtmlForDiff(newStr.replace(/\s+$/, '').replace(/^\s+/, ''));

            var out = this._diff(this._tokenizeHtml(oldStr), this._tokenizeHtml(newStr));

            // This fixes the problem tested by "does not lose words when changes are moved X-wise"
            var lastRow = 0;
            for (var z = 0; z < out.n.length; z++) {
                if (out.n[z].row && out.n[z].row > lastRow) {
                    lastRow = out.n[z].row;
                }
                if (out.n[z].row && out.n[z].row < lastRow) {
                    out.o[out.n[z].row] = out.o[out.n[z].row].text;
                    out.n[z] = out.n[z].text;
                }
            }

            var str = "";
            var i;

            if (out.n.length === 0) {
                for (i = 0; i < out.o.length; i++) {
                    str += '<del>' + out.o[i] + "</del>";
                }
            } else {
                if (out.n[0].text === undefined) {
                    for (var k = 0; k < out.o.length && out.o[k].text === undefined; k++) {
                        str += '<del>' + out.o[k] + "</del>";
                    }
                }

                var currOldRow = 0;
                for (i = 0; i < out.n.length; i++) {
                    if (out.n[i].text === undefined) {
                        if (out.n[i] !== "") {
                            str += '<ins>' + out.n[i] + "</ins>";
                        }
                    } else if (out.n[i].row < currOldRow) {
                        str += '<ins>' + out.n[i].text + "</ins>";
                    } else {
                        var pre = "";

                        if ((i + 1) < out.n.length && out.n[i + 1].row !== undefined && out.n[i + 1].row > out.n[i].row + 1) {
                            for (var n = out.n[i].row + 1; n < out.n[i + 1].row; n++) {
                                if (out.o[n].text === undefined) {
                                    pre += '<del>' + out.o[n] + "</del>";
                                } else {
                                    pre += '<del>' + out.o[n].text + "</del>";
                                }
                            }
                        } else {
                            for (var j = out.n[i].row + 1; j < out.o.length && out.o[j].text === undefined; j++) {
                                pre += '<del>' + out.o[j] + "</del>";
                            }
                        }
                        str += out.n[i].text + pre;

                        currOldRow = out.n[i].row;
                    }
                }
            }

            return str.replace(/^\s+/g, '').replace(/\s+$/g, '').replace(/ {2,}/g, ' ');
        };

        /**
         *
         * @param {string} html
         * @returns {boolean}
         * @private
         */
        this._diffDetectBrokenDiffHtml = function(html) {
            // If a regular HTML tag is enclosed by INS/DEL, the HTML is broken
            var match = html.match(/<(ins|del)><[^>]*><\/(ins|del)>/gi);
            if (match !== null && match.length > 0) {
                return true;
            }

            // Opening tags, followed by </del> or </ins>, indicate broken HTML (if it's not a <ins> / <del>)
            var brokenRegexp = /<(\w+)[^>]*><\/(ins|del)>/gi,
                result;
            while ((result = brokenRegexp.exec(html)) !== null) {
                if (result[1].toLowerCase() !== 'ins' && result[1].toLowerCase() !== 'del') {
                    return true;
                }
            }


            // If other HTML tags are contained within INS/DEL (e.g. "<ins>Test</p></ins>"), let's better be cautious
            // The "!!(found=...)"-construction is only used to make jshint happy :)
            var findDel = /<del>(.*?)<\/del>/gi,
                findIns = /<ins>(.*?)<\/ins>/gi,
                found, inner;
            while (!!(found = findDel.exec(html))) {
                inner = found[1].replace(/<br[^>]*>/gi, '');
                if (inner.match(/<[^>]*>/)) {
                    return true;
                }
            }
            while (!!(found = findIns.exec(html))) {
                inner = found[1].replace(/<br[^>]*>/gi, '');
                if (inner.match(/<[^>]*>/)) {
                    return true;
                }
            }

            // If non of the conditions up to now is met, we consider the diff as being sane
            return false;
        };

        this._diffParagraphs = function(oldText, newText, lineLength, firstLineNumber) {
            var oldTextWithBreaks, newTextWithBreaks, currChild;

            if (lineLength !== undefined) {
                oldTextWithBreaks = lineNumberingService.insertLineNumbersNode(oldText, lineLength, null, firstLineNumber);
                newTextWithBreaks = lineNumberingService.insertLineNumbersNode(newText, lineLength, null, firstLineNumber);
            } else {
                oldTextWithBreaks = document.createElement('div');
                oldTextWithBreaks.innerHTML = oldText;
                newTextWithBreaks = document.createElement('div');
                newTextWithBreaks.innerHTML = newText;
            }

            for (var i = 0; i < oldTextWithBreaks.childNodes.length; i++) {
                currChild = oldTextWithBreaks.childNodes[i];
                if (currChild.nodeType === TEXT_NODE) {
                    var wrapDel = document.createElement('del');
                    oldTextWithBreaks.insertBefore(wrapDel, currChild);
                    oldTextWithBreaks.removeChild(currChild);
                    wrapDel.appendChild(currChild);
                } else {
                    this.addCSSClass(currChild, 'delete');
                    this._removeColorStyles(currChild);
                }
            }
            for (i = 0; i < newTextWithBreaks.childNodes.length; i++) {
                currChild = newTextWithBreaks.childNodes[i];
                if (currChild.nodeType === TEXT_NODE) {
                    var wrapIns = document.createElement('ins');
                    newTextWithBreaks.insertBefore(wrapIns, currChild);
                    newTextWithBreaks.removeChild(currChild);
                    wrapIns.appendChild(currChild);
                } else {
                    this.addCSSClass(currChild, 'insert');
                    this._removeColorStyles(currChild);
                }
            }

            var mergedFragment = document.createDocumentFragment(),
                el;
            while (oldTextWithBreaks.firstChild) {
                el = oldTextWithBreaks.firstChild;
                oldTextWithBreaks.removeChild(el);
                mergedFragment.appendChild(el);
            }
            while (newTextWithBreaks.firstChild) {
                el = newTextWithBreaks.firstChild;
                newTextWithBreaks.removeChild(el);
                mergedFragment.appendChild(el);
            }

            return this._serializeDom(mergedFragment);
        };

        this.addCSSClassToFirstTag = function (html, className) {
            return html.replace(/<[a-z][^>]*>/i, function (match) {
                if (match.match(/class=["'][a-z0-9 _-]*["']/i)) {
                    return match.replace(/class=["']([a-z0-9 _-]*)["']/i, function (match2, previousClasses) {
                        return "class=\"" + previousClasses + " " + className + "\"";
                    });
                } else {
                    return match.substring(0, match.length - 1) + " class=\"" + className + "\">";
                }
            });
        };

        this._addClassToLastNode = function (html, className) {
            var node = document.createElement('div');
            node.innerHTML = html;
            var foundLast = false;
            for (var i = node.childNodes.length - 1; i >= 0 && !foundLast; i--) {
                if (node.childNodes[i].nodeType === ELEMENT_NODE) {
                    var classes = [];
                    if (node.childNodes[i].getAttribute("class")) {
                        classes = node.childNodes[i].getAttribute("class").split(" ");
                    }
                    classes.push(className);
                    node.childNodes[i].setAttribute("class", classes.sort().join(' ').replace(/^\s+/, '').replace(/\s+$/, ''));
                    foundLast = true;
                }
            }
            return node.innerHTML;
        };

        /**
         * This function removes color-Attributes from the styles of this node or a descendant,
         * as they interfer with the green/red color in HTML and PDF
         *
         * For the moment, it is sufficient to do this only in paragraph diff mode, as we fall back to this mode anyway
         * once we encounter SPANs or other tags inside of INS/DEL-tags
         *
         * @param {Element} node
         * @private
         */
        this._removeColorStyles = function (node) {
            var styles = node.getAttribute('style');
            if (styles && styles.indexOf('color') > -1) {
                var stylesNew = [];
                styles.split(';').forEach(function(style) {
                    if (!style.match(/^\s*color\s*:/i)) {
                        stylesNew.push(style);
                    }
                });
                if (stylesNew.join(";") === '') {
                    node.removeAttribute('style');
                } else {
                    node.setAttribute('style', stylesNew.join(";"));
                }
            }
            for (var i = 0; i < node.childNodes.length; i++) {
                if (node.childNodes[i].nodeType === ELEMENT_NODE) {
                    this._removeColorStyles(node.childNodes[i]);
                }
            }
        };

        /**
         * This function calculates the diff between two strings and tries to fix problems with the resulting HTML.
         * If lineLength and firstLineNumber is given, line numbers will be returned es well
         *
         * @param {number} lineLength
         * @param {number} firstLineNumber
         * @param {string} htmlOld
         * @param {string} htmlNew
         * @returns {string}
         */
        this.diff = function (htmlOld, htmlNew, lineLength, firstLineNumber) {
            var cacheKey = lineLength + ' ' + firstLineNumber + ' ' +
                    lineNumberingService.djb2hash(htmlOld) + lineNumberingService.djb2hash(htmlNew),
                cached = diffCache.get(cacheKey);
            if (!angular.isUndefined(cached)) {
                return cached;
            }

            // This fixes a really strange artefact with the diff that occures under the following conditions:
            // - The first tag of the two texts is identical, e.g. <p>
            // - A change happens in the next tag, e.g. inserted text
            // - The first tag occures a second time in the text, e.g. another <p>
            // In this condition, the first tag is deleted first and inserted afterwards again
            // Test case: "does not break when an insertion followes a beginning tag occuring twice"
            // The work around inserts to tags at the beginning and removes them afterwards again,
            // to make sure this situation does not happen (and uses invisible pseudo-tags in case something goes wrong)
            var workaroundPrepend = "<DUMMY><PREPEND>";

            // os-split-after should not be considered for detecting changes in paragraphs, so we strip it here
            // and add it afterwards.
            // We only do this for P for now, as for more complex types like UL/LI that tend to be nestend,
            // information would get lost by this that we will need to recursively merge it again later on.
            var oldIsSplitAfter = false,
                newIsSplitAfter = false;
            htmlOld = htmlOld.replace(/(\s*<p[^>]+class\s*=\s*["'][^"']*)os-split-after/gi, function(match, beginning) {
                oldIsSplitAfter = true;
                return beginning;
            });
            htmlNew = htmlNew.replace(/(\s*<p[^>]+class\s*=\s*["'][^"']*)os-split-after/gi, function(match, beginning) {
                newIsSplitAfter = true;
                return beginning;
            });

            // Performing the actual diff
            var str = this._diffString(workaroundPrepend + htmlOld, workaroundPrepend + htmlNew),
                diffUnnormalized = str.replace(/^\s+/g, '').replace(/\s+$/g, '').replace(/ {2,}/g, ' ');

            // Remove <del> tags that only delete line numbers
            // We need to do this before removing </del><del> as done in one of the next statements
            diffUnnormalized = diffUnnormalized.replace(
                /<del>((<BR CLASS="os-line-break"><\/del><del>)?(<span[^>]+os-line-number[^>]+?>)(\s|<\/?del>)*<\/span>)<\/del>/gi,
                function(found,tag,br,span) {
                    return (br !== undefined ? br : '') + span + ' </span>';
                }
            );

            diffUnnormalized = diffUnnormalized.replace(/<\/ins><ins>/gi, '').replace(/<\/del><del>/gi, '');

            // Move whitespaces around inserted P's out of the INS-tag
            diffUnnormalized = diffUnnormalized.replace(
                /<ins>(\s*)(<p( [^>]*)?>[\s\S]*?<\/p>)(\s*)<\/ins>/gim,
                function(match, whiteBefore, inner, tagInner, whiteAfter) {
                    return whiteBefore +
                        inner
                        .replace(/<p( [^>]*)?>/gi, function(match) {
                            return match + "<ins>";
                        })
                        .replace(/<\/p>/gi, "</ins></p>") +
                        whiteAfter;
                }
            );

            // Fixes HTML produced by the diff like this:
            // from: <del></P></del><ins> Inserted Text</P>\n<P>More inserted text</P></ins>
            // into: <ins> Inserted Text</ins></P>\n<P>More inserted text</ins></P>
            diffUnnormalized = diffUnnormalized.replace(
                /<del><\/p><\/del><ins>([\s\S]*?)<\/p><\/ins>/gim,
                "<ins>$1</ins></p>"
            );
            diffUnnormalized = diffUnnormalized.replace(
                /<ins>[\s\S]*?<\/ins>/gim,
                function(match) {
                    return match.replace(/(<\/p>\s*<p>)/gi, "</ins>$1<ins>");
                }
            );

            // If only a few characters of a word have changed, don't display this as a replacement of the whole word,
            // but only of these specific characters
            diffUnnormalized = diffUnnormalized.replace(/<del>([a-z0-9,_-]* ?)<\/del><ins>([a-z0-9,_-]* ?)<\/ins>/gi, function (found, oldText, newText) {
                var foundDiff = false, commonStart = '', commonEnd = '',
                    remainderOld = oldText, remainderNew = newText;

                while (remainderOld.length > 0 && remainderNew.length > 0 && !foundDiff) {
                    if (remainderOld[0] == remainderNew[0]) {
                        commonStart += remainderOld[0];
                        remainderOld = remainderOld.substr(1);
                        remainderNew = remainderNew.substr(1);
                    } else {
                        foundDiff = true;
                    }
                }

                foundDiff = false;
                while (remainderOld.length > 0 && remainderNew.length > 0 && !foundDiff) {
                    if (remainderOld[remainderOld.length - 1] == remainderNew[remainderNew.length - 1]) {
                        commonEnd = remainderOld[remainderOld.length - 1] + commonEnd;
                        remainderNew = remainderNew.substr(0, remainderNew.length - 1);
                        remainderOld = remainderOld.substr(0, remainderOld.length - 1);
                    } else {
                        foundDiff = true;
                    }
                }

                var out = commonStart;
                if (remainderOld !== '') {
                    out += '<del>' + remainderOld + '</del>';
                }
                if (remainderNew !== '') {
                    out += '<ins>' + remainderNew + '</ins>';
                }
                out += commonEnd;

                return out;
            });

            // Replace spaces in line numbers by &nbsp;
            diffUnnormalized = diffUnnormalized.replace(
                /<span[^>]+os-line-number[^>]+?>\s*<\/span>/gi,
                function(found) {
                    return found.toLowerCase().replace(/> <\/span/gi, ">&nbsp;</span");
                }
            );


            if (diffUnnormalized.substr(0, workaroundPrepend.length) === workaroundPrepend) {
                diffUnnormalized = diffUnnormalized.substring(workaroundPrepend.length);
            }

            var diff;
            if (this._diffDetectBrokenDiffHtml(diffUnnormalized)) {
                diff = this._diffParagraphs(htmlOld, htmlNew, lineLength, firstLineNumber);
            } else {
                diffUnnormalized = diffUnnormalized.replace(/<ins>.*?(\n.*?)*<\/ins>/gi, function (found) {
                    found = found.replace(/<(div|p|li)[^>]*>/gi, function(match) { return match + '<ins>'; });
                    found = found.replace(/<\/(div|p|li)[^>]*>/gi, function(match) { return '</ins>' + match; });
                    return found;
                });
                diffUnnormalized = diffUnnormalized.replace(/<del>.*?(\n.*?)*<\/del>/gi, function (found) {
                    found = found.replace(/<(div|p|li)[^>]*>/gi, function(match) { return match + '<del>'; });
                    found = found.replace(/<\/(div|p|li)[^>]*>/gi, function(match) { return '</del>' + match; });
                    return found;
                });
                diffUnnormalized = diffUnnormalized.replace(/^<del><p>(.*)<\/p><\/del>$/gi, function(match, inner) { return "<p>" + inner + "</p>"; });

                var node = document.createElement('div');
                node.innerHTML = diffUnnormalized;
                diff = node.innerHTML;

                if (lineLength !== undefined && firstLineNumber !== undefined) {
                    node = lineNumberingService.insertLineNumbersNode(diff, lineLength, null, firstLineNumber);
                    diff = node.innerHTML;
                }
            }

            if (oldIsSplitAfter || newIsSplitAfter) {
                diff = this._addClassToLastNode(diff, "os-split-after");
            }

            diffCache.put(cacheKey, diff);
            return diff;
        };
    }
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.motions.docx', ['OpenSlidesApp.core.docx'])

.factory('MotionDocxExport', [
    '$http',
    '$q',
    'operator',
    'Config',
    'Category',
    'gettextCatalog',
    'FileSaver',
    'lineNumberingService',
    'Html2DocxConverter',
    'MotionComment',
    function ($http, $q, operator, Config, Category, gettextCatalog, FileSaver, lineNumberingService,
        Html2DocxConverter, MotionComment) {

        var PAGEBREAK = '<w:p><w:r><w:br w:type="page" /></w:r></w:p>';

        var converter;

        var getData = function (motions, params) {
            var data = {};
            // header
            var headerline1 = [
                Config.translate(Config.get('general_event_name').value),
                Config.translate(Config.get('general_event_description').value)
            ].filter(Boolean).join(' – ');
            var headerline2 = [
                Config.get('general_event_location').value,
                Config.get('general_event_date').value
            ].filter(Boolean).join(', ');
            data.header = [headerline1, headerline2].join('\n');

            // motion catalog title/preamble
            data.title = Config.translate(Config.get('motions_export_title').value);
            data.preamble = Config.get('motions_export_preamble').value;

            // categories
            var categories = getCategoriesData(motions);
            data.has_categories = categories.length === 0 ? false : true;
            data.categories_translation = gettextCatalog.getString('Categories');
            data.categories = categories;
            data.no_categories = gettextCatalog.getString('No categories available.');
            data.pagebreak_main = categories.length === 0 ? '' : PAGEBREAK;

            // motions
            data.tableofcontents_translation = gettextCatalog.getString('Table of contents');
            data.motions_list = getMotionShortData(motions, params);
            data.no_motions = gettextCatalog.getString('No motions available.');

            return $q(function (resolve) {
                getMotionFullData(motions, params).then(function (motionData) {
                    data.motions = motionData;
                    resolve(data);
                });
            });
        };

        var getCategoriesData = function (motions) {
            var categories = _.map(motions, function (motion) {
                if (motion.category) {
                    return {
                        prefix: motion.category.prefix,
                        name: motion.category.name,
                    };
                }
            });
            // clear out 'undefined' and make the categories unique.
            categories = _.uniqBy(_.filter(categories, function(category) {
                return category;
            }), 'prefix');
            var sortKey = Config.get('motions_export_category_sorting').value;
            return _.orderBy(categories, [sortKey]);
        };

        var getMotionShortData = function (motions, params) {
            return _.map(motions, function (motion) {
                return {
                    identifier: motion.identifier || '',
                    title: motion.getTitleWithChanges(params.changeRecommendationMode),
                };
            });
        };

        var getMotionFullData = function (motions, params) {
            // All translations
            var translation = gettextCatalog.getString('Motion'),
                sequential_translation = gettextCatalog.getString('Sequential number'),
                submitters_translation = gettextCatalog.getString('Submitters'),
                status_translation = gettextCatalog.getString('Status'),
                reason_translation = gettextCatalog.getString('Reason'),
                comment_translation = gettextCatalog.getString('Comments');
            var sequential_enabled = Config.get('motions_export_sequential_number').value;
            // promises for create the actual motion data
            var promises = _.map(motions, function (motion) {
                var title = motion.getTitleWithChanges(params.changeRecommendationMode);
                var text = params.include.text ? motion.getTextByMode(params.changeRecommendationMode, null, null, false) : '';
                var reason = params.include.reason ? motion.getReason() : '';
                var comments = getMotionComments(motion, params.includeComments);

                // Data for one motions. Must include translations, ...
                var motionData = {
                    // Translations
                    motion_translation: translation,
                    sequential_translation: sequential_translation,
                    submitters_translation: submitters_translation,
                    reason_translation: reason.length === 0 ? '' : reason_translation,
                    status_translation: status_translation,
                    comment_translation: comments.length === 0 ? '' : comment_translation,
                    sequential_enabled: sequential_enabled,
                    // Actual data
                    id: motion.id,
                    identifier: motion.identifier || '',
                    title: title,
                    submitters: params.include.submitters ?  _.map(motion.submitters, function (submitter) {
                                    return submitter.get_full_name();
                                }).join(', ') : '',
                    status: motion.getStateName(),
                    // Miscellaneous stuff
                    preamble: gettextCatalog.getString(Config.get('motions_preamble').value),
                    pagebreak: PAGEBREAK,
                };
                // converting html to docx is async, so text, reason and comments are inserted here.
                return $q(function (resolve) {
                    var convertPromises = _.map(comments, function (comment) {
                        return converter.html2docx(comment.comment).then(function (commentAsDocx) {
                            comment.comment = commentAsDocx;
                        });
                    });
                    convertPromises.push(converter.html2docx(text).then(function (textAsDocx) {
                        motionData.text = textAsDocx;
                    }));
                    convertPromises.push(converter.html2docx(reason).then(function (reasonAsDocx) {
                        motionData.reason = reasonAsDocx;
                    }));
                    $q.all(convertPromises).then(function () {
                        motionData.comments = comments;
                        resolve(motionData);
                    });
                });
            });
            // resolve, if all motion data is fetched.
            return $q(function (resolve) {
                $q.all(promises).then(function (data) {
                    if (data.length) {
                        // clear pagebreak on last element
                        data[data.length - 1].pagebreak = '';
                    }
                    resolve(data);
                });
            });
        };

        var getMotionComments = function (motion, fieldsIncluded) {
            var fields = MotionComment.getNoSpecialCommentsFields();
            var comments = [];
            _.forEach(fieldsIncluded, function (ok, id) {
                if (ok && motion.comments[id]) {
                    var title = fields[id].name;
                    if (!fields[id].public) {
                        title += ' (' + gettextCatalog.getString('internal') + ')';
                    }
                    var comment = motion.comments[id];
                    if (comment.indexOf('<p>') !== 0) {
                        comment = '<p>' + comment + '</p>';
                    }
                    comments.push({
                        title: title,
                        comment: comment,
                    });
                }
            });
            return comments;
        };

        return {
            export: function (motions, params) {
                converter = Html2DocxConverter.createInstance();
                params = _.clone(params || {}); // Clone this to avoid sideeffects.
                _.defaults(params, {
                    filename: gettextCatalog.getString('motions') + '.docx',
                    changeRecommendationMode: Config.get('motions_recommendation_text_mode').value,
                    include: {
                        text: true,
                        reason: true,
                        submitters: true,
                    },
                    includeComments: {},
                });
                if (!_.includes(['original', 'changed', 'agreed'], params.changeRecommendationMode)) {
                    params.changeRecommendationMode = 'original';
                }

                $http.get('/motions/docxtemplate/').then(function (success) {
                    var content = window.atob(success.data);
                    var doc = new Docxgen(content);

                    getData(motions, params).then(function (data) {
                        doc.setData(data);
                        doc.render();

                        var zip = doc.getZip();
                        zip = converter.updateZipFile(zip);

                        var out = zip.generate({type: 'blob'});
                        FileSaver.saveAs(out, params.filename);
                    });
                });
            },
        };
    }
]);

}());

(function () {

"use strict";

angular.module('OpenSlidesApp.motions.lineNumbering', [])

/**
 * Current limitations of this implementation:
 *
 * Only the following inline elements are supported:
 * - 'SPAN', 'A', 'EM', 'S', 'B', 'I', 'STRONG', 'U', 'BIG', 'SMALL', 'SUB', 'SUP', 'TT'
 * - 'INS' and 'DEL' are supported, but line numbering does not affect the content of 'INS'-elements
 *
 * Only other inline elements are allowed within inline elements.
 * No constructs like <a...><div></div></a> are allowed. CSS-attributes like 'display: block' are ignored.
 */

.service('lineNumberingService', [
    '$cacheFactory',
    function ($cacheFactory) {
        var ELEMENT_NODE = 1,
            TEXT_NODE = 3;

        // Counts the number of characters in the current line, beyond singe nodes.
        // Needs to be resetted after each line break and after entering a new block node.
        this._currentInlineOffset = null;

        // The last position of a point suitable for breaking the line. null or an object with the following values:
        // - node: the node that contains the position. Guaranteed to be a TextNode
        // - offset: the offset of the breaking characters (like the space)
        // Needs to be resetted after each line break and after entering a new block node.
        this._lastInlineBreakablePoint = null;

        // The line number counter
        this._currentLineNumber = null;

        // Indicates that we just entered a block element and we want to add a line number without line break at the beginning.
        this._prependLineNumberToFirstText = false;

        // A workaround to prevent double line numbers
        this._ignoreNextRegularLineNumber = false;

        // Decides if the content of inserted nodes should count as well. This is used so we can use the algorithm on a
        // text with inline diff annotations and get the same line numbering as with the original text (when set to false)
        this._ignoreInsertedText = false;

        var lineNumberCache = $cacheFactory('linenumbering.service');

        this.djb2hash = function(str) {
            var hash = 5381, char;
            for (var i = 0; i < str.length; i++) {
                char = str.charCodeAt(i);
                hash = ((hash << 5) + hash) + char;
            }
            return hash.toString();
        };

        this._isInlineElement = function (node) {
            var inlineElements = [
                'SPAN', 'A', 'EM', 'S', 'B', 'I', 'STRONG', 'U', 'BIG', 'SMALL', 'SUB', 'SUP', 'TT', 'INS', 'DEL',
                'STRIKE'
            ];
            return (inlineElements.indexOf(node.nodeName) > -1);
        };

        this._isIgnoredByLineNumbering = function (node) {
            if (node.nodeName === 'INS') {
                return this._ignoreInsertedText;
            } else if (this._isOsLineNumberNode(node)) {
                return true;
            } else {
                return false;
            }
        };

        this._isOsLineBreakNode = function (node) {
            var isLineBreak = false;
            if (node && node.nodeType === ELEMENT_NODE && node.nodeName == 'BR' && node.hasAttribute('class')) {
                var classes = node.getAttribute('class').split(' ');
                if (classes.indexOf('os-line-break') > -1) {
                    isLineBreak = true;
                }
            }
            return isLineBreak;
        };

        this._isOsLineNumberNode = function (node) {
            var isLineNumber = false;
            if (node && node.nodeType === ELEMENT_NODE && node.nodeName == 'SPAN' && node.hasAttribute('class')) {
                var classes = node.getAttribute('class').split(' ');
                if (classes.indexOf('os-line-number') > -1) {
                    isLineNumber = true;
                }
            }
            return isLineNumber;
        };

        this._getLineNumberNode = function(fragment, lineNumber) {
            return fragment.querySelector('.os-line-number.line-number-' + lineNumber);
        };

        this._htmlToFragment = function(html) {
            var fragment = document.createDocumentFragment(),
                div = document.createElement('DIV');
            div.innerHTML = html;
            while (div.childElementCount) {
                var child = div.childNodes[0];
                div.removeChild(child);
                fragment.appendChild(child);
            }
            return fragment;
        };

        this._fragmentToHtml = function(fragment) {
            var div = document.createElement('DIV');
            while (fragment.firstChild) {
                var child = fragment.firstChild;
                fragment.removeChild(child);
                div.appendChild(child);
            }
            return div.innerHTML;
        };

        this._createLineBreak = function () {
            var br = document.createElement('br');
            br.setAttribute('class', 'os-line-break');
            return br;
        };

        this._createLineNumber = function () {
            if (this._ignoreNextRegularLineNumber) {
                this._ignoreNextRegularLineNumber = false;
                return;
            }
            var node = document.createElement('span');
            var lineNumber = this._currentLineNumber;
            this._currentLineNumber++;
            node.setAttribute('class', 'os-line-number line-number-' + lineNumber);
            node.setAttribute('data-line-number', lineNumber + '');
            node.setAttribute('contenteditable', 'false');
            node.innerHTML = '&nbsp;'; // Prevent ckeditor from stripping out empty span's
            return node;
        };

        /**
         * Splits a TEXT_NODE into an array of TEXT_NODEs and BR-Elements separating them into lines.
         * Each line has a maximum length of 'length', with one exception: spaces are accepted to exceed the length.
         * Otherwise the string is split by the last space or dash in the line.
         *
         * @param node
         * @param length
         * @param highlight
         * @returns Array
         * @private
         */
        this._textNodeToLines = function (node, length, highlight) {
            var out = [],
                currLineStart = 0,
                i = 0,
                firstTextNode = true,
                service = this;
            var addLine = function (text, highlight) {
                var node;
                if (typeof highlight === 'undefined') {
                    highlight = -1;
                }
                if (firstTextNode) {
                    if (highlight === service._currentLineNumber - 1) {
                        node = document.createElement('span');
                        node.setAttribute('class', 'highlight');
                        node.innerHTML = text;
                    } else {
                        node = document.createTextNode(text);
                    }
                    firstTextNode = false;
                } else {
                    if (service._currentLineNumber === highlight && highlight !== null) {
                        node = document.createElement('span');
                        node.setAttribute('class', 'highlight');
                        node.innerHTML = text;
                    } else {
                        node = document.createTextNode(text);
                    }
                    out.push(service._createLineBreak());
                    if (service._currentLineNumber !== null) {
                        out.push(service._createLineNumber());
                    }
                }
                out.push(node);
                return node;
            };
            var addLinebreakToPreviousNode = function (node, offset, highlight) {
                var firstText = node.nodeValue.substr(0, offset + 1),
                    secondText = node.nodeValue.substr(offset + 1);
                var lineBreak = service._createLineBreak();
                var firstNode = document.createTextNode(firstText);
                node.parentNode.insertBefore(firstNode, node);
                node.parentNode.insertBefore(lineBreak, node);
                node.nodeValue = secondText;
            };

            if (node.nodeValue === "\n") {
                out.push(node);
            } else {

                // This happens if a previous inline element exactly stretches to the end of the line
                if (this._currentInlineOffset >= length) {
                    out.push(service._createLineBreak());
                    if (this._currentLineNumber !== null) {
                        out.push(service._createLineNumber());
                    }
                    this._currentInlineOffset = 0;
                    this._lastInlineBreakablePoint = null;
                } else if (this._prependLineNumberToFirstText) {
                    if (this._ignoreNextRegularLineNumber) {
                        this._ignoreNextRegularLineNumber = false;
                    } else if (service._currentLineNumber !== null) {
                        out.push(service._createLineNumber());
                    }
                }
                this._prependLineNumberToFirstText = false;

                while (i < node.nodeValue.length) {
                    var lineBreakAt = null;
                    if (this._currentInlineOffset >= length) {
                        if (this._lastInlineBreakablePoint !== null) {
                            lineBreakAt = this._lastInlineBreakablePoint;
                        } else {
                            lineBreakAt = {
                                'node': node,
                                'offset': i - 1
                            };
                        }
                    }
                    if (lineBreakAt !== null && (node.nodeValue[i] !== ' ' && node.nodeValue[i] !== "\n")) {
                        if (lineBreakAt.node === node) {
                            // The last possible breaking point is in this text node
                            var currLine = node.nodeValue.substring(currLineStart, lineBreakAt.offset + 1);
                            addLine(currLine, highlight);

                            currLineStart = lineBreakAt.offset + 1;
                            this._currentInlineOffset = i - lineBreakAt.offset - 1;
                            this._lastInlineBreakablePoint = null;
                        } else {
                            // The last possible breaking point was not in this text not, but one we have already passed
                            var remainderOfPrev = lineBreakAt.node.nodeValue.length - lineBreakAt.offset - 1;
                            addLinebreakToPreviousNode(lineBreakAt.node, lineBreakAt.offset, highlight);

                            this._currentInlineOffset = i + remainderOfPrev;
                            this._lastInlineBreakablePoint = null;
                        }

                    }

                    if (node.nodeValue[i] === ' ' || node.nodeValue[i] === '-' || node.nodeValue[i] === "\n") {
                        this._lastInlineBreakablePoint = {
                            'node': node,
                            'offset': i
                        };
                    }

                    this._currentInlineOffset++;
                    i++;

                }
                var lastLine = addLine(node.nodeValue.substring(currLineStart), highlight);
                if (this._lastInlineBreakablePoint !== null) {
                    this._lastInlineBreakablePoint.node = lastLine;
                }
            }
            return out;
        };


        /**
         * Moves line breaking and line numbering markup before inline elements
         *
         * @param innerNode
         * @param outerNode
         * @private
         */
        this._moveLeadingLineBreaksToOuterNode = function (innerNode, outerNode) {
            if (this._isInlineElement(innerNode)) {
                if (this._isOsLineBreakNode(innerNode.firstChild)) {
                    var br = innerNode.firstChild;
                    innerNode.removeChild(br);
                    outerNode.appendChild(br);
                }
                if (this._isOsLineNumberNode(innerNode.firstChild)) {
                    var span = innerNode.firstChild;
                    innerNode.removeChild(span);
                    outerNode.appendChild(span);
                }
            }
        };

        this._lengthOfFirstInlineWord = function (node) {
            if (!node.firstChild) {
                return 0;
            }
            if (node.firstChild.nodeType == TEXT_NODE) {
                var parts = node.firstChild.nodeValue.split(' ');
                return parts[0].length;
            } else {
                return this._lengthOfFirstInlineWord(node.firstChild);
            }
        };

        this._insertLineNumbersToInlineNode = function (node, length, highlight) {
            var oldChildren = [], i;
            for (i = 0; i < node.childNodes.length; i++) {
                oldChildren.push(node.childNodes[i]);
            }

            while (node.firstChild) {
                node.removeChild(node.firstChild);
            }

            for (i = 0; i < oldChildren.length; i++) {
                if (oldChildren[i].nodeType == TEXT_NODE) {
                    var ret = this._textNodeToLines(oldChildren[i], length, highlight);
                    for (var j = 0; j < ret.length; j++) {
                        node.appendChild(ret[j]);
                    }
                } else if (oldChildren[i].nodeType == ELEMENT_NODE) {
                    var firstword = this._lengthOfFirstInlineWord(oldChildren[i]),
                        overlength = ((this._currentInlineOffset + firstword) > length && this._currentInlineOffset > 0);
                    if (overlength && this._isInlineElement(oldChildren[i])) {
                        this._currentInlineOffset = 0;
                        this._lastInlineBreakablePoint = null;
                        node.appendChild(this._createLineBreak());
                        if (this._currentLineNumber !== null) {
                            node.appendChild(this._createLineNumber());
                        }
                    }
                    var changedNode = this._insertLineNumbersToNode(oldChildren[i], length, highlight);
                    this._moveLeadingLineBreaksToOuterNode(changedNode, node);
                    node.appendChild(changedNode);
                } else {
                    throw 'Unknown nodeType: ' + i + ': ' + oldChildren[i];
                }
            }

            return node;
        };

        this._calcBlockNodeLength = function (node, oldLength) {
            var newLength = oldLength;
            switch (node.nodeName) {
                case 'LI':
                    newLength -= 5;
                    break;
                case 'BLOCKQUOTE':
                    newLength -= 20;
                    break;
                case 'DIV':
                case 'P':
                    var styles = node.getAttribute("style"),
                        padding = 0;
                    if (styles) {
                        var leftpad = styles.split("padding-left:");
                        if (leftpad.length > 1) {
                            leftpad = parseInt(leftpad[1]);
                            padding += leftpad;
                        }
                        var rightpad = styles.split("padding-right:");
                        if (rightpad.length > 1) {
                            rightpad = parseInt(rightpad[1]);
                            padding += rightpad;
                        }
                        newLength -= (padding / 5);
                    }
                    break;
                case 'H1':
                    newLength *= 0.66;
                    break;
                case 'H2':
                    newLength *= 0.75;
                    break;
                case 'H3':
                    newLength *= 0.85;
                    break;
            }
            return Math.ceil(newLength);
        };

        this._insertLineNumbersToBlockNode = function (node, length, highlight) {
            this._currentInlineOffset = 0;
            this._lastInlineBreakablePoint = null;
            this._prependLineNumberToFirstText = true;

            var oldChildren = [], i;
            for (i = 0; i < node.childNodes.length; i++) {
                oldChildren.push(node.childNodes[i]);
            }

            while (node.firstChild) {
                node.removeChild(node.firstChild);
            }

            for (i = 0; i < oldChildren.length; i++) {
                if (oldChildren[i].nodeType == TEXT_NODE) {
                    if (!oldChildren[i].nodeValue.match(/\S/)) {
                        // White space nodes between block elements should be ignored
                        var prevIsBlock = (i > 0 && !this._isInlineElement(oldChildren[i - 1]));
                        var nextIsBlock = (i < oldChildren.length - 1 && !this._isInlineElement(oldChildren[i + 1]));
                        if ((prevIsBlock && nextIsBlock) || (i === 0 && nextIsBlock) || (i === oldChildren.length - 1 && prevIsBlock)) {
                            node.appendChild(oldChildren[i]);
                            continue;
                        }
                    }
                    var ret = this._textNodeToLines(oldChildren[i], length, highlight);
                    for (var j = 0; j < ret.length; j++) {
                        node.appendChild(ret[j]);
                    }
                } else if (oldChildren[i].nodeType == ELEMENT_NODE) {
                    var firstword = this._lengthOfFirstInlineWord(oldChildren[i]),
                        overlength = ((this._currentInlineOffset + firstword) > length && this._currentInlineOffset > 0);
                    if (overlength && this._isInlineElement(oldChildren[i]) && !this._isIgnoredByLineNumbering(oldChildren[i])) {
                        this._currentInlineOffset = 0;
                        this._lastInlineBreakablePoint = null;
                        node.appendChild(this._createLineBreak());
                        if (this._currentLineNumber !== null) {
                            node.appendChild(this._createLineNumber());
                        }
                    }
                    var changedNode = this._insertLineNumbersToNode(oldChildren[i], length, highlight);
                    this._moveLeadingLineBreaksToOuterNode(changedNode, node);
                    node.appendChild(changedNode);
                } else {
                    throw 'Unknown nodeType: ' + i + ': ' + oldChildren[i];
                }
            }

            this._currentInlineOffset = 0;
            this._lastInlineBreakablePoint = null;
            this._prependLineNumberToFirstText = true;
            this._ignoreNextRegularLineNumber = false;

            return node;
        };

        this._insertLineNumbersToNode = function (node, length, highlight) {
            if (node.nodeType !== ELEMENT_NODE) {
                throw 'This method may only be called for ELEMENT-nodes: ' + node.nodeValue;
            }
            if (this._isIgnoredByLineNumbering(node)) {
                if (this._currentInlineOffset === 0 && this._currentLineNumber !== null) {
                    var lineNumberNode = this._createLineNumber();
                    if (lineNumberNode) {
                        node.insertBefore(lineNumberNode, node.firstChild);
                        this._ignoreNextRegularLineNumber = true;
                    }
                }
                return node;
            } else if (this._isInlineElement(node)) {
                return this._insertLineNumbersToInlineNode(node, length, highlight);
            } else {
                var newLength = this._calcBlockNodeLength(node, length);
                return this._insertLineNumbersToBlockNode(node, newLength, highlight);
            }
        };

        this._stripLineNumbers = function (node) {
            for (var i = 0; i < node.childNodes.length; i++) {
                if (this._isOsLineBreakNode(node.childNodes[i]) || this._isOsLineNumberNode(node.childNodes[i])) {
                    // If a newline character follows a line break, it's been very likely inserted by the WYSIWYG-editor
                    if (node.childNodes.length > (i + 1) && node.childNodes[i + 1].nodeType === TEXT_NODE) {
                        if (node.childNodes[i + 1].nodeValue[0] === "\n") {
                            node.childNodes[i + 1].nodeValue = " " + node.childNodes[i + 1].nodeValue.substring(1);
                        }
                    }
                    node.removeChild(node.childNodes[i]);
                    i--;
                } else {
                    this._stripLineNumbers(node.childNodes[i]);
                }
            }
        };

        this._nodesToHtml = function (nodes) {
            var root = document.createElement('div');
            for (var i in nodes) {
                if (nodes.hasOwnProperty(i)) {
                    root.appendChild(nodes[i]);
                }
            }
            return root.innerHTML;
        };

        /**
         *
         * @param {string} html
         * @param {number} lineLength
         * @param {number|null} highlight - optional
         * @param {number|null} firstLine
         */
        this.insertLineNumbersNode = function (html, lineLength, highlight, firstLine) {
            // Removing newlines after BRs, as they lead to problems like #3410
            if (html) {
                html = html.replace(/(<br[^>]*>)[\n\r]+/gi, '$1');
            }

            var root = document.createElement('div');
            root.innerHTML = html;

            this._currentInlineOffset = 0;
            this._lastInlineBreakablePoint = null;
            if (firstLine) {
                this._currentLineNumber = parseInt(firstLine);
            } else {
                this._currentLineNumber = 1;
            }
            if (highlight !== null) {
                highlight = parseInt(highlight);
            }
            this._prependLineNumberToFirstText = true;
            this._ignoreNextRegularLineNumber = false;
            this._ignoreInsertedText = true;

            return this._insertLineNumbersToNode(root, lineLength, highlight);
        };

        /**
         *
         * @param {string} html
         * @param {number} lineLength
         * @param {number|null} highlight - optional
         * @param {function} callback
         * @param {number} firstLine
         * @returns {string}
         */
        this.insertLineNumbers = function (html, lineLength, highlight, callback, firstLine) {
            var newHtml, newRoot;

            if (highlight > 0) {
                // Caching versions with highlighted line numbers is probably not worth it
                newRoot = this.insertLineNumbersNode(html, lineLength, highlight, firstLine);
                newHtml = newRoot.innerHTML;
            } else {
                var firstLineStr = (firstLine === undefined || firstLine === null ? '' : firstLine.toString());
                var cacheKey = this.djb2hash(firstLineStr + "-" + lineLength.toString() + html);
                newHtml = lineNumberCache.get(cacheKey);

                if (angular.isUndefined(newHtml)) {
                    newRoot = this.insertLineNumbersNode(html, lineLength, null, firstLine);
                    newHtml = newRoot.innerHTML;
                    lineNumberCache.put(cacheKey, newHtml);
                }
            }

            if (callback) {
                callback();
            }

            return newHtml;
        };

        /**
         * @param {string} html
         * @param {number} lineLength
         * @param {boolean} countInserted
         */
        this.insertLineBreaksWithoutNumbers = function (html, lineLength, countInserted) {
            var root = document.createElement('div');
            root.innerHTML = html;

            this._currentInlineOffset = 0;
            this._lastInlineBreakablePoint = null;
            this._currentLineNumber = null;
            this._prependLineNumberToFirstText = true;
            this._ignoreNextRegularLineNumber = false;
            this._ignoreInsertedText = !countInserted;

            var newRoot = this._insertLineNumbersToNode(root, lineLength, null);

            return newRoot.innerHTML;
        };

        /**
         * @param {string} html
         * @returns {string}
         */
        this.stripLineNumbers = function (html) {
            var root = document.createElement('div');
            root.innerHTML = html;
            this._stripLineNumbers(root);
            return root.innerHTML;
        };

        /**
         * Traverses up the DOM tree until it finds a node with a nextSibling, then returns that sibling
         *
         * @param node
         * @private
         */
        this._findNextAuntNode = function(node) {
            if (node.nextSibling) {
                return node.nextSibling;
            } else if (node.parentNode) {
                return this._findNextAuntNode(node.parentNode);
            } else {
                return null;
            }
        };

        this._highlightUntilNextLine = function(lineNumberNode) {
            var currentNode = lineNumberNode,
                foundNextLineNumber = false;

            do {
                var wasHighlighted = false;
                if (currentNode.nodeType === TEXT_NODE) {
                    var node = document.createElement('span');
                    node.setAttribute('class', 'highlight');
                    node.innerHTML = currentNode.nodeValue;
                    currentNode.parentNode.insertBefore(node, currentNode);
                    currentNode.parentNode.removeChild(currentNode);
                    currentNode = node;
                    wasHighlighted = true;
                } else {
                    wasHighlighted = false;
                }

                if (currentNode.childNodes.length > 0 && !this._isOsLineNumberNode(currentNode) && !wasHighlighted) {
                    currentNode = currentNode.childNodes[0];
                } else if (currentNode.nextSibling) {
                    currentNode = currentNode.nextSibling;
                } else {
                    currentNode = this._findNextAuntNode(currentNode);
                }

                if (this._isOsLineNumberNode(currentNode)) {
                    foundNextLineNumber = true;
                }
            } while (!foundNextLineNumber && currentNode !== null);
        };

        /**
         * @param {string} html
         * @param {number} lineNumber
         * @return {string}
         */
        this.highlightLine = function (html, lineNumber) {
            lineNumber = parseInt(lineNumber);
            var fragment = this._htmlToFragment(html),
                lineNumberNode = this._getLineNumberNode(fragment, lineNumber);

            if (lineNumberNode) {
                this._highlightUntilNextLine(lineNumberNode);
                html = this._fragmentToHtml(fragment);
            }

            return html;
        };
    }
]);


}());

(function () {

'use strict';

angular.module('OpenSlidesApp.motions.motionBlockProjector', [])


// MotionBlock projector elements

.config([
    'slidesProvider',
    function(slidesProvider) {
        slidesProvider.registerSlide('motions/motion-block', {
            template: 'static/templates/motions/slide_motion_block.html',
        });
    }
])

.controller('SlideMotionBlockCtrl', [
    '$scope',
    'Motion',
    'MotionBlock',
    function($scope, Motion, MotionBlock) {
        // Attention! Each object that is used here has to be dealt on server side.
        // Add it to the coresponding get_requirements method of the ProjectorElement
        // class.
        var id = $scope.element.id;
        MotionBlock.bindOne(id, $scope, 'motionBlock');

        // Returns a shortened motion title. If the title is longer then maxLength, it is
        // split at the last whitespace that is in maxLength. Three dots are added then.
        $scope.getShortTitle = function (motion) {
            var maxLength = 40;
            var title = motion.getTitle();

            if (title.length <= maxLength) {
                return title;
            }

            // Find last whitespace that is before maxLength. Split the title
            // there and append dots.
            var whitespaceIndex = -1;
            for (var i = 0; i < maxLength+1; i++) {
                if (title[i] === ' ') {
                    whitespaceIndex = i;
                }
            }

            if (whitespaceIndex === -1) {
                // just one long word.. split it :/
                return title.substr(0, maxLength) + '...';
            } else {
                return title.substr(0, whitespaceIndex) + '...';
            }
        };
    }
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.motions.motionBlock', [])


// MotionBlock model

.factory('MotionBlock', [
    'DS',
    'jsDataModel',
    'gettext',
    function(DS, jsDataModel, gettext) {
        var name = 'motions/motion-block';
        return DS.defineResource({
            name: name,
            useClass: jsDataModel,
            verboseName: gettext('Motion block'),
            methods: {
                getResourceName: function () {
                    return name;
                },
                getAgendaTitle: function () {
                    return this.title;
                },
            },
            relations: {
                belongsTo: {
                    'agenda/item': {
                        localKey: 'agenda_item_id',
                        localField: 'agenda_item',
                    }
                },
                hasMany: {
                    'motions/motion': {
                        localField: 'motions',
                        foreignKey: 'motion_block_id',
                        osProtectedRelation: true,
                    }
                },
            }
        });
    }
])

.run(['MotionBlock', function(MotionBlock) {}])

// MotionBlock views (list view, create dialog, update dialog)
.factory('MotionBlockForm', [
    '$http',
    'operator',
    'gettextCatalog',
    'Agenda',
    'AgendaTree',
    function ($http, operator, gettextCatalog, Agenda, AgendaTree) {
        return {
            // Get ngDialog configuration.
            getDialog: function (motionBlock) {
                return {
                    template: 'static/templates/motions/motion-block-form.html',
                    controller: (motionBlock) ? 'MotionBlockUpdateCtrl' : 'MotionBlockCreateCtrl',
                    className: 'ngdialog-theme-default wide-form',
                    closeByEscape: false,
                    closeByDocument: false,
                    resolve: {
                        motionBlockId: function () {return motionBlock ? motionBlock.id : void 0;}
                    }
                };
            },
            // Get angular-formly fields.
            getFormFields: function (isCreateForm) {
                var formFields = [
                    {
                        key: 'title',
                        type: 'input',
                        templateOptions: {
                            label: gettextCatalog.getString('Title')
                        }
                    },
                ];

                // show as agenda item + parent item
                if (isCreateForm) {
                    formFields.push({
                        key: 'showAsAgendaItem',
                        type: 'checkbox',
                        templateOptions: {
                            label: gettextCatalog.getString('Show as agenda item'),
                            description: gettextCatalog.getString('If deactivated it appears as internal item on agenda.')
                        },
                        hide: !(operator.hasPerms('motions.can_manage') && operator.hasPerms('agenda.can_manage'))
                    });
                    formFields.push({
                        key: 'agenda_parent_id',
                        type: 'select-single',
                        templateOptions: {
                            label: gettextCatalog.getString('Parent item'),
                            options: AgendaTree.getFlatTree(Agenda.getAll()),
                            ngOptions: 'item.id as item.getListViewTitle() for item in to.options | notself : model.agenda_item_id',
                            placeholder: gettextCatalog.getString('Select a parent item ...')
                        },
                        hide: !operator.hasPerms('agenda.can_manage')
                    });
                }

                return formFields;
            }
        };
    }
])

.controller('MotionBlockListCtrl', [
    '$scope',
    'ngDialog',
    'MotionBlock',
    'MotionBlockForm',
    'Projector',
    'ProjectionDefault',
    function ($scope, ngDialog, MotionBlock, MotionBlockForm, Projector, ProjectionDefault) {
        $scope.$watch(function () {
            return Projector.lastModified();
        }, function () {
            var projectiondefault = ProjectionDefault.filter({name: 'motionBlocks'})[0];
            if (projectiondefault) {
                $scope.defaultProjectorId = projectiondefault.projector_id;
            }
        });
        // Two-way data binding for all MotionBlock instances.
        MotionBlock.bindAll({}, $scope, 'motionBlocks');

        // Dialog with a form to create or update a MotionBlock instance.
        $scope.openFormDialog = function (motionBlock) {
            ngDialog.open(MotionBlockForm.getDialog(motionBlock));
        };

        // Confirm dialog to delete a MotionBlock instance.
        $scope.delete = function (motionBlock) {
            MotionBlock.destroy(motionBlock.id);
        };
    }
])

.controller('MotionBlockDetailCtrl', [
    '$scope',
    '$http',
    'ngDialog',
    'Motion',
    'MotionBlockForm',
    'MotionBlock',
    'motionBlockId',
    'Projector',
    'ProjectionDefault',
    'WebpageTitle',
    'gettextCatalog',
    'ErrorMessage',
    function($scope, $http, ngDialog, Motion, MotionBlockForm, MotionBlock, motionBlockId, Projector,
        ProjectionDefault, WebpageTitle, gettextCatalog, ErrorMessage) {
        $scope.$watch(function () {
            return MotionBlock.lastModified(motionBlockId);
        }, function () {
            $scope.motionBlock = MotionBlock.get(motionBlockId);
            WebpageTitle.updateTitle(gettextCatalog.getString('Motion block') + ' ' +
                $scope.motionBlock.agenda_item.getTitle());
        });
        Motion.bindAll({}, $scope, 'motions');
        $scope.$watch(function () {
            return Projector.lastModified();
        }, function () {
            var projectiondefault = ProjectionDefault.filter({name: 'motionBlocks'})[0];
            if (projectiondefault) {
                $scope.defaultProjectorId = projectiondefault.projector_id;
            }
        });
        $scope.openDialog = function (motionBlock) {
            ngDialog.open(MotionBlockForm.getDialog(motionBlock));
        };
        $scope.followRecommendations = function () {
            $http.post('/rest/motions/motion-block/' + motionBlockId + '/follow_recommendations/').then(
                function (success) {
                $scope.alert = { type: 'success', msg: success.data.detail, show: true };
            }, function (error) {
                $scope.alert = ErrorMessage.forAlert(error);
            });
        };
        $scope.delete = function (motion) {
            motion.motion_block_id = null;
            motion.title = motion.getTitle(-1);
            motion.text = motion.getText(-1);
            motion.reason = motion.getReason(-1);
            Motion.save(motion);
        };
    }
])

.controller('MotionBlockCreateCtrl', [
    '$scope',
    'MotionBlock',
    'MotionBlockForm',
    function($scope, MotionBlock, MotionBlockForm) {
        // Prepare form.
        $scope.model = {};
        $scope.model.showAsAgendaItem = true;

        // Get all form fields.
        $scope.formFields = MotionBlockForm.getFormFields(true);

        // Save form.
        $scope.save = function (motionBlock) {
            motionBlock.agenda_type = motionBlock.showAsAgendaItem ? 1 : 2;
            // The attribute motionBlock.agenda_parent_id is set by the form, see form definition.
            MotionBlock.create(motionBlock).then(
                function (success) {
                    $scope.closeThisDialog();
                },
                function (error) {
                    var message = '';
                    for (var e in error.data) {
                        message += e + ': ' + error.data[e] + ' ';
                    }
                    $scope.alert = {type: 'danger', msg: message, show: true};
                }
            );
        };
    }
])

.controller('MotionBlockUpdateCtrl', [
    '$scope',
    '$state',
    'MotionBlock',
    'MotionBlockForm',
    'motionBlockId',
    function($scope, $state, MotionBlock, MotionBlockForm, motionBlockId) {
        $scope.alert = {};

        // Prepare form. Set initial values by creating a deep copy of
        // motionBlock object so list/detail view is not updated while editing.
        var motionBlock = MotionBlock.get(motionBlockId);
        $scope.model = angular.copy(motionBlock);

        // Get all form fields.
        $scope.formFields = MotionBlockForm.getFormFields();

        // Save form.
        $scope.save = function (motionBlock) {
            // inject the changed motionBlock (copy) object back into DS store
            MotionBlock.inject(motionBlock);
            // save changed motionBlock object on server
            MotionBlock.create(motionBlock).then(
                function (success) {
                    $scope.closeThisDialog();
                },
                function (error) {
                    // Save error: revert all changes by restore
                    // (refresh) original motionBlock object from server
                    MotionBlock.refresh(motionBlock);  // TODO: Why do we need a refresh here?
                    var message = '';
                    for (var e in error.data) {
                        message += e + ': ' + error.data[e] + ' ';
                    }
                    $scope.alert = {type: 'danger', msg: message, show: true};
                }
            );
        };
    }
]);

}());

(function () {

"use strict";

angular.module('OpenSlidesApp.motions.motionservices', ['OpenSlidesApp.motions', 'OpenSlidesApp.motions.lineNumbering'])

/* Generic inline editing factory.
 *
 * getOriginalData: Function that should return the editor data. The editor object is passed.
 * saveData: Function that is called whith the editor object as argument. This function
 *      should prepare the save. If the function returns true, the save process won't be
 *      continued. Else a patch request is send.
 */
.factory('MotionInlineEditing', [
    'Motion',
    '$timeout',
    'gettextCatalog',
    function (Motion, $timeout, gettextCatalog) {
        var createInstance = function ($scope, motion, selector, versioning, ckeditorOptions, getOriginalData, saveData) {
            var obj = {
                active: false,
                changed: false,
                isEditable: false,
                trivialChange: false,
                originalHtml: null,
            };
            ckeditorOptions.readOnly = true;

            obj.setVersion = function (_motion, versionId) {
                motion = _motion; // If this is not updated,
                obj.originalHtml = motion.getTextWithLineBreaks(versionId);
                obj.changed = false;
                if (obj.editor) {
                    obj.editor.setReadOnly(true);
                    obj.editor.setData(obj.originalHtml);
                }
            };

            obj.enable = function () {
                obj.active = true;
                obj.isEditable = true;
                ckeditorOptions.language = localStorage.getItem('language');
                obj.editor = CKEDITOR.inline(selector, ckeditorOptions);
                obj.editor.on('change', function () {
                    $timeout(function() {
                        if (obj.editor.getData() !== obj.originalHtml) {
                            obj.changed = true;
                        } else {
                            obj.changed = false;
                        }
                    });
                });
                obj.revert();
            };

            obj.disable = function () {
                if (obj.editor) {
                    obj.editor.setReadOnly(true);
                    obj.editor.setData(obj.originalHtml, {
                        callback: function() {
                            obj.editor.destroy();
                        }
                    });
                }
                $timeout(function() {
                    obj.active = false;
                    obj.changed = false;
                    obj.isEditable = false;
                });
            };

            // sets editor content to the initial motion state
            obj.revert = function(originalData) {
                if (obj.editor) {
                    obj.originalHtml = getOriginalData(obj);
                    obj.editor.setData(
                        getOriginalData(obj), {
                        callback: function() {
                            obj.originalHtml = obj.editor.getData();
                            obj.editor.setReadOnly(false);
                            $timeout(function() {
                                obj.changed = false;
                            });
                            $timeout(function () {
                                obj.editor.focus();
                            }, 100);
                        }
                    });
                }
            };

            obj.save = function () {
                if (!saveData(obj)) {
                    obj.disable();

                    Motion.inject(motion);
                    // save change motion object on server
                    Motion.save(motion, {method: 'PATCH'}).then(
                        function (success) {
                            if (versioning) {
                                $scope.showVersion(motion.getVersion(-1));
                            }
                            obj.revert();
                        },
                        function (error) {
                            // save error: revert all changes by restore
                            // (refresh) original motion object from server
                            Motion.refresh(motion);
                            obj.revert();
                            var message = '';
                            for (var e in error.data) {
                                message += e + ': ' + error.data[e] + ' ';
                            }
                            $scope.alert = {type: 'danger', msg: message, show: true};
                        }
                    );
                }
            };

            return obj;
        };
        return {
            createInstance: createInstance
        };
    }
])

.factory('MotionCommentsInlineEditing', [
    'MotionInlineEditing',
    'Editor',
    function (MotionInlineEditing, Editor) {
        var createInstances = function ($scope, motion) {
            var commentsInlineEditing = {
                editors: {}, // Map comment id to editor instance.
            };
            var options = Editor.getOptions('inline', 'YOffset');
            _.forEachRight($scope.noSpecialCommentsFields, function (field, id) {
                var inlineEditing = MotionInlineEditing.createInstance($scope, motion,
                    'view-original-comment-inline-editor-' + id, false, options,
                    function (obj) {
                        return motion['comment_' + id];
                    },
                    function (obj) {
                        if (obj.editor) {
                            motion['comment_' + id] = obj.editor.getData();
                        }
                    }
                );
                commentsInlineEditing.editors[id] = inlineEditing;
            });
            commentsInlineEditing.saveToolbarVisible = function () {
                return _.some(commentsInlineEditing.editors, function (instance) {
                    return instance.changed && instance.active;
                });
            };
            commentsInlineEditing.active = function (commentId) {
                return commentsInlineEditing.editors[commentId].active;
            };
            commentsInlineEditing.save = function () {
                _.forEach(commentsInlineEditing.editors, function (instance) {
                    instance.save();
                });
            };
            commentsInlineEditing.revert = function () {
                _.forEach(commentsInlineEditing.editors, function (instance) {
                    instance.revert();
                });
            };
            commentsInlineEditing.enable = function (commentId) {
                commentsInlineEditing.editors[commentId].enable();
            };
            commentsInlineEditing.disable = function (commentId) {
                commentsInlineEditing.editors[commentId].disable();
            };

            return commentsInlineEditing;
        };
        return {
            createInstances: createInstances,
        };
    }
])

.factory('ChangeRecommendationCreate', [
    'ngDialog',
    'ChangeRecommendationTitleForm',
    'ChangeRecommendationTextForm',
    function(ngDialog, ChangeRecommendationTitleForm, ChangeRecommendationTextForm) {
        var MODE_INACTIVE = 0,
            MODE_SELECTING_FROM = 1,
            MODE_SELECTING_TO = 2,

            TITLE_DUMMY_LINE_NUMBER = 0;

        var obj = {
            mode: MODE_INACTIVE,
            lineFrom: 1,
            lineTo: 2,
            html: '',
            reviewingHtml: ''
        };

        var $scope, motion, version;

        obj._getAffectedLineNumbers = function () {
            var changeRecommendations = motion.getTextChangeRecommendations(version.id),
                affectedLines = [];
            for (var i = 0; i < changeRecommendations.length; i++) {
                var change = changeRecommendations[i];
                for (var j = change.line_from; j < change.line_to; j++) {
                    affectedLines.push(j);
                }
            }
            return affectedLines;
        };

        // startCreating is called right at the beginning after the users interacts with the text for the first time.
        // This ensures all necessary nodes have been initialized
        obj.startCreating = function () {
            if (obj.mode > MODE_SELECTING_FROM || !motion.isAllowed('can_manage')) {
                return;
            }

            $(".tt_change_recommendation_create_help").removeClass("opened");
            var $lineNumbers = $(".motion-text-original .os-line-number"),
                $title = $(".motion-title .change-title");
            if ($lineNumbers.filter(".selectable").length === 0) {
                obj.mode = MODE_SELECTING_FROM;
                var alreadyAffectedLines = obj._getAffectedLineNumbers();
                $lineNumbers.each(function () {
                    var $this = $(this),
                        lineNumber = $this.data("line-number");
                    if (alreadyAffectedLines.indexOf(lineNumber) === -1) {
                        $(this).addClass("selectable");
                    }
                });
                if (alreadyAffectedLines.indexOf(TITLE_DUMMY_LINE_NUMBER) === -1) {
                    $title.addClass("selectable");
                }
            }
        };

        obj.cancelCreating = function (ev) {
            var $target = $(ev.target),
                query = ".line-numbers-outside .os-line-number.selectable";
            if (!$target.is(query) && $target.parents(query).length === 0) {
                obj.mode = MODE_INACTIVE;
                obj.lineFrom = 0;
                obj.lineTo = 0;
                $(".motion-text-original .os-line-number").removeClass("selected selectable");
                obj.startCreating();
            }
        };

        obj.setFromLine = function (line) {
            obj.mode = MODE_SELECTING_TO;
            obj.lineFrom = line;

            var alreadyAffectedLines = obj._getAffectedLineNumbers(),
                foundCollission = false;

            $(".motion-text-original .os-line-number").each(function () {
                var $this = $(this);
                if ($this.data("line-number") >= line && !foundCollission) {
                    if (alreadyAffectedLines.indexOf($this.data("line-number")) === -1) {
                        $(this).addClass("selectable");
                    } else {
                        $(this).removeClass("selectable");
                        foundCollission = true;
                    }
                } else {
                    $(this).removeClass("selectable");
                }
            });

            var tt_pos = $(".motion-text-original .line-number-" + line).position().top - 45;
            $(".tt_change_recommendation_create_help").css("top", tt_pos).addClass("opened");
        };

        obj.titleClicked = function () {
            ngDialog.open(ChangeRecommendationTitleForm.getCreateDialog(motion, version));

            obj.mode = MODE_INACTIVE;
            obj.lineFrom = 0;
            obj.lineTo = 0;
            $(".motion-text-original .os-line-number").removeClass("selected selectable");
            obj.startCreating();
        };

        obj.setToLine = function (line) {
            if (line < obj.lineFrom) {
                return;
            }
            obj.mode = MODE_INACTIVE;
            ngDialog.open(ChangeRecommendationTextForm.getCreateDialog(motion, version, obj.lineFrom, line + 1));

            obj.lineFrom = 0;
            obj.lineTo = 0;
            $(".motion-text-original .os-line-number").removeClass("selected selectable");
            obj.startCreating();
        };

        obj.lineClicked = function (ev) {
            if (obj.mode === MODE_INACTIVE) {
                return;
            }
            if (obj.mode === MODE_SELECTING_FROM) {
                obj.setFromLine($(ev.target).data("line-number"));
                $(ev.target).addClass("selected");
            } else if (obj.mode === MODE_SELECTING_TO) {
                obj.setToLine($(ev.target).data("line-number"));
            }
        };

        obj.mouseOver = function (ev) {
            if (obj.mode !== MODE_SELECTING_TO) {
                return;
            }
            var hoverLine = $(ev.target).data("line-number");
            $(".motion-text-original .os-line-number").each(function () {
                var line = $(this).data("line-number");
                if (line >= obj.lineFrom && line <= hoverLine) {
                    $(this).addClass("selected");
                } else {
                    $(this).removeClass("selected");
                }
            });
        };

        obj.setVersion = function (_motion, _version) {
            motion = _motion;
            version = motion.getVersion(_version);
        };

        obj.editTextDialog = function(change_recommendation) {
            ngDialog.open(ChangeRecommendationTextForm.getEditDialog(change_recommendation));
        };

        obj.editTitleDialog = function(change_recommendation) {
            ngDialog.open(ChangeRecommendationTitleForm.getEditDialog(change_recommendation));
        };

        obj.init = function (_scope, _motion) {
            $scope = _scope;
            motion = _motion;
            version = motion.getVersion($scope.version);

            var $content = $("#content");
            $content.on("click", ".line-numbers-outside .os-line-number.selectable", obj.lineClicked);
            $content.on("click", ".motion-title .change-title.selectable", obj.titleClicked);
            $content.on("click", obj.cancelCreating);
            $content.on("mouseover", ".line-numbers-outside .os-line-number.selectable", obj.mouseOver);
            $content.on("mouseover", ".motion-text-original, .motion-title", obj.startCreating);

            $scope.$watch(function () {
                return $scope.change_recommendations.length;
            }, function () {
                if (obj.mode === MODE_INACTIVE || obj.mode === MODE_SELECTING_FROM) {
                    // Recalculate the affected lines so we cannot select lines affected by a recommendation
                    // that has just been created
                    $(".motion-text-original .os-line-number").removeClass("selected selectable");
                    $(".motion-title .change-title").removeClass("selected selectable");
                    obj.startCreating();
                }
            });

            $scope.$on("$destroy", function () {
                obj.destroy();
            });
        };

        obj.destroy = function () {
            var $content = $("#content");
            $content.off("click", ".line-numbers-outside .os-line-number.selectable", obj.lineClicked);
            $content.off("click", ".motion-title .change-title.selectable", obj.titleClicked);
            $content.off("click", obj.cancelCreating);
            $content.off("mouseover", ".line-numbers-outside .os-line-number.selectable", obj.mouseOver);
            $content.off("mouseover", ".motion-text-original, .motion-title", obj.startCreating);
        };

        return obj;
    }
])

.factory('ChangeRecommendationView', [
    'Motion',
    'MotionChangeRecommendation',
    'Config',
    'lineNumberingService',
    'diffService',
    '$interval',
    '$timeout',
    function (Motion, MotionChangeRecommendation, Config, lineNumberingService, diffService, $interval, $timeout) {
        var $scope;

        var obj = {
            mode: 'original'
        };

        obj.diffFormatterCb = function (change, oldFragment, newFragment) {
            for (var i = 0; i < oldFragment.childNodes.length; i++) {
                diffService.addCSSClass(oldFragment.childNodes[i], 'delete');
            }
            for (i = 0; i < newFragment.childNodes.length; i++) {
                diffService.addCSSClass(newFragment.childNodes[i], 'insert');
            }
            var mergedFragment = document.createDocumentFragment(),
                diffSection = document.createElement('SECTION'),
                el;

            mergedFragment.appendChild(diffSection);
            diffSection.setAttribute('class', 'diff');
            diffSection.setAttribute('data-change-id', change.id);

            while (oldFragment.firstChild) {
                el = oldFragment.firstChild;
                oldFragment.removeChild(el);
                diffSection.appendChild(el);
            }
            while (newFragment.firstChild) {
                el = newFragment.firstChild;
                newFragment.removeChild(el);
                diffSection.appendChild(el);
            }

            return mergedFragment;
        };

        obj.delete = function (changeId) {
            MotionChangeRecommendation.destroy(changeId);
        };

        obj.rejectAll = function (motion) {
            var changeRecommendations = MotionChangeRecommendation.filter({
                'where': {'motion_version_id': {'==': motion.active_version}}
            });
            _.forEach(changeRecommendations, function(change) {
                change.rejected = true;
                change.saveStatus();
            });
        };

        obj.repositionOriginalAnnotations = function () {
            var $changeRecommendationList = $('.change-recommendation-list'),
                $lineNumberReference = $('.motion-text-original');

            $changeRecommendationList.children().each(function() {
                var $this = $(this),
                    lineFrom = $this.data('line-from'),
                    lineTo = ($this.data('line-to') - 1),
                    $lineFrom = $lineNumberReference.find('.line-number-' + lineFrom),
                    $lineTo = $lineNumberReference.find('.line-number-' + lineTo),
                    fromTop = $lineFrom.position().top + 3,
                    toTop = $lineTo.position().top + 20,
                    height = (toTop - fromTop);

                if (height < 10) {
                    height = 10;
                }

                // $lineFrom.position().top seems to depend on the scrolling position when the line numbers
                // have position: absolute. Maybe a bug in the used version of jQuery?
                // This cancels the effect.
                /*
                if ($lineNumberReference.hasClass('line-numbers-outside')) {
                    fromTop += window.scrollY;
                }
                */

                $this.css({ 'top': fromTop, 'height': height });
            });
        };

        obj.newVersionIncludingChanges = function (motion, version) {
            if (!motion.isAllowed('update')) {
                throw 'No permission to update motion';
            }

            var newHtml = motion.getTextByMode('agreed');
            motion.setTextStrippingLineBreaks(newHtml);

            Motion.inject(motion);
            // save change motion object on server
            Motion.save(motion, {method: 'PATCH'}).then(
                function (success) {
                    $scope.showVersion(motion.getVersion(-1));
                },
                function (error) {
                    // save error: revert all changes by restore
                    // (refresh) original motion object from server
                    Motion.refresh(motion);
                    var message = '';
                    for (var e in error.data) {
                        message += e + ': ' + error.data[e] + ' ';
                    }
                    $scope.alert = {type: 'danger', msg: message, show: true};
                }
            );
        };

        obj.scrollToDiffBox = function (changeId) {
            obj.mode = 'diff';
            $timeout(function() {
                var $diffBox = $('.diff-box-' + changeId);
                $('html, body').animate({
                    scrollTop: $diffBox.offset().top - 50
                }, 300);
            }, 0, false);
        };

        obj.init = function (_scope, viewMode) {
            $scope = _scope;
            $scope.$evalAsync(function() {
                obj.repositionOriginalAnnotations();
            });
            $scope.$watch(function() {
                return $('.change-recommendation-list').children().length;
            }, obj.repositionOriginalAnnotations);

            var checkGotoOriginal = function () {
                if ($scope.change_recommendations.length === 0 && $scope.title_change_recommendation === null) {
                    obj.mode = 'original';
                }
            };
            $scope.$watch(function () {
                return $scope.change_recommendations.length;
            }, checkGotoOriginal);
            $scope.$watch(function () {
                return $scope.title_change_recommendation;
            }, checkGotoOriginal);

            var sizeCheckerLastSize = null,
                sizeCheckerLastClass = null,
                sizeChecker = $interval(function() {
                    var $holder = $(".motion-text-original"),
                        newHeight = $holder.height(),
                        classes = $holder.attr("class");
                    if (newHeight != sizeCheckerLastSize || sizeCheckerLastClass != classes) {
                        sizeCheckerLastSize = newHeight;
                        sizeCheckerLastClass = classes;
                        obj.repositionOriginalAnnotations();
                    }
                }, 100, 0, false);

            $scope.$on('$destroy', function() {
                $interval.cancel(sizeChecker);
            });

            obj.mode = viewMode;
        };

        return obj;
    }
]);

}());

(function () {

"use strict";

angular.module('OpenSlidesApp.motions.pdf', ['OpenSlidesApp.core.pdf'])

.factory('MotionContentProvider', [
    '$q',
    'operator',
    'gettextCatalog',
    'PDFLayout',
    'PdfMakeConverter',
    'ImageConverter',
    'HTMLValidizer',
    'Category',
    'Config',
    'Motion',
    'MotionComment',
    'OpenSlidesSettings',
    function($q, operator, gettextCatalog, PDFLayout, PdfMakeConverter, ImageConverter,
        HTMLValidizer, Category, Config, Motion, MotionComment, OpenSlidesSettings) {
        /**
         * Provides the content as JS objects for Motions in pdfMake context
         * @constructor
         */

        var createInstance = function(motion, motionVersion, params) {
            params = _.clone(params || {}); // Clone this to avoid sideeffects.
            _.defaults(params, {
                changeRecommendationMode: Config.get('motions_recommendation_text_mode').value,
                lineNumberMode: Config.get('motions_default_line_numbering').value,
                include: {
                    text: true,
                    reason: true,
                    state: true,
                    submitters: true,
                    votingresult: true,
                    motionBlock: true,
                    origin: true,
                    recommendation: true,
                },
                includeComments: {},
            });

            var converter, imageMap = {};

            // Query all image sources from motion text and reason
            var getImageSources = function () {
                var text = motion.getTextByMode(params.changeRecommendationMode, null);
                var reason = motion.getReason();
                var comments = '';
                _.forEach(params.includeComments, function (ok, id) {
                    if (ok && motion.comments[id]) {
                        comments += HTMLValidizer.validize(motion.comments[id]);
                    }
                });
                var content = HTMLValidizer.validize(text) + HTMLValidizer.validize(motion.getReason()) + comments;
                var map = Function.prototype.call.bind([].map);
                return map($(content).find('img'), function(element) {
                    return element.getAttribute('src');
                });
            };

            // title
            var identifier = motion.identifier ? ' ' + motion.identifier : '';
            var titlePlain = motion.getTitleWithChanges(params.changeRecommendationMode, motionVersion);
            var title = PDFLayout.createTitle(gettextCatalog.getString('Motion') + identifier + ': ' + titlePlain);

            // subtitle and sequential number
            var subtitleLines = [];
            if (motion.parent_id) {
                var parentMotion = Motion.get(motion.parent_id);
                subtitleLines.push(
                    gettextCatalog.getString('Amendment of motion') + ': ' +
                    (parentMotion.identifier ? parentMotion.identifier : parentMotion.getTitle())
                );
            }
            if (Config.get('motions_export_sequential_number').value) {
                subtitleLines.push(gettextCatalog.getString('Sequential number') + ': ' +
                    motion.getSequentialNumber());
            }
            var subtitle = PDFLayout.createSubtitle(subtitleLines);

            // meta data table
            var metaTable = function() {
                var metaTableBody = [];

                // submitters
                var submitters = _.map(motion.submitters, function (submitter) {
                    return submitter.get_full_name();
                }).join(', ');
                if (params.include.submitters) {
                    metaTableBody.push([
                        {
                            text: gettextCatalog.getString('Submitters') + ':',
                            style: ['bold', 'grey'],
                        },
                        {
                            text: submitters,
                            style: 'grey'
                        }
                    ]);
                }

                // state
                if (params.include.state) {
                    metaTableBody.push([
                        {
                            text: gettextCatalog.getString('State') + ':',
                            style: ['bold', 'grey']
                        },
                        {
                            text: motion.getStateName(),
                            style: 'grey'
                        }
                    ]);
                }

                // recommendation
                if (params.include.recommendation && motion.getRecommendationName()) {
                    metaTableBody.push([
                        {
                            text: Config.get('motions_recommendations_by').value + ':',
                            style: ['bold', 'grey']
                        },
                        {
                            text: motion.getRecommendationName(),
                            style: 'grey'
                        }
                    ]);
                }

                // category
                if (motion.category) {
                    metaTableBody.push([
                        {
                            text: gettextCatalog.getString('Category') + ':',
                            style: ['bold', 'grey'] },
                        {
                            text: motion.category.prefix + ' - ' + motion.category.name,
                            style: 'grey'
                        }
                    ]);
                }

                // motion block
                if (params.include.motionBlock && motion.motionBlock) {
                    metaTableBody.push([
                        {
                            text: gettextCatalog.getString('Motion block') + ':',
                            style: ['bold', 'grey'] },
                        {
                            text: motion.motionBlock.title,
                            style: 'grey'
                        }
                    ]);
                }

                // origin
                if (params.include.origin && motion.origin) {
                    metaTableBody.push([
                        {
                            text: gettextCatalog.getString('Origin') + ':',
                            style: ['bold', 'grey'] },
                        {
                            text: motion.origin,
                            style: 'grey'
                        }
                    ]);
                }

                // voting result
                if (params.include.votingresult && motion.polls.length > 0 && motion.polls[0].has_votes) {
                    var column1 = [];
                    var column2 = [];
                    var column3 = [];
                    motion.polls.map(function(poll, index) {
                        if (poll.has_votes) {
                            // votenumber
                            if (motion.polls.length > 1) {
                                column1.push(index + 1 + '. ' + gettextCatalog.getString('Vote'));
                                column2.push('');
                                column3.push('');
                            }
                            // yes
                            var yes = poll.getVote(poll.yes, 'yes');
                            column1.push(gettextCatalog.getString('Yes') + ':');
                            column2.push(yes.value);
                            column3.push(yes.percentStr);
                            // no
                            var no = poll.getVote(poll.no, 'no');
                            column1.push(gettextCatalog.getString('No') + ':');
                            column2.push(no.value);
                            column3.push(no.percentStr);
                            // abstain
                            var abstain = poll.getVote(poll.abstain, 'abstain');
                            column1.push(gettextCatalog.getString('Abstain') + ':');
                            column2.push(abstain.value);
                            column3.push(abstain.percentStr);
                            // votes valid
                            if (poll.votesvalid) {
                                var valid = poll.getVote(poll.votesvalid, 'votesvalid');
                                column1.push(gettextCatalog.getString('Valid votes') + ':');
                                column2.push(valid.value);
                                column3.push(valid.percentStr);
                            }
                            // votes invalid
                            if (poll.votesvalid) {
                                var invalid = poll.getVote(poll.votesinvalid, 'votesinvalid');
                                column1.push(gettextCatalog.getString('Invalid votes') + ':');
                                column2.push(invalid.value);
                                column3.push(invalid.percentStr);
                            }
                            // votes cast
                            if (poll.votescast) {
                                var cast = poll.getVote(poll.votescast, 'votescast');
                                column1.push(gettextCatalog.getString('Votes cast') + ':');
                                column2.push(cast.value);
                                column3.push(cast.percentStr);
                            }
                        }
                    });
                    metaTableBody.push([
                        {
                            text: gettextCatalog.getString('Voting result') + ':',
                            style: ['bold', 'grey']
                        },
                        {
                            columns: [
                                {
                                    text: column1.join('\n'),
                                    width: 'auto'
                                },
                                {
                                    text: column2.join('\n'),
                                    width: 'auto',
                                    alignment: 'right'
                                },
                                {
                                    text: column3.join('\n'),
                                    width: 'auto',
                                    alignment: 'right'
                                },
                            ],
                            columnGap: 7,
                            style: 'grey'
                        }
                    ]);
                }

                // summary of change recommendations (for motion diff version only)
                if (params.changeRecommendationMode === "diff" && motion.changeRecommendations.length) {
                    var columnLineNumbers = [];
                    var columnChangeType = [];
                    angular.forEach(_.orderBy(motion.changeRecommendations, ['line_from']), function(change) {
                        if (change.isTitleRecommendation()) {
                            columnLineNumbers.push(
                                gettextCatalog.getString('Title') + ': '
                            );
                        } else {
                            // line numbers column
                            var line;
                            if (change.line_from >= change.line_to - 1) {
                                line = change.line_from;
                            } else {
                                line = change.line_from + ' - ' + (change.line_to - 1);
                            }
                            columnLineNumbers.push(
                                gettextCatalog.getString('Line') + ' ' + line + ': '
                            );
                        }
                        // change type column
                        if (change.getType(motion.getVersion(motionVersion).text) === 0) {
                            columnChangeType.push(gettextCatalog.getString("Replacement"));
                        } else if (change.getType(motion.getVersion(motionVersion).text) === 1) {
                            columnChangeType.push(gettextCatalog.getString("Insertion"));
                        } else if (change.getType(motion.getVersion(motionVersion).text) === 2) {
                            columnChangeType.push(gettextCatalog.getString("Deletion"));
                        } else if (change.getType(motion.getVersion(motionVersion).text) === 3) {
                            columnChangeType.push(change.other_description);
                        }
                    });
                    metaTableBody.push([
                        {
                            text: gettextCatalog.getString('Summary of change recommendations'),
                            style: ['bold', 'grey']
                        },
                        {
                            columns: [
                                {
                                    text: columnLineNumbers.join('\n'),
                                    width: 'auto'
                                },
                                {
                                    text: columnChangeType.join('\n'),
                                    width: 'auto'
                                }
                            ],
                            columnGap: 7,
                            style: 'grey'
                        }
                    ]);
                }

                if (metaTableBody.length) {
                    // build table
                    // Used placeholder for 'layout' functions whiche are
                    // replaced by lineWitdh/lineColor function in pfd-worker.js.
                    // TODO: Remove placeholder and us static values for LineWidth and LineColor
                    // if pdfmake has fixed this.
                    var metaTable = {
                        table: {
                            widths: ['35%','65%'],
                            body: metaTableBody,
                        },
                        margin: [0, 0, 0, 20],
                        layout: '{{motion-placeholder-to-insert-functions-here}}'
                    };
                    return metaTable;
                } else {
                    return {};
                }
            };

            // motion title
            var motionTitle = function() {
                if (params.include.text) {
                    return [{
                        text: titlePlain,
                        style: 'heading3'
                    }];
                }
            };

            // motion preamble
            var motionPreamble = function () {
                if (params.include.text) {
                    return {
                        text: Config.translate(Config.get('motions_preamble').value),
                        margin: [0, 10, 0, 0]
                    };
                }
            };

            var escapeHtml = function(text) {
                return text.replace(/&/, "&amp;").replace(/</, "&lt;").replace(/>/, "&gt;");
            };

            // motion text (with line-numbers)
            var motionText = function() {
                if (params.include.text) {
                    var motionTextContent = '';
                    var titleChange = motion.getTitleChangeRecommendation();
                    if (params.changeRecommendationMode === 'diff' && titleChange) {
                        motionTextContent += '<p><strong>' + gettextCatalog.getString('New title') + ':</strong> ' +
                            escapeHtml(titleChange.text) + '</p>';
                    }
                    motionTextContent += motion.getTextByMode(params.changeRecommendationMode, motionVersion);
                    return converter.convertHTML(motionTextContent, params.lineNumberMode);
                }
            };

            // motion reason heading
            var motionReason = function() {
                if (params.include.reason) {
                    var reason = [];
                    if (motion.getReason(motionVersion)) {
                        reason.push({
                            text:  gettextCatalog.getString('Reason'),
                            style: 'heading3',
                            marginTop: 25,
                        });
                        var width;
                        if (params.lineNumberMode == 'outside') {
                            width = '80%';
                        } else {
                            width = '100%';
                        }
                        reason.push({
                            columns: [
                                {
                                    width: width,
                                    stack: converter.convertHTML(motion.getReason(motionVersion), 'none'),
                                },
                            ]
                        });
                    }
                    return reason;
                }
            };

            // motion comments handling
            var motionComments = function () {
                if (_.keys(params.includeComments).length !== 0) {
                    var fields = MotionComment.getNoSpecialCommentsFields();
                    var comments = [];
                    _.forEach(params.includeComments, function (ok, id) {
                        if (ok && motion.comments[id]) {
                            var title = fields[id].name;
                            if (!fields[id].public) {
                                title += ' (' + gettextCatalog.getString('internal') + ')';
                            }
                            comments.push({
                                text: title,
                                style: 'heading3',
                                marginTop: 25,
                            });
                            comments.push(converter.convertHTML(motion.comments[id]));
                        }
                    });
                    return comments;
                }
            };

            // Generates content as a pdfmake consumable
            var getContent = function() {
                var content = [
                    title,
                    subtitle,
                    metaTable(),
                    motionTitle(),
                    motionPreamble(),
                    motionText(),
                ];
                var reason = motionReason();
                if (reason) {
                    content.push(reason);
                }
                var comments = motionComments();
                if (comments) {
                    content.push(comments);
                }
                return content;
            };

            // getters
            var getTitle = function() {
                return motion.getTitle(motionVersion);
            };

            var getIdentifier = function() {
                return motion.identifier ? motion.identifier : '';
            };

            var getCategory = function() {
                return motion.category;
            };

            var getImageMap = function() {
                return imageMap;
            };

            return $q(function (resolve, reject) {
                ImageConverter.toBase64(getImageSources()).then(function (_imageMap) {
                    imageMap = _imageMap;
                    converter = PdfMakeConverter.createInstance(_imageMap);
                    resolve({
                        getContent: getContent,
                        getTitle: getTitle,
                        getIdentifier: getIdentifier,
                        getCategory: getCategory,
                        getImageMap: getImageMap,
                    });
                }, reject);
            });
        };

        return {
            createInstance: createInstance
        };
    }
])

.factory('MotionPartialContentProvider', [
    '$q',
    'gettextCatalog',
    'Config',
    'PDFLayout',
    'PdfMakeConverter',
    'ImageConverter',
    'HTMLValidizer',
    function ($q, gettextCatalog, Config, PDFLayout, PdfMakeConverter, ImageConverter, HTMLValidizer) {
        /*
         * content should be an array of content blocks. Each content is an object providing a
         * heading and a text. E.g.
         * [{heading: 'comment1', text: '<html in here>'}, {heading: ...}, ...]
         * */
        var createInstance = function (motion, content) {

            var converter, imageMap = {};

            // Query all image sources from the content
            var getImageSources = function () {
                var imageSources = [];
                _.forEach(content, function (contentBlock) {
                    var html = HTMLValidizer.validize(contentBlock.text);
                    imageSources = imageSources.concat(_.map($(html).find('img'), function(element) {
                        return element.getAttribute('src');
                    }));
                });
                return imageSources;
            };

            // title
            var identifier = motion.identifier ? ' ' + motion.identifier : '';
            var title = PDFLayout.createTitle(
                    gettextCatalog.getString('Motion') + identifier + ': ' + motion.getTitle()
            );

            // subtitle and sequential number
            var subtitleLines = [];
            if (motion.parent_id) {
                var parentMotion = Motion.get(motion.parent_id);
                subtitleLines.push(
                    gettextCatalog.getString('Amendment of motion') + ': ' +
                    (parentMotion.identifier ? parentMotion.identifier : parentMotion.getTitle())
                );
            }
            if (Config.get('motions_export_sequential_number').value) {
                subtitleLines.push(gettextCatalog.getString('Sequential number') + ': ' +  motion.id);
            }
            var subtitle = PDFLayout.createSubtitle(subtitleLines);

            // meta data table
            var metaTable = function() {
                var metaTableBody = [];

                // submitters
                var submitters = _.map(motion.submitters, function (submitter) {
                    return submitter.get_full_name();
                }).join(', ');
                metaTableBody.push([
                    {
                        text: gettextCatalog.getString('Submitters') + ':',
                        style: ['bold', 'grey'],
                    },
                    {
                        text: submitters,
                        style: 'grey'
                    }
                ]);

                // state
                metaTableBody.push([
                    {
                        text: gettextCatalog.getString('State') + ':',
                        style: ['bold', 'grey']
                    },
                    {
                        text: motion.getStateName(),
                        style: 'grey'
                    }
                ]);

                // recommendation
                if (motion.getRecommendationName()) {
                    metaTableBody.push([
                        {
                            text: Config.get('motions_recommendations_by').value + ':',
                            style: ['bold', 'grey']
                        },
                        {
                            text: motion.getRecommendationName(),
                            style: 'grey'
                        }
                    ]);
                }

                // category
                if (motion.category) {
                    metaTableBody.push([
                        {
                            text: gettextCatalog.getString('Category') + ':',
                            style: ['bold', 'grey'] },
                        {
                            text: motion.category.prefix + ' - ' + motion.category.name,
                            style: 'grey'
                        }
                    ]);
                }

                // build table
                // Used placeholder for 'layout' functions whiche are
                // replaced by lineWitdh/lineColor function in pfd-worker.js.
                // TODO: Remove placeholder and us static values for LineWidth and LineColor
                // if pdfmake has fixed this.
                var metaTableJsonString = {
                    table: {
                        widths: ['30%','70%'],
                        body: metaTableBody,
                    },
                    margin: [0, 0, 0, 20],
                    layout: '{{motion-placeholder-to-insert-functions-here}}'
                };
                return metaTableJsonString;
            };

            var getContentBlockData = function (block) {
                var data = [];
                data.push({
                    text:  block.heading,
                    style: 'heading3',
                    marginTop: 25,
                });
                data.push(converter.convertHTML(block.text));
                return data;
            };

            // Generates content as a pdfmake consumable
            var getContent = function() {
                var pdfContent = [
                    title,
                    subtitle,
                    metaTable(),
                ];
                _.forEach(content, function (contentBlock) {
                    pdfContent.push(getContentBlockData(contentBlock));
                });
                return pdfContent;
            };

            var getImageMap = function () {
                return imageMap;
            };

            return $q(function (resolve, reject) {
                ImageConverter.toBase64(getImageSources()).then(function (_imageMap) {
                    imageMap = _imageMap;
                    converter = PdfMakeConverter.createInstance(_imageMap);
                    resolve({
                        getContent: getContent,
                        getImageMap: getImageMap,
                    });
                }, reject);
            });
        };

        return {
            createInstance: createInstance
        };
    }
])

.factory('PollContentProvider', [
    '$q',
    'PDFLayout',
    'gettextCatalog',
    'Config',
    'User',
    'ImageConverter',
    function($q, PDFLayout, gettextCatalog, Config, User, ImageConverter) {
        /**
        * Generates a content provider for polls
        * @constructor
        * @param {string} title - title of poll
        * @param {string} id - if of poll
        */
        var createInstance = function(title, id) {

            var logoBallotPaperUrl = Config.get('logo_pdf_ballot_paper').value.path;
            var imageMap = {};

            // PDF header
            var header = function() {
                var columns = [];

                var text = Config.get('general_event_name').value;
                columns.push({
                    text: text,
                    fontSize: 8,
                    alignment: 'left',
                    width: '60%'
                });

                // logo
                if (logoBallotPaperUrl) {
                    columns.push({
                        image: logoBallotPaperUrl,
                        fit: [90,25],
                        alignment: 'right',
                        width: '40%'
                    });
                }
                return {
                    color: '#555',
                    fontSize: 10,
                    margin: [30, 10, 10, -10], // [left, top, right, bottom]
                    columns: columns,
                    columnGap: 5
                };
            };

            /**
            * Returns a single section on the ballot paper
            * @function
            */
            var createSection = function() {
                var sheetend = 40;
                return {
                    stack: [
                        header(),
                        {
                            text: gettextCatalog.getString('Motion') + ' ' + id,
                            style: 'title',
                        },
                        {
                            text: title,
                            style: 'description'
                        },
                        PDFLayout.createBallotEntry(gettextCatalog.getString('Yes')),
                        PDFLayout.createBallotEntry(gettextCatalog.getString('No')),
                        PDFLayout.createBallotEntry(gettextCatalog.getString('Abstain')),
                    ],
                    margin: [0, 0, 0, sheetend],
                };
            };

            /**
            * Returns Content for single motion
            * @function
            * @param {string} id - if of poll
            */
            var getContent = function() {
                var content = [];
                var amount;
                var amount_method = Config.get('motions_pdf_ballot_papers_selection').value;
                switch (amount_method) {
                        case 'NUMBER_OF_ALL_PARTICIPANTS':
                            amount = User.getAll().length;
                            break;
                        case 'NUMBER_OF_DELEGATES':
                            //TODO: assumption that DELEGATES is always group id 2. This may not be true
                            var group_id = 2;
                            amount = User.filter({where: {'groups_id': {contains:group_id} }}).length;
                            break;
                        case 'CUSTOM_NUMBER':
                            amount = Config.get('motions_pdf_ballot_papers_number').value;
                            break;
                        default:
                            // should not happen.
                            amount = 0;
                }
                var fullpages = Math.floor(amount / 8);

                for (var i=0; i < fullpages; i++) {
                    content.push({
                        table: {
                            headerRows: 1,
                            widths: ['*', '*'],
                            body: [
                                [createSection(), createSection()],
                                [createSection(), createSection()],
                                [createSection(), createSection()],
                                [createSection(), createSection()]
                            ],
                            pageBreak: 'after'
                        },
                        layout: PDFLayout.getBallotLayoutLines(),
                        rowsperpage: 4
                    });
                }
                amount = amount  - (fullpages * 8);
                if (amount > 0) {
                    var partialpagebody = [];
                    while (amount > 1) {
                        partialpagebody.push([createSection(), createSection()]);
                        amount -=2;
                    }
                    if (amount == 1) {
                        partialpagebody.push([createSection(), '']);
                    }
                    content.push({
                        table: {
                            headerRows: 1,
                            widths: ['50%', '50%'],
                            body: partialpagebody
                        },
                        layout: PDFLayout.getBallotLayoutLines(),
                        rowsperpage: 4
                    });
                }
                return content;
            };

            var getImageMap = function () {
                return imageMap;
            };

            return $q(function (resolve, reject) {
                var imageSources = [
                    logoBallotPaperUrl,
                ];
                ImageConverter.toBase64(imageSources).then(function (_imageMap) {
                    imageMap = _imageMap;
                    resolve({
                        getContent: getContent,
                        getImageMap: getImageMap,
                    });
                }, reject);
            });
        };
        return {
            createInstance: createInstance
        };
    }
])

.factory('MotionCatalogContentProvider', [
    'gettextCatalog',
    'PDFLayout',
    'Category',
    'Config',
    function(gettextCatalog, PDFLayout, Category, Config) {
        /**
        * Constructor
        * @function
        * @param {object} allMotions - A sorted array of all motions to parse
        */
        var createInstance = function(allMotions) {

            var title = PDFLayout.createTitle(
                Config.translate(Config.get('motions_export_title').value)
            );

            var createPreamble = function() {
                var preambleText = Config.get('motions_export_preamble').value;
                if (preambleText) {
                    return {
                        text: preambleText,
                        style: "preamble"
                    };
                } else {
                    return "";
                }
            };

            var createTOContent = function() {
                var heading = {
                    text: gettextCatalog.getString("Table of contents"),
                    style: "heading2"
                };

                var toc = [];
                angular.forEach(allMotions, function(motion) {
                    var identifier = motion.getIdentifier() ? motion.getIdentifier() : '';
                    toc.push(
                        {
                            columns: [
                                {
                                    text: identifier,
                                    style: 'tableofcontent',
                                    width: 70
                                },
                                {
                                    text: motion.getTitle(),
                                    style: 'tableofcontent'
                                }
                            ]
                        }
                    );
                });

                return [
                    heading,
                    toc,
                    PDFLayout.addPageBreak()
                ];
            };

            // function to create the table of catergories (if any)
            var createTOCategories = function() {
                var categories = [];
                _.forEach(allMotions, function (motion) {
                    var category = motion.getCategory();
                    if (category) {
                        categories.push(category);
                    }
                });
                var sortKey = Config.get('motions_export_category_sorting').value;
                categories = _.orderBy(_.uniqBy(categories, 'id'), [sortKey]);
                if (categories.length > 1) {
                    var heading = {
                        text: gettextCatalog.getString('Categories'),
                        style: 'heading2',
                    };

                    var toc = [];
                    angular.forEach(categories, function(cat) {
                        toc.push(
                            {
                                columns: [
                                    {
                                        text: cat.prefix,
                                        style: 'tableofcontent',
                                        width: 50
                                    },
                                    {
                                        text: cat.name,
                                        style: 'tableofcontent'
                                    }
                                ]
                            }
                        );
                    });

                    return [
                        heading,
                        toc,
                        PDFLayout.addPageBreak()
                    ];
                } else {
                    // if there are no categories, return "empty string"
                    // pdfmake takes "null" literally and throws an error
                    return "";
                }
            };

            // returns the pure content of the motion, parseable by pdfmake
            var getContent = function() {
                var motionContent = [];
                _.forEach(allMotions, function(motion, key) {
                    motionContent.push(motion.getContent());
                    if (key < allMotions.length - 1) {
                        motionContent.push(PDFLayout.addPageBreak());
                    }
                });
                var content = [];
                // print extra data (title, preamble, categories, toc) only for more than 1 motion
                if (allMotions.length > 1) {
                    content.push(
                        title,
                        createPreamble(),
                        createTOCategories(),
                        createTOContent()
                    );
                }
                content.push(motionContent);
                return content;
            };

            var getImageMap = function () {
                var imageMap = {};
                _.forEach(allMotions, function (motion) {
                    _.forEach(motion.getImageMap(), function (data, path) {
                        if (!imageMap[path]) {
                            imageMap[path] = data;
                        }
                    });
                });
                return imageMap;
            };

            return {
                getContent: getContent,
                getImageMap: getImageMap,
            };
        };

        return {
            createInstance: createInstance
        };
    }
])

.factory('MotionPdfExport', [
    '$http',
    '$q',
    'operator',
    'Config',
    'gettextCatalog',
    'MotionChangeRecommendation',
    'HTMLValidizer',
    'PdfMakeConverter',
    'MotionContentProvider',
    'MotionCatalogContentProvider',
    'PdfMakeDocumentProvider',
    'PollContentProvider',
    'PdfMakeBallotPaperProvider',
    'MotionPartialContentProvider',
    'PdfCreate',
    'PDFLayout',
    'PersonalNoteManager',
    'MotionComment',
    'Messaging',
    'FileSaver',
    function ($http, $q, operator, Config, gettextCatalog, MotionChangeRecommendation, HTMLValidizer,
        PdfMakeConverter, MotionContentProvider, MotionCatalogContentProvider, PdfMakeDocumentProvider,
        PollContentProvider, PdfMakeBallotPaperProvider, MotionPartialContentProvider, PdfCreate,
        PDFLayout, PersonalNoteManager, MotionComment, Messaging, FileSaver) {
        return {
            getDocumentProvider: function (motions, params, singleMotion) {
                params = _.clone(params || {}); // Clone this to avoid sideeffects.

                if (singleMotion) {
                    _.defaults(params, {
                        version: motions.active_version,
                    });
                    motions = [motions];
                }

                //save the arrays of all motions to an array
                angular.forEach(motions, function (motion) {
                    if (singleMotion) {
                        motion.changeRecommendations = MotionChangeRecommendation.filter({
                            'where': {'motion_version_id': {'==': params.version}}
                        });
                    } else {
                        motion.changeRecommendations = MotionChangeRecommendation.filter({
                            'where': {'motion_version_id': {'==': motion.active_version}}
                        });
                    }
                });

                var motionContentProviderArray = [];
                var motionContentProviderPromises = _.map(motions, function (motion) {
                    var version = (singleMotion ? params.version : motion.active_version);
                    return $q(function (resolve, reject) {
                        MotionContentProvider.createInstance(
                            motion, version, params
                        ).then(function (contentProvider) {
                            motionContentProviderArray.push(contentProvider);
                            resolve();
                        }, reject);
                    });
                });

                return $q(function (resolve, reject) {
                    $q.all(motionContentProviderPromises).then(function() {
                        var documentProviderPromise;
                        if (singleMotion) {
                            documentProviderPromise = PdfMakeDocumentProvider.createInstance(motionContentProviderArray[0]);
                        } else {
                            var motionCatalogContentProvider = MotionCatalogContentProvider.createInstance(motionContentProviderArray);
                            documentProviderPromise = PdfMakeDocumentProvider.createInstance(motionCatalogContentProvider);
                        }
                        documentProviderPromise.then(function (documentProvider) {
                            resolve(documentProvider);
                        }, reject);
                    }, reject);
                });
            },
            export: function (motions, params, singleMotion) {
                params = params || {};
                _.defaults(params, {
                    filename: gettextCatalog.getString('motions') + '.pdf',
                });
                this.getDocumentProvider(motions, params, singleMotion).then(
                    function (documentProvider) {
                        PdfCreate.download(documentProvider, params.filename);
                    }, function (error) {
                        Messaging.addMessage(error.msg, 'error');
                    }
                );
            },
            exportZip: function (motions, params) {
                var messageId = Messaging.addMessage('<i class="fa fa-spinner fa-pulse fa-lg spacer-right"></i>' +
                    gettextCatalog.getString('Generating PDFs and ZIP archive') + ' ...', 'info');
                var zipFilename = params.filename || gettextCatalog.getString('motions') + '.zip';
                params.filename = void 0; // clear this, so we do not override the default filenames for each pdf.

                var self = this;
                var usedFilenames = [];
                var docMap = {};
                var docPromises = _.map(motions, function (motion) {
                    var identifier = motion.identifier ? '-' + motion.identifier : '';
                    var filename = gettextCatalog.getString('Motion') + identifier;

                    // If the filename is already in use, try to append a number to it (like '(2)')
                    if (_.includes(usedFilenames, filename)) {
                        var i = 1;
                        var filenameWithNumber = filename;
                        while(_.includes(usedFilenames, filenameWithNumber)) {
                            filenameWithNumber = filename + ' (' + i + ')';
                            i++;
                        }
                        filename = filenameWithNumber;
                    }
                    usedFilenames.push(filename);
                    filename += '.pdf';

                    return $q(function (resolve, reject) {
                        // get documentProvider for every motion.
                        self.getDocumentProvider(motion, params, true).then(function (documentProvider) {
                            docMap[filename] = documentProvider;
                            resolve();
                        }, reject);
                    });
                });
                $q.all(docPromises).then(function () {
                    PdfCreate.getBase64FromMultipleDocuments(docMap).then(function (pdfMap) {
                        var zip = new JSZip();
                        _.forEach(pdfMap, function (data, filename) {
                            zip.file(filename, data, {base64: true});
                        });
                        Messaging.createOrEditMessage(messageId, '<i class="fa fa-check fa-lg spacer-right"></i>' +
                            gettextCatalog.getString('ZIP successfully generated.'), 'success', {timeout: 3000});
                        zip.generateAsync({type: 'blob'}).then(function (content) {
                            FileSaver.saveAs(content, zipFilename);
                        });
                    }, function (error) {
                        Messaging.createOrEditMessage(messageId, '<i class="fa fa-exclamation-triangle fa-lg ' +
                            'spacer-right"></i>' + gettextCatalog.getString('Error while generating ZIP file') +
                            ': <code>' + error + '</code>', 'error');
                    });
                }, function (error) {
                    Messaging.createOrEditMessage(messageId, error.msg, 'error');
                });
            },
            createPollPdf: function (motion, version) {
                var id = motion.identifier.replace(' ', '');
                var title = motion.getTitle(version);
                var filename = gettextCatalog.getString('Motion') + '-' + id + '-' + gettextCatalog.getString('ballot-paper') + '.pdf';
                PollContentProvider.createInstance(title, id).then(function (pollContentProvider) {
                    var documentProvider = PdfMakeBallotPaperProvider.createInstance(pollContentProvider);
                    PdfCreate.download(documentProvider, filename);
                }, function (error) {
                    Messaging.addMessage(error.msg, 'error');
                });
            },
            exportPersonalNote: function (motion, filename) {
                var personalNote = PersonalNoteManager.getNote(motion);
                var content = [{
                    heading: gettextCatalog.getString('Personal note'),
                    text: personalNote ? personalNote.note : '',
                }];
                MotionPartialContentProvider.createInstance(motion, content).then(function (contentProvider) {
                    PdfMakeDocumentProvider.createInstance(contentProvider).then(function (documentProvider) {
                        PdfCreate.download(documentProvider, filename);
                    }, function (error) {
                        Messaging.addMessage(error.msg, 'error');
                    });
                }, function (error) {
                    Messaging.addMessage(error.msg, 'error');
                });
            },
            exportComment: function (motion, commentId, filename) {
                var field = MotionComment.getNoSpecialCommentsFields()[commentId];
                if (field && motion.comments[commentId]) {
                    var title = field.name;
                    if (!field.public) {
                        title += ' (' + gettextCatalog.getString('internal') + ')';
                    }
                    var content = [{
                        heading: title,
                        text: motion.comments[commentId],
                    }];
                    MotionPartialContentProvider.createInstance(motion, content).then(function (contentProvider) {
                        PdfMakeDocumentProvider.createInstance(contentProvider).then(function (documentProvider) {
                            PdfCreate.download(documentProvider, filename);
                        }, function (error) {
                            Messaging.addMessage(error.msg, 'error');
                        });
                    }, function (error) {
                        Messaging.addMessage(error.msg, 'error');
                    });
                }
            },
        };
    }
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.motions.projector', [
    'OpenSlidesApp.motions',
    'OpenSlidesApp.motions.motionBlockProjector',
])

.config([
    'slidesProvider',
    function(slidesProvider) {
        slidesProvider.registerSlide('motions/motion', {
            template: 'static/templates/motions/slide_motion.html',
        });
    }
])

.controller('SlideMotionCtrl', [
    '$scope',
    'Motion',
    'MotionChangeRecommendation',
    'User',
    function($scope, Motion, MotionChangeRecommendation, User) {
        // Attention! Each object that is used here has to be dealt on server side.
        // Add it to the coresponding get_requirements method of the ProjectorElement
        // class.
        var id = $scope.element.id;
        $scope.mode = $scope.element.mode || 'original';

        Motion.bindOne(id, $scope, 'motion');
        User.bindAll({}, $scope, 'users');

        $scope.$watch(function () {
            return MotionChangeRecommendation.lastModified();
        }, function () {
            $scope.change_recommendations = [];
            $scope.title_change_recommendation = null;
            if ($scope.motion) {
                MotionChangeRecommendation.filter({
                    'where': {'motion_version_id': {'==': $scope.motion.active_version}}
                }).forEach(function(change) {
                    if (change.isTextRecommendation()) {
                        $scope.change_recommendations.push(change);
                    }
                    if (change.isTitleRecommendation()) {
                        $scope.title_change_recommendation = change;
                    }
                });
            }
        });
    }
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.motions.site', [
    'OpenSlidesApp.motions',
    'OpenSlidesApp.motions.motionservices',
    'OpenSlidesApp.poll.majority',
    'OpenSlidesApp.core.pdf',
    'OpenSlidesApp.motions.docx',
    'OpenSlidesApp.motions.pdf',
    'OpenSlidesApp.motions.csv',
])

.config([
    'mainMenuProvider',
    'gettext',
    function (mainMenuProvider, gettext) {
        mainMenuProvider.register({
            'ui_sref': 'motions.motion.list',
            'img_class': 'file-text',
            'title': gettext('Motions'),
            'weight': 300,
            'perm': 'motions.can_see',
        });
    }
])

.config([
    'SearchProvider',
    'gettext',
    function (SearchProvider, gettext) {
        SearchProvider.register({
            'verboseName': gettext('Motions'),
            'collectionName': 'motions/motion',
            'urlDetailState': 'motions.motion.detail',
            'weight': 300,
        });
    }
])

.config([
    '$stateProvider',
    'gettext',
    function($stateProvider, gettext) {
        $stateProvider
            .state('motions', {
                url: '/motions',
                abstract: true,
                template: "<ui-view/>",
                data: {
                    title: gettext('Motions'),
                    basePerm: 'motions.can_see',
                },
            })
            .state('motions.motion', {
                abstract: true,
                template: "<ui-view/>",
            })
            .state('motions.motion.list', {})
            .state('motions.motion.detail', {
                resolve: {
                    motionId: ['$stateParams', function($stateParams) {
                        return $stateParams.id;
                    }],
                }
            })
            // redirects to motion detail and opens motion edit form dialog, uses edit url,
            // used by ui-sref links from agenda only
            // (from motion controller use MotionForm factory instead to open dialog in front of
            // current view without redirect)
            .state('motions.motion.detail.update', {
                onEnter: ['$stateParams', '$state', 'ngDialog', 'Motion',
                    function($stateParams, $state, ngDialog, Motion) {
                        ngDialog.open({
                            template: 'static/templates/motions/motion-form.html',
                            controller: 'MotionUpdateCtrl',
                            className: 'ngdialog-theme-default wide-form',
                            closeByEscape: false,
                            closeByDocument: false,
                            resolve: {
                                motionId: function () {return $stateParams.id;},
                            },
                            preCloseCallback: function () {
                                $state.go('motions.motion.detail', {motion: $stateParams.id});
                                return true;
                            }
                        });
                    }
                ]
            })
            .state('motions.motion.import', {
                url: '/import',
                controller: 'MotionImportCtrl',
            })
            // categories
            .state('motions.category', {
                url: '/category',
                abstract: true,
                template: "<ui-view/>",
                data: {
                    title: gettext('Categories'),
                },
            })
            .state('motions.category.list', {})
            .state('motions.category.sort', {
                url: '/sort/{id}',
                controller: 'CategorySortCtrl',
                templateUrl: 'static/templates/motions/category-sort.html',
                resolve: {
                    categoryId: ['$stateParams', function($stateParams) {
                        return $stateParams.id;
                    }],
                },
            })
            // MotionBlock
            .state('motions.motionBlock', {
                url: '/blocks',
                abstract: true,
                template: '<ui-view/>',
                data: {
                    title: gettext('Motion blocks'),
                },
            })
            .state('motions.motionBlock.list', {})
            .state('motions.motionBlock.detail', {
                resolve: {
                    motionBlockId: ['$stateParams', function($stateParams) {
                        return $stateParams.id;
                    }],
                }
            })
            // redirects to motionBlock detail and opens motionBlock edit form dialog, uses edit url,
            // used by ui-sref links from agenda only
            // (from motionBlock controller use MotionBlockForm factory instead to open dialog in front
            // of current view without redirect)
            .state('motions.motionBlock.detail.update', {
                onEnter: ['$stateParams', '$state', 'ngDialog',
                    function($stateParams, $state, ngDialog) {
                        ngDialog.open({
                            template: 'static/templates/motions/motion-block-form.html',
                            controller: 'MotionBlockUpdateCtrl',
                            className: 'ngdialog-theme-default wide-form',
                            closeByEscape: false,
                            closeByDocument: false,
                            resolve: {
                                motionBlockId: function () {
                                    return $stateParams.id;
                                }
                            },
                            preCloseCallback: function() {
                                $state.go('motions.motionBlock.detail', {motionBlock: $stateParams.id});
                                return true;
                            }
                        });
                    }
                ],
            });
    }
])

.factory('ChangeRecommendationTitleForm', [
    'gettextCatalog',
    'Editor',
    'Config',
    function(gettextCatalog) {
        return {
            // ngDialog for motion form
            getCreateDialog: function (motion, version) {
                return {
                    template: 'static/templates/motions/change-recommendation-form.html',
                    controller: 'ChangeRecommendationTitleCreateCtrl',
                    className: 'ngdialog-theme-default wide-form',
                    closeByEscape: false,
                    closeByDocument: false,
                    resolve: {
                        motion: function() {
                            return motion;
                        },
                        version: function() {
                            return version;
                        }
                    }
                };
            },
            getEditDialog: function(change) {
                return {
                    template: 'static/templates/motions/change-recommendation-form.html',
                    controller: 'ChangeRecommendationTitleUpdateCtrl',
                    className: 'ngdialog-theme-default wide-form',
                    closeByEscape: false,
                    closeByDocument: false,
                    resolve: {
                        change: function() {
                            return change;
                        }
                    }
                };
            },
            // angular-formly fields for motion form
            getFormFields: function () {
                return [
                    {
                        key: 'identifier',
                        type: 'input',
                        templateOptions: {
                            label: gettextCatalog.getString('Identifier')
                        },
                        hide: true
                    },
                    {
                        key: 'motion_version_id',
                        type: 'input',
                        templateOptions: {
                            label: gettextCatalog.getString('Motion')
                        },
                        hide: true
                    },
                    {
                        key: 'text',
                        type: 'input',
                        templateOptions: {
                            label: gettextCatalog.getString('New title'),
                            required: false
                        }
                    }
                ];
            }
        };
    }
])

.factory('ChangeRecommendationTextForm', [
    'gettextCatalog',
    'Editor',
    'Config',
    function(gettextCatalog, Editor) {
        return {
            // ngDialog for motion form
            getCreateDialog: function (motion, version, lineFrom, lineTo) {
                return {
                    template: 'static/templates/motions/change-recommendation-form.html',
                    controller: 'ChangeRecommendationTextCreateCtrl',
                    className: 'ngdialog-theme-default wide-form',
                    closeByEscape: false,
                    closeByDocument: false,
                    resolve: {
                        motion: function() {
                            return motion;
                        },
                        version: function() {
                            return version;
                        },
                        lineFrom: function() {
                            return lineFrom;
                        },
                        lineTo: function() {
                            return lineTo;
                        }
                    }
                };
            },
            getEditDialog: function(change) {
                return {
                    template: 'static/templates/motions/change-recommendation-form.html',
                    controller: 'ChangeRecommendationTextUpdateCtrl',
                    className: 'ngdialog-theme-default wide-form',
                    closeByEscape: false,
                    closeByDocument: false,
                    resolve: {
                        change: function() {
                            return change;
                        }
                    }
                };
            },
            // angular-formly fields for motion form
            getFormFields: function (line_from, line_to) {
                return [
                    {
                        key: 'identifier',
                        type: 'input',
                        templateOptions: {
                            label: gettextCatalog.getString('Identifier')
                        },
                        hide: true
                    },
                    {
                        key: 'motion_version_id',
                        type: 'input',
                        templateOptions: {
                            label: gettextCatalog.getString('Motion')
                        },
                        hide: true
                    },
                    {
                        key: 'line_from',
                        type: 'input',
                        templateOptions: {
                            label: gettextCatalog.getString('From Line')
                        },
                        hide: true
                    },
                    {
                        key: 'line_to',
                        type: 'input',
                        templateOptions: {
                            label: gettextCatalog.getString('To Line')
                        },
                        hide: true
                    },
                    {
                        key: 'type',
                        type: 'radio-buttons',
                        templateOptions: {
                            label: 'Type',
                            options: [
                                {name: gettextCatalog.getString('Replacement'), value: 0},
                                {name: gettextCatalog.getString('Insertion'), value: 1},
                                {name: gettextCatalog.getString('Deletion'), value: 2},
                                {name: gettextCatalog.getString('Other'), value: 3},
                            ]
                        }
                    },
                    {
                        key: 'other_description',
                        type: 'input',
                        templateOptions: {
                            label: gettextCatalog.getString('Description'),
                        },
                        hideExpression: "model.type !== 3",
                    },
                    {
                        key: 'text',
                        type: 'editor',
                        templateOptions: {
                            label: (
                                line_from == line_to - 1 ?
                                gettextCatalog.getString('Text in line %from%').replace(/%from%/, line_from) :
                                gettextCatalog.getString('Text from line %from% to %to%')
                                  .replace(/%from%/, line_from).replace(/%to%/, line_to - 1)
                            ),
                            required: false
                        },
                        data: {
                            ckeditorOptions: Editor.getOptions()
                        }
                    }
                ];
            }
        };
    }
])

// Service for generic motion form (create and update)
.factory('MotionForm', [
    '$filter',
    'gettextCatalog',
    'operator',
    'Editor',
    'MotionComment',
    'Category',
    'Config',
    'Mediafile',
    'MotionBlock',
    'Tag',
    'User',
    'Workflow',
    'Agenda',
    'AgendaTree',
    function ($filter, gettextCatalog, operator, Editor, MotionComment, Category,
        Config, Mediafile, MotionBlock, Tag, User, Workflow, Agenda, AgendaTree) {
        return {
            // ngDialog for motion form
            getDialog: function (motion) {
                return {
                    template: 'static/templates/motions/motion-form.html',
                    controller: motion ? 'MotionUpdateCtrl' : 'MotionCreateCtrl',
                    className: 'ngdialog-theme-default wide-form',
                    closeByEscape: false,
                    closeByDocument: false,
                    resolve: {
                        motionId: function () {return motion ? motion.id : void 0;},
                    },
                };
            },
            // angular-formly fields for motion form
            getFormFields: function (isCreateForm) {
                var workflows = Workflow.getAll();
                var images = Mediafile.getAllImages();
                var formFields = [
                {
                    key: 'identifier',
                    type: 'input',
                    templateOptions: {
                        label: gettextCatalog.getString('Identifier')
                    },
                    hide: true
                },
                {
                    key: 'submitters_id',
                    type: 'select-multiple',
                    templateOptions: {
                        label: gettextCatalog.getString('Submitters'),
                        options: User.getAll(),
                        ngOptions: 'option.id as option.full_name for option in to.options',
                        placeholder: gettextCatalog.getString('Select or search a submitter ...')
                    },
                    hide: !operator.hasPerms('motions.can_manage')
                },
                {
                    key: 'title',
                    type: 'input',
                    templateOptions: {
                        label: gettextCatalog.getString('Title'),
                        required: true
                    }
                },
                {
                    template: '<p class="spacer-top-lg no-padding">' + Config.translate(Config.get('motions_preamble').value) + '</p>'
                },
                {
                    key: 'text',
                    type: 'editor',
                    templateOptions: {
                        label: gettextCatalog.getString('Text'),
                        required: true
                    },
                    data: {
                        ckeditorOptions: Editor.getOptions()
                    }
                },
                {
                    key: 'reason',
                    type: 'editor',
                    templateOptions: {
                        label: gettextCatalog.getString('Reason'),
                    },
                    data: {
                        ckeditorOptions: Editor.getOptions()
                    }
                },
                {
                    key: 'disable_versioning',
                    type: 'checkbox',
                    templateOptions: {
                        label: gettextCatalog.getString('Trivial change'),
                        description: gettextCatalog.getString("Don't create a new version.")
                    },
                    hide: true
                }];

                // show as agenda item + parent item
                if (isCreateForm) {
                    formFields.push({
                        key: 'showAsAgendaItem',
                        type: 'checkbox',
                        templateOptions: {
                            label: gettextCatalog.getString('Show as agenda item'),
                            description: gettextCatalog.getString('If deactivated the motion appears as internal item on agenda.')
                        },
                        hide: !(operator.hasPerms('motions.can_manage') && operator.hasPerms('agenda.can_manage'))
                    });
                    formFields.push({
                        key: 'agenda_parent_id',
                        type: 'select-single',
                        templateOptions: {
                            label: gettextCatalog.getString('Parent item'),
                            options: AgendaTree.getFlatTree(Agenda.getAll()),
                            ngOptions: 'item.id as item.getListViewTitle() for item in to.options | notself : model.agenda_item_id',
                            placeholder: gettextCatalog.getString('Select a parent item ...')
                        },
                        hide: !operator.hasPerms('agenda.can_manage')
                    });
                }

                // motion comments
                formFields = formFields.concat(MotionComment.getFormFields());

                // more
                formFields.push(
                    {
                        key: 'more',
                        type: 'checkbox',
                        templateOptions: {
                            label: gettextCatalog.getString('Show extended fields')
                        },
                        hide: !operator.hasPerms('motions.can_manage')
                    },
                    {
                        template: '<hr class="smallhr">',
                        hideExpression: '!model.more'
                    }
                );
                // attachments
                if (Mediafile.getAll().length > 0) {
                    formFields.push({
                        key: 'attachments_id',
                        type: 'select-multiple',
                        templateOptions: {
                            label: gettextCatalog.getString('Attachment'),
                            options: $filter('orderBy')(Mediafile.getAll(), 'title_or_filename'),
                            ngOptions: 'option.id as option.title_or_filename for option in to.options',
                            placeholder: gettextCatalog.getString('Select or search an attachment ...')
                        },
                        hideExpression: '!model.more'
                    });
                }
                // category
                if (Category.getAll().length > 0) {
                    formFields.push({
                        key: 'category_id',
                        type: 'select-single',
                        templateOptions: {
                            label: gettextCatalog.getString('Category'),
                            options: Category.getAll(),
                            ngOptions: 'option.id as option.name for option in to.options',
                            placeholder: gettextCatalog.getString('Select or search a category ...')
                        },
                        hideExpression: '!model.more'
                    });
                }
                // motion block
                if (MotionBlock.getAll().length > 0) {
                    formFields.push({
                        key: 'motion_block_id',
                        type: 'select-single',
                        templateOptions: {
                            label: gettextCatalog.getString('Motion block'),
                            options: MotionBlock.getAll(),
                            ngOptions: 'option.id as option.title for option in to.options',
                            placeholder: gettextCatalog.getString('Select or search a motion block ...')
                        },
                        hideExpression: '!model.more'
                    });
                }
                // origin
                formFields.push({
                    key: 'origin',
                    type: 'input',
                    templateOptions: {
                        label: gettextCatalog.getString('Origin'),
                    },
                    hideExpression: '!model.more'
                });
                // tags
                if (Tag.getAll().length > 0) {
                    formFields.push({
                        key: 'tags_id',
                        type: 'select-multiple',
                        templateOptions: {
                            label: gettextCatalog.getString('Tags'),
                            options: Tag.getAll(),
                            ngOptions: 'option.id as option.name for option in to.options',
                            placeholder: gettextCatalog.getString('Select or search a tag ...')
                        },
                        hideExpression: '!model.more'
                    });
                }
                // supporters
                if (Config.get('motions_min_supporters').value > 0) {
                    formFields.push({
                        key: 'supporters_id',
                        type: 'select-multiple',
                        templateOptions: {
                            label: gettextCatalog.getString('Supporters'),
                            options: User.getAll(),
                            ngOptions: 'option.id as option.full_name for option in to.options',
                            placeholder: gettextCatalog.getString('Select or search a supporter ...')
                        },
                        hideExpression: '!model.more'
                    });
                }
                // workflows
                if (workflows.length > 1) {
                    formFields.push({
                        key: 'workflow_id',
                        type: 'select-single',
                        templateOptions: {
                            label: gettextCatalog.getString('Workflow'),
                            optionsAttr: 'bs-options',
                            options: workflows,
                            ngOptions: 'option.id as option.name | translate for option in to.options',
                            placeholder: gettextCatalog.getString('Select or search a workflow ...')
                        },
                        hideExpression: '!model.more',
                    });
                }

                return formFields;
            }
        };
    }
])

.factory('CategoryForm', [
    'gettextCatalog',
    function (gettextCatalog) {
        return {
            getDialog: function (category) {
                return {
                    template: 'static/templates/motions/category-form.html',
                    controller: category ? 'CategoryUpdateCtrl' : 'CategoryCreateCtrl',
                    className: 'ngdialog-theme-default wide-form',
                    closeByEscape: false,
                    closeByDocument: false,
                    resolve: {
                        categoryId: function () {return category ? category.id : void 0;},
                    },
                };

            },
            getFormFields: function () {
                return [
                    {
                        key: 'prefix',
                        type: 'input',
                        templateOptions: {
                            label: gettextCatalog.getString('Prefix')
                        },
                    },
                    {
                        key: 'name',
                        type: 'input',
                        templateOptions: {
                            label: gettextCatalog.getString('Name')
                        },
                    }
                ];
            },
        };
    }
])

// Provide generic motionpoll form fields for poll update view
.factory('MotionPollForm', [
    'gettextCatalog',
    function (gettextCatalog) {
        return {
            getFormFields: function () {
                return [
                {
                    key: 'yes',
                    type: 'input',
                    templateOptions: {
                        label: gettextCatalog.getString('Yes'),
                        type: 'number',
                        required: true
                    }
                },
                {
                    key: 'no',
                    type: 'input',
                    templateOptions: {
                        label: gettextCatalog.getString('No'),
                        type: 'number',
                        required: true
                    }
                },
                {
                    key: 'abstain',
                    type: 'input',
                    templateOptions: {
                        label: gettextCatalog.getString('Abstain'),
                        type: 'number',
                        required: true
                    }
                },
                {
                    key: 'votesvalid',
                    type: 'input',
                    templateOptions: {
                        label: gettextCatalog.getString('Valid votes'),
                        type: 'number'
                    }
                },
                {
                    key: 'votesinvalid',
                    type: 'input',
                    templateOptions: {
                        label: gettextCatalog.getString('Invalid votes'),
                        type: 'number'
                    }
                },
                {
                    key: 'votescast',
                    type: 'input',
                    templateOptions: {
                        label: gettextCatalog.getString('Votes cast'),
                        type: 'number'
                    }
                }];
            }
        };
    }
])

.factory('MotionExportForm', [
    'operator',
    'gettextCatalog',
    'Config',
    'MotionComment',
    function (operator, gettextCatalog, Config, MotionComment) {
        var noSpecialCommentsFields = MotionComment.getNoSpecialCommentsFields();
        return {
            getDialog: function (motions, params, singleMotion) {
                return {
                    template: 'static/templates/motions/motion-export-form.html',
                    controller: 'MotionExportCtrl',
                    className: 'ngdialog-theme-default wide-form',
                    closeByEscape: false,
                    closeByDocument: false,
                    resolve: {
                        motions: function () {return motions;},
                        params: function () {return params;},
                        singleMotion: function () {return singleMotion;},
                    },
                };
            },
            getFormFields: function (singleMotion, motions, formatChangeCallback) {
                var fields = [];
                var commentsAvailable = _.keys(noSpecialCommentsFields).length !== 0;
                var getMetaInformationOptions = function (disabled) {
                    if (!disabled) {
                        disabled = {};
                    }
                    var options = [
                        {name: gettextCatalog.getString('State'), id: 'state', disabled: disabled.state},
                        {name: gettextCatalog.getString('Submitters'), id: 'submitters', disabled: disabled.submitters},
                        {name: gettextCatalog.getString('Voting result'), id: 'votingresult', disabled: disabled.votingResult}
                    ];
                    if (_.some(motions, function (motion) { return motion.motionBlock; })) {
                        options.push({
                            name: gettextCatalog.getString('Motion block'),
                            id: 'motionBlock',
                            disabled: disabled.motionBlock,
                        });
                    }
                    if (_.some(motions, function (motion) { return motion.origin; })) {
                        options.push({
                            name: gettextCatalog.getString('Origin'),
                            id: 'origin',
                            disabled: disabled.origin,
                        });
                    }
                    if (Config.get('motions_recommendations_by').value) {
                        options.push({
                            name: gettextCatalog.getString('Recommendation'),
                            id: 'recommendation',
                            disabled: disabled.recommendation
                        });
                    }
                    return options;
                };
                if (!singleMotion) {
                    fields = [
                        {
                            key: 'format',
                            type: 'radio-buttons',
                            templateOptions: {
                                label: gettextCatalog.getString('Format'),
                                options: [
                                    {name: 'PDF', value: 'pdf'},
                                    {name: 'CSV', value: 'csv'},
                                    {name: 'DOCX', value: 'docx'},
                                ],
                                change: formatChangeCallback,
                            },
                        }
                    ];
                }
                if (operator.hasPerms('motions.can_manage')) {
                    fields.push.apply(fields, [
                        {
                            key: 'lineNumberMode',
                            type: 'radio-buttons',
                            templateOptions: {
                                label: gettextCatalog.getString('Line numbering'),
                                options: [
                                    {name: gettextCatalog.getString('None'), value: 'none'},
                                    {name: gettextCatalog.getString('inline'), value: 'inline'},
                                    {name: gettextCatalog.getString('outside'), value: 'outside'},
                                ],
                            },
                            hideExpression: "model.format !== 'pdf'",
                        },
                        {
                            key: 'lineNumberMode',
                            type: 'radio-buttons',
                            templateOptions: {
                                label: gettextCatalog.getString('Line numbering'),
                                options: [
                                    {name: gettextCatalog.getString('None'), value: 'none'},
                                    {name: gettextCatalog.getString('inline'), value: 'inline', disabled: true},
                                    {name: gettextCatalog.getString('outside'), value: 'outside', disabled: true},
                                ],
                            },
                            hideExpression: "model.format === 'pdf'",
                        },
                        {
                            key: 'changeRecommendationMode',
                            type: 'radio-buttons',
                            templateOptions: {
                                label: gettextCatalog.getString('Change recommendations'),
                                options: [
                                    {name: gettextCatalog.getString('Original version'), value: 'original'},
                                    {name: gettextCatalog.getString('Changed version'), value: 'changed'},
                                    {name: gettextCatalog.getString('Diff version'), value: 'diff'},
                                    {name: gettextCatalog.getString('Final version'), value: 'agreed'},
                                ],
                            },
                            hideExpression: "model.format !== 'pdf'",
                        },
                        {
                            key: 'changeRecommendationMode',
                            type: 'radio-buttons',
                            templateOptions: {
                                label: gettextCatalog.getString('Change recommendations'),
                                options: [
                                    {name: gettextCatalog.getString('Original version'), value: 'original'},
                                    {name: gettextCatalog.getString('Changed version'), value: 'changed'},
                                    {name: gettextCatalog.getString('Diff version'), value: 'diff', disabled: true},
                                    {name: gettextCatalog.getString('Final version'), value: 'agreed'},
                                ],
                            },
                            hideExpression: "model.format === 'pdf'",
                        },
                        {
                            key: 'include',
                            type: 'checkbox-buttons',
                            templateOptions: {
                                label: gettextCatalog.getString('Content'),
                                options: [
                                    {name: gettextCatalog.getString('Text'), id: 'text'},
                                    {name: gettextCatalog.getString('Reason'), id: 'reason'},
                                ],
                            },
                        },
                        {
                            key: 'include',
                            type: 'checkbox-buttons',
                            templateOptions: {
                                label: gettextCatalog.getString('Meta information'),
                                options: getMetaInformationOptions(),
                            },
                            hideExpression: "model.format !== 'pdf'",
                        },
                        {
                            key: 'include',
                            type: 'checkbox-buttons',
                            templateOptions: {
                                label: gettextCatalog.getString('Meta information'),
                                options: getMetaInformationOptions({votingResult: true}),
                            },
                            hideExpression: "model.format !== 'csv'",
                        },
                    ]);
                    if (commentsAvailable) {
                        fields.push({
                            key: 'includeComments',
                            type: 'checkbox-buttons',
                            templateOptions: {
                                label: gettextCatalog.getString('Comments'),
                                options: _.map(noSpecialCommentsFields, function (field, id) {
                                    return {
                                        name: gettextCatalog.getString(field.name),
                                        id: id,
                                    };
                                }),
                            },
                            hideExpression: "model.format === 'csv'",
                        });
                    }
                }
                if (!singleMotion) {
                    fields.push({
                        key: 'pdfFormat',
                        type: 'radio-buttons',
                        templateOptions: {
                            label: gettextCatalog.getString('PDF format'),
                            options: [
                                {name: gettextCatalog.getString('One PDF'), value: 'pdf'},
                                {name: gettextCatalog.getString('Multiple PDFs in a zip arcive'), value: 'zip'},
                            ],
                        },
                        hideExpression: "model.format !== 'pdf'",
                    });
                }
                return fields;
            },
        };
    }
])

.controller('MotionExportCtrl', [
    '$scope',
    'Config',
    'MotionExportForm',
    'MotionPdfExport',
    'MotionCsvExport',
    'MotionDocxExport',
    'motions',
    'params',
    'singleMotion',
    function ($scope, Config, MotionExportForm, MotionPdfExport, MotionCsvExport,
            MotionDocxExport, motions, params, singleMotion) {
        $scope.formFields = MotionExportForm.getFormFields(singleMotion, motions, function () {
            if ($scope.params.format !== 'pdf') {
                $scope.params.changeRecommendationMode = 'original';
                $scope.params.lineNumberMode = 'none';
                $scope.params.include.votingresult = false;
            }
            if ($scope.params.format === 'docx') {
                $scope.params.include.state = false;
                $scope.params.include.submitter = true;
                $scope.params.include.motionBlock = false;
                $scope.params.include.origin = false;
                $scope.params.include.recommendation = false;
            } else {
                $scope.params.include.state = true;
                $scope.params.include.motionBlock = true;
                $scope.params.include.origin = true;
                $scope.params.include.recommendation = true;
            }
            if ($scope.params.format === 'pdf') {
                $scope.params.include.state = true;
                $scope.params.include.votingresult = true;
            }
        });
        $scope.params = params || {};
        _.defaults($scope.params, {
            format: 'pdf',
            pdfFormat: 'pdf',
            changeRecommendationMode: Config.get('motions_recommendation_text_mode').value,
            lineNumberMode: Config.get('motions_default_line_numbering').value,
            include: {
                text: true,
                reason: true,
                state: true,
                submitters: true,
                votingresult: true,
                motionBlock: true,
                origin: true,
                recommendation: true,
            },
            includeComments: {},
        });
        $scope.motions = motions;
        $scope.singleMotion = singleMotion;

        $scope.export = function () {
            switch ($scope.params.format) {
                case 'pdf':
                    if ($scope.params.pdfFormat === 'pdf') {
                        MotionPdfExport.export(motions, $scope.params, singleMotion);
                    } else {
                        MotionPdfExport.exportZip(motions, $scope.params);
                    }
                    break;
                case 'csv':
                    MotionCsvExport.export(motions, $scope.params);
                    break;
                case 'docx':
                    MotionDocxExport.export(motions, $scope.params);
                    break;
            }
            $scope.closeThisDialog();
        };
    }
])

// Cache for MotionPollDetailCtrl so that users choices are keeped during user actions (e. g. save poll form).
.value('MotionPollDetailCtrlCache', {})

// Child controller of MotionDetailCtrl for each single poll.
.controller('MotionPollDetailCtrl', [
    '$scope',
    'MajorityMethodChoices',
    'Config',
    'MotionPollDetailCtrlCache',
    function ($scope, MajorityMethodChoices, Config, MotionPollDetailCtrlCache) {
        // Define choices.
        $scope.methodChoices = MajorityMethodChoices;
        // TODO: Get $scope.baseChoices from config_variables.py without copying them.

        // Setup empty cache with default values.
        if (typeof MotionPollDetailCtrlCache[$scope.poll.id] === 'undefined') {
            MotionPollDetailCtrlCache[$scope.poll.id] = {
                method: $scope.config('motions_poll_default_majority_method'),
            };
        }

        // Fetch users choices from cache.
        $scope.method = MotionPollDetailCtrlCache[$scope.poll.id].method;

        // Define result function.
        $scope.isReached = function () {
            return $scope.poll.isReached($scope.method);
        };

        // Define template controll function
        $scope.hideMajorityCalculation = function () {
            return typeof $scope.isReached() === 'undefined' && $scope.method !== 'disabled';
        };

        // Save current values to cache on detroy of this controller.
        $scope.$on('$destroy', function() {
            MotionPollDetailCtrlCache[$scope.poll.id] = {
                method: $scope.method,
            };
        });
    }
])

.controller('MotionListCtrl', [
    '$scope',
    '$state',
    '$http',
    'gettext',
    'gettextCatalog',
    'operator',
    'ngDialog',
    'MotionForm',
    'Motion',
    'MotionComment',
    'Category',
    'Config',
    'Tag',
    'Workflow',
    'User',
    'Agenda',
    'MotionBlock',
    'Projector',
    'ProjectionDefault',
    'osTableFilter',
    'osTableSort',
    'osTablePagination',
    'MotionExportForm',
    'MotionPdfExport',
    'PersonalNoteManager',
    function($scope, $state, $http, gettext, gettextCatalog, operator, ngDialog, MotionForm, Motion,
                MotionComment, Category, Config, Tag, Workflow, User, Agenda, MotionBlock, Projector,
                ProjectionDefault, osTableFilter, osTableSort, osTablePagination, MotionExportForm,
                MotionPdfExport, PersonalNoteManager) {
        Category.bindAll({}, $scope, 'categories');
        MotionBlock.bindAll({}, $scope, 'motionBlocks');
        Tag.bindAll({}, $scope, 'tags');
        Workflow.bindAll({}, $scope, 'workflows');
        User.bindAll({}, $scope, 'users');
        Projector.bindAll({}, $scope, 'projectors');
        $scope.$watch(function () {
            return Projector.lastModified();
        }, function () {
            var projectiondefault = ProjectionDefault.filter({name: 'motions'})[0];
            if (projectiondefault) {
                $scope.defaultProjectorId = projectiondefault.projector_id;
            }
        });
        $scope.$watch(function () {
            return Motion.lastModified();
        }, function () {
            // always order by identifier (after custom ordering)
            $scope.motions = _.orderBy(Motion.getAll(), ['identifier']);
            _.forEach($scope.motions, function (motion) {
                MotionComment.populateFields(motion);
                motion.personalNote = PersonalNoteManager.getNote(motion);
                // For filtering, we cannot filter for .personalNote.star
                motion.star = motion.personalNote ? motion.personalNote.star : false;
                motion.hasPersonalNote = motion.personalNote ? !!motion.personalNote.note : false;
                if (motion.star === undefined) {
                    motion.star = false;
                }
            });
            $scope.collectStatesAndRecommendations();
        });
        $scope.alert = {};

        // Motion comments
        $scope.noSpecialCommentsFields = MotionComment.getNoSpecialCommentsFields();
        $scope.showCommentsFilter = function () {
            return _.keys($scope.noSpecialCommentsFields).length > 0;
        };

        // collect all states and all recommendations of all workflows
        $scope.collectStatesAndRecommendations = function () {
            // Special case: If it is the first time updated, update the state filter.
            // This causes to set the done/undone states correct on page load.
            var doStateFilterUpdate = !$scope.states;
            $scope.states = [];
            $scope.recommendations = [];
            var workflows = $scope.collectAllUsedWorkflows();
            _.forEach(workflows, function (workflow) {
                if (workflows.length > 1) {
                    var workflowHeader = {
                        headername: workflow.name,
                        workflowHeader: true,
                    };
                    $scope.states.push(workflowHeader);
                    $scope.recommendations.push(workflowHeader);
                }

                var firstEndStateSeen = false;
                _.forEach(_.orderBy(workflow.states, 'id'), function (state) {
                    if (state.next_states_id.length === 0 && !firstEndStateSeen) {
                        $scope.states.push({divider: true});
                        firstEndStateSeen = true;
                    }
                    $scope.states.push(state);
                    if (state.recommendation_label) {
                        $scope.recommendations.push(state);
                    }
                });
            });
            if (doStateFilterUpdate) {
                updateStateFilter();
            }
        };
        $scope.collectAllUsedWorkflows = function () {
            return _.filter(Workflow.getAll(), function (workflow) {
                return _.some($scope.motions, function (motion) {
                    return motion.state.workflow_id === workflow.id;
                });
            });
        };

        $scope.stateFilter = [];
        var updateStateFilter = function () {
            $scope.stateFilter = _.clone($scope.filter.multiselectFilters.state);

            var doneIndex = _.indexOf($scope.stateFilter, -1);
            if (doneIndex > -1) { // contains -1 (done)
                $scope.stateFilter.splice(doneIndex, 1); // remove -1
                _.forEach($scope.states, function (state) {
                    if (!state.workflowHeader && !state.divider) {
                        if (state.next_states_id.length === 0) { // add all done state
                            $scope.stateFilter.push(state.id);
                        }
                    }
                });
            }

            var undoneIndex = _.indexOf($scope.stateFilter, -2);
            if (undoneIndex > -1) { // contains -2 (undone)
                $scope.stateFilter.splice(undoneIndex, 1); // remove -2
                _.forEach($scope.states, function (state) {
                    if (!state.workflowHeader && !state.divider) {
                        if (state.next_states_id.length !== 0) { // add all undone state
                            $scope.stateFilter.push(state.id);
                        }
                    }
                });
            }
            $scope.stateFilter = _.uniq($scope.stateFilter);
        };

        // Filtering
        $scope.filter = osTableFilter.createInstance('MotionTableFilter');

        if (!$scope.filter.existsStorageEntry()) {
            $scope.filter.multiselectFilters = {
                state: [],
                category: [],
                motionBlock: [],
                tag: [],
                recommendation: [],
                comment: [],
            };
            $scope.filter.booleanFilters = {
                isAmendment: {
                    value: undefined,
                    choiceYes: gettext('Is an amendment'),
                    choiceNo: gettext('Is not an amendment'),
                },
                isFavorite: {
                    value: undefined,
                    choiceYes: gettext('Marked as favorite'),
                    choiceNo: gettext('Not marked as favorite'),
                },
                hasPersonalNote: {
                    value: undefined,
                    choiceYes: gettext('Personal note set'),
                    choiceNo: gettext('Personal note not set'),
                },
            };
        }
        $scope.filter.propertyList = ['identifier', 'origin'];
        $scope.filter.propertyFunctionList = [
            function (motion) {return motion.getTitle();},
            function (motion) {return motion.category ? motion.category.name : '';},
            function (motion) {return motion.motionBlock ? motion.motionBlock.name : '';},
            function (motion) {return motion.recommendation ? motion.getRecommendationName() : '';},
        ];
        $scope.filter.propertyDict = {
            'submitters': function (submitter) {
                return submitter.get_short_name();
            },
            'supporters': function (supporter) {
                return supporter.get_short_name();
            },
            'tags': function (tag) {
                return tag.name;
            },
        };
        $scope.getItemId = {
            state: function (motion) {return motion.state_id;},
            comment: function (motion) {
                var ids = [];
                _.forEach(motion.comments, function (comment, id) {
                    if (comment) {
                        ids.push(id);
                    }
                });
                return ids;
            },
            category: function (motion) {return motion.category_id;},
            motionBlock: function (motion) {return motion.motion_block_id;},
            tag: function (motion) {return motion.tags_id;},
            recommendation: function (motion) {return motion.recommendation_id;},
        };
        $scope.operateStateFilter = function (id, danger) {
            $scope.filter.operateMultiselectFilter('state', id, danger);
            updateStateFilter();
        };
        $scope.resetFilters = function (danger) {
            $scope.filter.reset(danger);
            updateStateFilter();
        };
        // Sorting
        $scope.sort = osTableSort.createInstance('MotionTableSort');
        if (!$scope.sort.column) {
            $scope.sort.column = 'identifier';
        }
        $scope.sortOptions = [
            {name: 'identifier',
             display_name: gettext('Identifier')},
            {name: 'getTitle()',
             display_name: gettext('Title')},
            {name: 'submitters[0].get_short_name()',
             display_name: gettext('Submitters')},
            {name: 'category.' + Config.get('motions_export_category_sorting').value,
             display_name: gettext('Category')},
            {name: 'motionBlock.title',
             display_name: gettext('Motion block')},
            {name: 'state.name',
             display_name: gettext('State')},
            {name: 'log_messages[log_messages.length-1].time',
             display_name: gettext('Creation date')},
            {name: 'log_messages[0].time',
             display_name: gettext('Last modified')},
        ];

        // pagination
        $scope.pagination = osTablePagination.createInstance('MotionTablePagination');

        // update state
        $scope.updateState = function (motion, state_id) {
            $http.put('/rest/motions/motion/' + motion.id + '/set_state/', {'state': state_id});
        };
        // reset state
        $scope.resetState = function (motion) {
            $http.put('/rest/motions/motion/' + motion.id + '/set_state/', {});
        };
        // update recommendation
        $scope.updateRecommendation = function (motion, recommendation_id) {
            $http.put('/rest/motions/motion/' + motion.id + '/set_recommendation/', {'recommendation': recommendation_id});
        };
        // reset recommendation
        $scope.resetRecommendation = function (motion) {
            $http.put('/rest/motions/motion/' + motion.id + '/set_recommendation/', {});
        };

        $scope.hasTag = function (motion, tag) {
            return _.indexOf(motion.tags_id, tag.id) > -1;
        };

        $scope.save = function (motion) {
            Motion.save(motion, {method: 'PATCH'});
        };
        // delete single motion
        $scope.delete = function (motion) {
            Motion.destroy(motion.id);
        };
        $scope.toggleTag = function (motion, tag) {
            if ($scope.hasTag(motion, tag)) {
                // remove
                motion.tags_id = _.filter(motion.tags_id, function (tag_id){
                    return tag_id != tag.id;
                });
            } else {
                motion.tags_id.push(tag.id);
            }
            $scope.save(motion);
        };
        $scope.toggleCategory = function (motion, category) {
            if (motion.category_id == category.id) {
                motion.category_id = null;
            } else {
                motion.category_id = category.id;
            }
            $scope.save(motion);
        };
        $scope.toggleMotionBlock = function (motion, block) {
            if (motion.motion_block_id == block.id) {
                motion.motion_block_id = null;
            } else {
                motion.motion_block_id = block.id;
            }
            $scope.save(motion);
        };
        $scope.toggleStar = function (motion) {
            if (motion.personalNote) {
                motion.personalNote.star = !motion.personalNote.star;
            } else {
                motion.personalNote = {star: true};
            }
            PersonalNoteManager.saveNote(motion, motion.personalNote);
        };

        // open new/edit dialog
        $scope.openDialog = function (motion) {
            ngDialog.open(MotionForm.getDialog(motion));
        };
        // Export dialog
        $scope.openExportDialog = function () {
            ngDialog.open(MotionExportForm.getDialog($scope.motionsFiltered));
        };
        $scope.pdfExport = function () {
            MotionPdfExport.export($scope.motionsFiltered);
        };

        // *** select mode functions ***
        $scope.isSelectMode = false;
        // check all checkboxes from filtered motions
        $scope.checkAll = function () {
            $scope.selectedAll = !$scope.selectedAll;
            angular.forEach($scope.motionsFiltered, function (motion) {
                motion.selected = $scope.selectedAll;
            });
        };
        // uncheck all checkboxes if isSelectMode is closed
        $scope.uncheckAll = function () {
            if (!$scope.isSelectMode) {
                $scope.selectedAll = false;
                angular.forEach($scope.motions, function (motion) {
                    motion.selected = false;
                });
            }
        };
        var selectModeAction = function (predicate) {
            angular.forEach($scope.motionsFiltered, function (motion) {
                if (motion.selected) {
                    predicate(motion);
                }
            });
            $scope.isSelectMode = false;
            $scope.uncheckAll();
        };
        // delete selected motions
        $scope.deleteMultiple = function () {
            selectModeAction(function (motion) {
                $scope.delete(motion);
            });
        };
        // set status for selected motions
        $scope.setStatusMultiple = function (stateId) {
            selectModeAction(function (motion) {
                $scope.updateState(motion, stateId);
            });
        };
        // set category for selected motions
        $scope.setCategoryMultiple = function (categoryId) {
            selectModeAction(function (motion) {
                motion.category_id = categoryId === 'no_category_selected' ? null : categoryId;
                $scope.save(motion);
            });
        };
        // set status for selected motions
        $scope.setMotionBlockMultiple = function (motionBlockId) {
            selectModeAction(function (motion) {
                motion.motion_block_id = motionBlockId === 'no_motionBlock_selected' ? null : motionBlockId;
                $scope.save(motion);
            });
        };
    }
])

.controller('MotionDetailCtrl', [
    '$scope',
    '$http',
    '$timeout',
    '$window',
    '$filter',
    'operator',
    'ngDialog',
    'gettextCatalog',
    'MotionForm',
    'ChangeRecommendationCreate',
    'ChangeRecommendationView',
    'MotionStateAndRecommendationParser',
    'MotionChangeRecommendation',
    'Motion',
    'MotionComment',
    'Category',
    'Mediafile',
    'Tag',
    'User',
    'Workflow',
    'Config',
    'motionId',
    'MotionInlineEditing',
    'MotionCommentsInlineEditing',
    'Editor',
    'Projector',
    'ProjectionDefault',
    'MotionBlock',
    'MotionPdfExport',
    'PersonalNoteManager',
    'WebpageTitle',
    'EditingWarning',
    function($scope, $http, $timeout, $window, $filter, operator, ngDialog, gettextCatalog,
            MotionForm, ChangeRecommendationCreate, ChangeRecommendationView,
            MotionStateAndRecommendationParser, MotionChangeRecommendation, Motion, MotionComment,
            Category, Mediafile, Tag, User, Workflow, Config, motionId, MotionInlineEditing,
            MotionCommentsInlineEditing, Editor, Projector, ProjectionDefault, MotionBlock,
            MotionPdfExport, PersonalNoteManager, WebpageTitle, EditingWarning) {
        var motion = Motion.get(motionId);
        Category.bindAll({}, $scope, 'categories');
        Mediafile.bindAll({}, $scope, 'mediafiles');
        Tag.bindAll({}, $scope, 'tags');
        User.bindAll({}, $scope, 'users');
        Workflow.bindAll({}, $scope, 'workflows');
        MotionBlock.bindAll({}, $scope, 'motionBlocks');
        Motion.bindAll({}, $scope, 'motions');
        $scope.$watch(function () {
            return MotionChangeRecommendation.lastModified();
        }, function () {
            $scope.change_recommendations = [];
            $scope.title_change_recommendation = null;
            MotionChangeRecommendation.filter({
                'where': {'motion_version_id': {'==': motion.active_version}}
            }).forEach(function(change) {
                if (change.isTextRecommendation()) {
                    $scope.change_recommendations.push(change);
                }
                if (change.isTitleRecommendation()) {
                    $scope.title_change_recommendation = change;
                }
            });

            if ($scope.change_recommendations.length === 0) {
                $scope.setProjectionMode($scope.projectionModes[0]);
            }
            if ($scope.change_recommendations.length > 0) {
                $scope.disableMotionInlineEditing();
            }
        });
        $scope.$watch(function () {
            return Projector.lastModified();
        }, function () {
            $scope.projectors = Projector.getAll();
            var projectiondefault = ProjectionDefault.filter({name: 'motions'})[0];
            if (projectiondefault) {
                $scope.defaultProjectorId = projectiondefault.projector_id;
            }
        });
        $scope.$watch(function () {
            return Motion.lastModified(motionId);
        }, function () {
            $scope.motion = Motion.get(motionId);
            MotionComment.populateFields($scope.motion);
            if (motion.comments) {
                $scope.stateExtension = $scope.motion.comments[$scope.commentFieldForStateId];
                $scope.recommendationExtension = $scope.motion.comments[$scope.commentFieldForRecommendationId];
            }
            $scope.motion.personalNote = PersonalNoteManager.getNote($scope.motion);
            $scope.navigation.evaluate();

            var webpageTitle = gettextCatalog.getString('Motion') + ' ';
            if ($scope.motion.identifier) {
                webpageTitle += $scope.motion.identifier + ' - ';
            }
            webpageTitle += $scope.motion.getTitle();
            WebpageTitle.updateTitle(webpageTitle);

            $scope.createChangeRecommendation.setVersion(motion, motion.active_version);
        });
        $scope.$watch(function () {
            return Motion.lastModified();
        }, function () {
            $scope.motions = Motion.getAll();
            $scope.amendments = Motion.filter({parent_id: motion.id});
        });
        $scope.projectionModes = [
            {mode: 'original',
            label: 'Original version'},
            {mode: 'changed',
            label: 'Changed version'},
            {mode: 'diff',
            label: 'Diff version'},
            {mode: 'agreed',
            label: 'Final version'},
        ];
        var motionDefaultTextMode = Config.get('motions_recommendation_text_mode').value;
        $scope.projectionMode = _.find($scope.projectionModes, function (mode) {
            return mode.mode == motionDefaultTextMode;
        });
        if (motion.isProjected().length) {
            var modeMapping = motion.isProjectedWithMode();
            _.forEach($scope.projectionModes, function (mode) {
                if (mode.mode === modeMapping[0].mode) {
                    $scope.projectionMode = mode;
                }
            });
        }
        $scope.setProjectionMode = function (mode) {
            $scope.projectionMode = mode;
            var isProjected = motion.isProjectedWithMode();
            if (isProjected.length) {
                _.forEach(isProjected, function (mapping) {
                    if (mapping.mode != mode.mode) { // change the mode if it is different
                        motion.project(mapping.projectorId, mode.mode);
                    }
                });
            }
        };
        $scope.commentsFields = MotionComment.getCommentsFields();
        $scope.noSpecialCommentsFields = MotionComment.getNoSpecialCommentsFields();
        $scope.commentFieldForStateId = MotionComment.getFieldIdForFlag('forState');
        $scope.commentFieldForRecommendationId = MotionComment.getFieldIdForFlag('forRecommendation');
        $scope.version = motion.active_version;
        $scope.isCollapsed = true;
        $scope.lineNumberMode = Config.get('motions_default_line_numbering').value;
        $scope.setLineNumberMode = function(mode) {
            $scope.lineNumberMode = mode;
        };

        if (motion.parent_id) {
            Motion.bindOne(motion.parent_id, $scope, 'parent');
        }

        $scope.highlight = 0;
        $scope.scrollToAndHighlight = function (line) {
            $scope.highlight = line;

            // The same line number can occur twice in diff view; we scroll to the first one in this case
            var scrollTop = null;
            $(".line-number-" + line).each(function() {
                var top = $(this).offset().top;
                if (top > 0 && (scrollTop === null || top < scrollTop)) {
                    scrollTop = top;
                }
            });

            if (scrollTop) {
                // Scroll local; 50 pixel above the line, so it's not completely squeezed to the screen border
                $('html, body').animate({
                    'scrollTop': scrollTop - 50
                }, 1000);
                // remove the line highlight after 2 seconds.
                $timeout(function () {
                    $scope.highlight = 0;
                }, 2000);
            }
        };

        // open edit dialog
        $scope.openDialog = function (motion) {
            if ($scope.inlineEditing.active) {
                $scope.disableMotionInlineEditing();
            }
            ngDialog.open(MotionForm.getDialog(motion));
        };
        $scope.save = function (motion) {
            Motion.save(motion, {method: 'PATCH'});
        };
        // Navigation buttons
        $scope.navigation = {
            evaluate: function () {
                var motions = $filter('orderByEmptyLast')(Motion.getAll(), 'identifier');
                var thisIndex = _.findIndex(motions, function (motion) {
                    return motion.id === $scope.motion.id;
                });
                this.count = motions.length;
                this.nextMotion = thisIndex < motions.length-1 ? motions[thisIndex+1] : _.head(motions);
                this.previousMotion = thisIndex > 0 ? motions[thisIndex-1] : _.last(motions);
            },
        };
        // support
        $scope.support = function () {
            $http.post('/rest/motions/motion/' + motion.id + '/support/');
        };
        // unsupport
        $scope.unsupport = function () {
            $http.delete('/rest/motions/motion/' + motion.id + '/support/');
        };
        // open dialog for new amendment
        $scope.newAmendment = function () {
            var dialog = MotionForm.getDialog();
            if (typeof dialog.scope === 'undefined') {
                dialog.scope = {};
            }
            dialog.scope = $scope;
            ngDialog.open(dialog);
        };
        // follow recommendation
        $scope.followRecommendation = function () {
            $http.post('/rest/motions/motion/' + motion.id + '/follow_recommendation/', {
                'recommendationExtension': $scope.recommendationExtension
            });
        };
        // update state
        $scope.updateState = function (state_id) {
            $http.put('/rest/motions/motion/' + motion.id + '/set_state/', {'state': state_id});
        };
        // reset state
        $scope.reset_state = function () {
            $http.put('/rest/motions/motion/' + motion.id + '/set_state/', {});
        };
        // toggle functions for meta information
        $scope.toggleCategory = function (category) {
            if ($scope.motion.category_id == category.id) {
                $scope.motion.category_id = null;
            } else {
                $scope.motion.category_id = category.id;
            }
            $scope.save($scope.motion);
        };
        $scope.toggleMotionBlock = function (block) {
            if ($scope.motion.motion_block_id == block.id) {
                $scope.motion.motion_block_id = null;
            } else {
                $scope.motion.motion_block_id = block.id;
            }
            $scope.save($scope.motion);

        };
        $scope.toggleTag = function (tag) {
            if (_.indexOf($scope.motion.tags_id, tag.id) > -1) {
                // remove
                $scope.motion.tags_id = _.filter($scope.motion.tags_id,
                    function (tag_id){
                        return tag_id != tag.id;
                    }
                );
            } else {
                $scope.motion.tags_id.push(tag.id);
            }
            $scope.save($scope.motion);
        };
        // save additional state field
        $scope.saveAdditionalStateField = function (stateExtension) {
            motion['comment_' + $scope.commentFieldForStateId] = stateExtension;
            $scope.save(motion);
        };
        // save additional recommendation field
        $scope.saveAdditionalRecommendationField = function (recommendationExtension) {
            motion['comment_' + $scope.commentFieldForRecommendationId] = recommendationExtension;
            $scope.save(motion);
        };
        $scope.addMotionToRecommendationField = function (motion) {
            $scope.recommendationExtension += MotionStateAndRecommendationParser.formatMotion(motion);
        };
        // update recommendation
        $scope.updateRecommendation = function (recommendation_id) {
            $http.put('/rest/motions/motion/' + motion.id + '/set_recommendation/', {'recommendation': recommendation_id});
        };
        // reset recommendation
        $scope.resetRecommendation = function () {
            $http.put('/rest/motions/motion/' + motion.id + '/set_recommendation/', {});
        };
        // create poll
        $scope.create_poll = function () {
            $http.post('/rest/motions/motion/' + motion.id + '/create_poll/', {});
        };
        // open poll update dialog
        $scope.openPollDialog = function (poll, voteNumber) {
            ngDialog.open({
                template: 'static/templates/motions/motion-poll-form.html',
                controller: 'MotionPollUpdateCtrl',
                className: 'ngdialog-theme-default',
                closeByEscape: false,
                closeByDocument: false,
                resolve: {
                    motionpollId: function () {
                        return poll.id;
                    },
                    voteNumber: function () {
                        return voteNumber;
                    }
                }
            });
        };
        // delete poll
        $scope.delete_poll = function (poll) {
            poll.DSDestroy();
        };
        // show specific version
        $scope.showVersion = function (version) {
            $scope.version = version.id;
            $scope.inlineEditing.setVersion(motion, version.id);
            $scope.reasonInlineEditing.setVersion(motion, version.id);
            $scope.createChangeRecommendation.setVersion(motion, version.id);
        };
        // permit specific version
        $scope.permitVersion = function (version) {
            $http.put('/rest/motions/motion/' + motion.id + '/manage_version/',
                {'version_number': version.version_number})
                .then(function(success) {
                    $scope.showVersion(version);
                });
        };
        // delete specific version
        $scope.deleteVersion = function (version) {
            $http.delete('/rest/motions/motion/' + motion.id + '/manage_version/',
                    {headers: {'Content-Type': 'application/json'},
                     data: JSON.stringify({version_number: version.version_number})})
                .then(function(success) {
                    $scope.showVersion({id: motion.active_version});
                });
        };
        // check if there is at least one comment field
        $scope.commentFieldsAvailable = function () {
            return _.keys($scope.noSpecialCommentsFields).length > 0;
        };
        // personal note
        // For pinning the personal note container we need to adjust the width with JS. We
        // do not use angular here, because on every window resize a digist cycle would trigger.
        // This costs too much performance. We use JQuery here, because it is fast for DOM
        // manipulation and very responsive.
        $scope.toggleStar = function () {
            if ($scope.motion.personalNote) {
                $scope.motion.personalNote.star = !$scope.motion.personalNote.star;
            } else {
                $scope.motion.personalNote = {star: true};
            }
            PersonalNoteManager.saveNote($scope.motion, $scope.motion.personalNote);
        };
        $scope.personalNotePinned = false;
        $scope.pinPersonalNote = function () {
            $scope.personalNotePinned = !$scope.personalNotePinned;
            if ($scope.personalNotePinned) {
                resizePersonalNoteContainer();
            } else {
                $('#personalNote').css('width', '');
            }
        };
        $scope.gotoPersonalNote = function () {
            var pos = $('#personalNote').offset();
            $window.scrollTo(pos.left, pos.top);
        };
        var resizePersonalNoteContainer = function () {
            if ($scope.personalNotePinned) {
                var width = $('#main-column').width() - 40; // Subtract 2x20px margin
                $('#personalNote').css('width', width + 'px');
            }
        };
        $(window).resize(resizePersonalNoteContainer);

        // Inline editing functions
        $scope.inlineEditing = MotionInlineEditing.createInstance($scope, motion,
            'view-original-text-inline-editor', true, Editor.getOptions('inline'),
            function (obj) {
                return motion.getTextWithLineBreaks($scope.version);
            },
            function (obj) {
                motion.setTextStrippingLineBreaks(obj.editor.getData());
                motion.disable_versioning = (obj.trivialChange &&
                    Config.get('motions_allow_disable_versioning').value);
            }
        );
        $scope.reasonInlineEditing = MotionInlineEditing.createInstance($scope, motion,
            'reason-inline-editor', true, Editor.getOptions('inline'),
            function (obj) {
                return motion.getReason($scope.version);
            },
            function (obj) {
                motion.reason = obj.editor.getData();
                motion.disable_versioning = (obj.trivialChange &&
                    Config.get('motions_allow_disable_versioning').value);
            }
        );
        // Wrapper functions for $scope.inlineEditing, to warn other users.
        var editingStoppedCallback;
        $scope.enableMotionInlineEditing = function () {
            editingStoppedCallback = EditingWarning.editingStarted('motion_update_' + motion.id);
            if ($scope.motion.getReason($scope.version)) {
                $scope.reasonInlineEditing.enable();
            }
            $scope.inlineEditing.enable();
        };
        $scope.disableMotionInlineEditing = function () {
            if (editingStoppedCallback) {
                editingStoppedCallback();
            }
            if ($scope.motion && $scope.motion.getReason($scope.version)) {
                $scope.reasonInlineEditing.disable();
            }
            $scope.inlineEditing.disable();
        };
        $scope.textReasonSaveToolbarVisible = function () {
            return ($scope.inlineEditing.changed && $scope.inlineEditing.active) ||
                ($scope.reasonInlineEditing.changed && $scope.reasonInlineEditing.active);
        };
        $scope.textReasonSave = function () {
            if ($scope.motion.getReason($scope.version)) {
                $scope.reasonInlineEditing.save();
            }
            $scope.inlineEditing.save();
        };
        $scope.textReasonRevert = function () {
            if ($scope.motion.getReason($scope.version)) {
                $scope.reasonInlineEditing.revert();
            }
            $scope.inlineEditing.revert();
        };
        $scope.commentsInlineEditing = MotionCommentsInlineEditing.createInstances($scope, motion);
        $scope.personalNoteInlineEditing = MotionInlineEditing.createInstance($scope, motion,
            'personal-note-inline-editor', false, Editor.getOptions('inline'),
            function (obj) {
                return motion.personalNote ? motion.personalNote.note : '';
            },
            function (obj) {
                if (motion.personalNote) {
                    motion.personalNote.note = obj.editor.getData();
                } else {
                    motion.personalNote = {note: obj.editor.getData()};
                }
                PersonalNoteManager.saveNote(motion, motion.personalNote);
                obj.revert();
                obj.disable();
                return true; // Do not update the motion via patch request.
            }
        );

        // Change recommendation creation functions
        $scope.createChangeRecommendation = ChangeRecommendationCreate;
        $scope.createChangeRecommendation.init($scope, motion);

        // Change recommendation viewing
        $scope.viewChangeRecommendations = ChangeRecommendationView;
        $scope.viewChangeRecommendations.init($scope, Config.get('motions_recommendation_text_mode').value);

        // PDF creating functions
        $scope.pdfExport = function () {
            var identifier = $scope.motion.identifier ? '-' + $scope.motion.identifier : '';
            var params = {
                filename: gettextCatalog.getString('Motion') + identifier + '.pdf',
                version: $scope.version,
                changeRecommendationMode: $scope.viewChangeRecommendations.mode,
                lineNumberMode: $scope.lineNumberMode,
            };
            MotionPdfExport.export(motion, params, true);
        };
        $scope.createPollPdf = function () {
            MotionPdfExport.createPollPdf($scope.motion, $scope.version);
        };
        $scope.exportComment = function (commentId) {
            var identifier = $scope.motion.identifier ? '-' + $scope.motion.identifier : '';
            var commentsString = ' - ' + gettextCatalog.getString('Comments');
            var filename = gettextCatalog.getString('Motion') + identifier + commentsString + '.pdf';
            MotionPdfExport.exportComment($scope.motion, commentId, filename);
        };
        $scope.exportPersonalNote = function () {
            var identifier = $scope.motion.identifier ? '-' + $scope.motion.identifier : '';
            var personalNoteString = ' - ' + gettextCatalog.getString('personal note');
            var filename = gettextCatalog.getString('Motion') + identifier + personalNoteString + '.pdf';
            MotionPdfExport.exportPersonalNote($scope.motion, filename);
        };
    }
])

.controller('ChangeRecommendationTitleUpdateCtrl', [
    '$scope',
    'MotionChangeRecommendation',
    'ChangeRecommendationTitleForm',
    'diffService',
    'change',
    'ErrorMessage',
    function ($scope, MotionChangeRecommendation, ChangeRecommendationTitleForm, diffService, change, ErrorMessage) {
        $scope.alert = {};
        $scope.model = angular.copy(change);

        // get all form fields
        $scope.formFields = ChangeRecommendationTitleForm.getFormFields();
        // save motion
        $scope.save = function (change) {
            // inject the changed change recommendation (copy) object back into DS store
            MotionChangeRecommendation.inject(change);
            // save changed change recommendation object on server
            MotionChangeRecommendation.save(change).then(
                function() {
                    $scope.closeThisDialog();
                },
                function (error) {
                    MotionChangeRecommendation.refresh(change);
                    $scope.alert = ErrorMessage.forAlert(error);
                }
            );
        };
    }
])

.controller('ChangeRecommendationTitleCreateCtrl', [
    '$scope',
    'Motion',
    'MotionChangeRecommendation',
    'ChangeRecommendationTitleForm',
    'Config',
    'diffService',
    'motion',
    'version',
    function($scope, Motion, MotionChangeRecommendation, ChangeRecommendationTitleForm, Config, diffService, motion,
             version) {
        $scope.alert = {};

        $scope.model = {
            text: version.title,
            motion_version_id: version.id
        };

        // get all form fields
        $scope.formFields = ChangeRecommendationTitleForm.getFormFields();
        // save motion
        $scope.save = function (change) {
            change.line_from = 0;
            change.line_to = 0;
            MotionChangeRecommendation.create(change).then(
                function() {
                    $scope.closeThisDialog();
                }
            );
        };
    }
])

.controller('ChangeRecommendationTextUpdateCtrl', [
    '$scope',
    'MotionChangeRecommendation',
    'ChangeRecommendationTextForm',
    'diffService',
    'change',
    'ErrorMessage',
    function ($scope, MotionChangeRecommendation, ChangeRecommendationTextForm, diffService, change, ErrorMessage) {
        $scope.alert = {};
        $scope.model = angular.copy(change);

        // get all form fields
        $scope.formFields = ChangeRecommendationTextForm.getFormFields(change.line_from, change.line_to);
        // save motion
        $scope.save = function (change) {
            change.text = diffService.removeDuplicateClassesInsertedByCkeditor(change.text);
            // inject the changed change recommendation (copy) object back into DS store
            MotionChangeRecommendation.inject(change);
            // save changed change recommendation object on server
            MotionChangeRecommendation.save(change).then(
                function() {
                    $scope.closeThisDialog();
                },
                function (error) {
                    MotionChangeRecommendation.refresh(change);
                    $scope.alert = ErrorMessage.forAlert(error);
                }
            );
        };
    }
])

.controller('ChangeRecommendationTextCreateCtrl', [
    '$scope',
    'Motion',
    'MotionChangeRecommendation',
    'ChangeRecommendationTextForm',
    'Config',
    'diffService',
    'motion',
    'version',
    'lineFrom',
    'lineTo',
    function($scope, Motion, MotionChangeRecommendation, ChangeRecommendationTextForm, Config, diffService, motion,
             version, lineFrom, lineTo) {
        $scope.alert = {};

        var html = motion.getTextWithLineBreaks(version.id),
            lineData = diffService.extractRangeByLineNumbers(html, lineFrom, lineTo);

        $scope.model = {
            text: lineData.outerContextStart + lineData.innerContextStart +
                lineData.html + lineData.innerContextEnd + lineData.outerContextEnd,
            line_from: lineFrom,
            line_to: lineTo,
            motion_version_id: version.id,
            type: 0
        };

        // get all form fields
        $scope.formFields = ChangeRecommendationTextForm.getFormFields(lineFrom, lineTo);
        // save motion
        $scope.save = function (motion) {
            motion.text = diffService.removeDuplicateClassesInsertedByCkeditor(motion.text);
            MotionChangeRecommendation.create(motion).then(
                function() {
                    $scope.closeThisDialog();
                }
            );
        };
    }
])

.controller('MotionCreateCtrl', [
    '$scope',
    '$state',
    'gettext',
    'gettextCatalog',
    'operator',
    'Motion',
    'MotionForm',
    'Category',
    'Config',
    'Mediafile',
    'Tag',
    'User',
    'Workflow',
    'Agenda',
    'ErrorMessage',
    function($scope, $state, gettext, gettextCatalog, operator, Motion, MotionForm,
        Category, Config, Mediafile, Tag, User, Workflow, Agenda, ErrorMessage) {
        Category.bindAll({}, $scope, 'categories');
        Mediafile.bindAll({}, $scope, 'mediafiles');
        Tag.bindAll({}, $scope, 'tags');
        User.bindAll({}, $scope, 'users');
        Workflow.bindAll({}, $scope, 'workflows');

        $scope.model = {};
        $scope.alert = {};

        // Check whether this is a new amendment.
        var isAmendment = $scope.$parent.motion && $scope.$parent.motion.id;

        // Set default values for create form
        // ... for amendments add parent_id
        if (isAmendment) {
            if (Config.get('motions_amendments_apply_text').value) {
                $scope.model.text = $scope.$parent.motion.getText();
            }
            $scope.model.title = $scope.$parent.motion.getTitle();
            $scope.model.parent_id = $scope.$parent.motion.id;
            $scope.model.category_id = $scope.$parent.motion.category_id;
            $scope.model.motion_block_id = $scope.$parent.motion.motion_block_id;
            Motion.bindOne($scope.model.parent_id, $scope, 'parent');
        }
        // ... preselect default workflow
        if (operator.hasPerms('motions.can_manage')) {
            $scope.model.workflow_id = Config.get('motions_workflow').value;
        }
        // get all form fields
        $scope.formFields = MotionForm.getFormFields(true);

        // save motion
        $scope.save = function (motion, gotoDetailView) {
            motion.agenda_type = motion.showAsAgendaItem ? 1 : 2;
            // The attribute motion.agenda_parent_id is set by the form, see form definition.
            Motion.create(motion).then(
                function(success) {
                    if (isAmendment || gotoDetailView) {
                        $state.go('motions.motion.detail', {id: success.id});
                    }
                    $scope.closeThisDialog();
                },
                function (error) {
                    $scope.alert = ErrorMessage.forAlert(error);
                }
            );
        };
    }
])

.controller('MotionUpdateCtrl', [
    '$scope',
    '$state',
    'Motion',
    'Category',
    'Config',
    'Mediafile',
    'MotionForm',
    'Tag',
    'User',
    'Workflow',
    'Agenda',
    'motionId',
    'operator',
    'ErrorMessage',
    'EditingWarning',
    function($scope, $state, Motion, Category, Config, Mediafile, MotionForm,
        Tag, User, Workflow, Agenda, motionId, operator, ErrorMessage,
        EditingWarning) {
        Category.bindAll({}, $scope, 'categories');
        Mediafile.bindAll({}, $scope, 'mediafiles');
        Tag.bindAll({}, $scope, 'tags');
        User.bindAll({}, $scope, 'users');
        Workflow.bindAll({}, $scope, 'workflows');
        $scope.alert = {};

        // set initial values for form model by create deep copy of motion object
        // so list/detail view is not updated while editing
        var motion = Motion.get(motionId);
        $scope.model = angular.copy(motion);
        $scope.model.disable_versioning = false;
        $scope.model.more = false;

        // get all form fields
        $scope.formFields = MotionForm.getFormFields();
        // override default values for update form
        for (var i = 0; i < $scope.formFields.length; i++) {
            if ($scope.formFields[i].key == "identifier") {
                // show identifier field if the operator has manage permissions
               $scope.formFields[i].hide = !operator.hasPerms('motions.can_manage');
            }
            if ($scope.formFields[i].key == "title") {
                // get title of latest version
                $scope.formFields[i].defaultValue = motion.getTitle(-1);
            }
            if ($scope.formFields[i].key == "text") {
                // get text of latest version
                $scope.formFields[i].defaultValue = motion.getText(-1);
            }
            if ($scope.formFields[i].key == "reason") {
                // get reason of latest version
                $scope.formFields[i].defaultValue = motion.getReason(-1);
            }
            if ($scope.formFields[i].key == "disable_versioning") {
                if (Config.get('motions_allow_disable_versioning').value && motion.state.versioning) {
                    // check current state if versioning is active
                    $scope.formFields[i].hide = false;
                }
            }
            if ($scope.formFields[i].key == "workflow_id") {
               // get saved workflow id from state
               $scope.formFields[i].defaultValue = motion.state.workflow_id;
            }
        }

        // Displaying a warning, if other users edit this motion too
        var editingStoppedCallback = EditingWarning.editingStarted('motion_update_' + motionId);
        $scope.$on('$destroy', editingStoppedCallback);

        // Save motion
        $scope.save = function (motion, gotoDetailView) {
            // inject the changed motion (copy) object back into DS store
            Motion.inject(motion);
            // save changed motion object on server
            Motion.save(motion).then(
                function(success) {
                    if (gotoDetailView) {
                        $state.go('motions.motion.detail', {id: success.id});
                    }
                    $scope.closeThisDialog();
                },
                function (error) {
                    // save error: revert all changes by restore
                    // (refresh) original motion object from server
                    Motion.refresh(motion);
                    $scope.alert = ErrorMessage.forAlert(error);
                }
            );
        };
    }
])

.controller('MotionPollUpdateCtrl', [
    '$scope',
    'gettextCatalog',
    'MotionPoll',
    'MotionPollForm',
    'motionpollId',
    'voteNumber',
    'ErrorMessage',
    function($scope, gettextCatalog, MotionPoll, MotionPollForm, motionpollId,
        voteNumber, ErrorMessage) {
        // set initial values for form model by create deep copy of motionpoll object
        // so detail view is not updated while editing poll
        var motionpoll = MotionPoll.get(motionpollId);
        $scope.model = angular.copy(motionpoll);
        $scope.voteNumber = voteNumber;
        $scope.formFields = MotionPollForm.getFormFields();
        $scope.alert = {};

        // save motionpoll
        $scope.save = function (poll) {
            poll.DSUpdate({
                    motion_id: poll.motion_id,
                    votes: {"Yes": poll.yes, "No": poll.no, "Abstain": poll.abstain},
                    votesvalid: poll.votesvalid,
                    votesinvalid: poll.votesinvalid,
                    votescast: poll.votescast
            })
            .then(function(success) {
                $scope.alert.show = false;
                $scope.closeThisDialog();
            }, function(error) {
                $scope.alert = ErrorMessage.forAlert(error);
            });
        };
    }
])

.controller('MotionImportCtrl', [
    '$scope',
    '$q',
    'gettext',
    'Category',
    'Motion',
    'MotionBlock',
    'User',
    'MotionCsvExport',
    function($scope, $q, gettext, Category, Motion, MotionBlock, User, MotionCsvExport) {
        // set initial data for csv import
        $scope.motions = [];

        // set csv
        $scope.csvConfig = {
            accept: '.csv, .txt',
            encodingOptions: ['UTF-8', 'ISO-8859-1'],
            parseConfig: {
                skipEmptyLines: true,
            },
        };

        var FIELDS = ['identifier', 'title', 'text', 'reason', 'submitter', 'category', 'origin', 'motionBlock'];
        $scope.motions = [];
        $scope.onCsvChange = function (csv) {
            $scope.motions = [];
            var motions = [];
            _.forEach(csv.data, function (row) {
                if (row.length >= 3) {
                    var filledRow = _.zipObject(FIELDS, row);
                    motions.push(filledRow);
                }
            });

            _.forEach(motions, function (motion) {
                motion.selected = true;
                // identifier
                if (motion.identifier !== '') {
                    // All motion objects are already loaded via the resolve statement from ui-router.
                    var motions = Motion.getAll();
                    if (_.find(motions, function (item) {
                        return item.identifier === motion.identifier;
                    })) {
                        motion.importerror = true;
                        motion.identifier_error = gettext('Error: Identifier already exists.');
                    }
                }
                // title
                if (!motion.title) {
                    motion.importerror = true;
                    motion.title_error = gettext('Error: Title is required.');
                }
                // text
                if (!motion.text) {
                    motion.importerror = true;
                    motion.text_error = gettext('Error: Text is required.');
                } else if (!motion.text.startsWith('<p>')) {
                    motion.text = '<p>' + motion.text + '</p>';
                }
                // Reason
                if (motion.reason && !motion.reason.startsWith('<p>')) {
                    motion.reason = '<p>' + motion.reason + '</p>';
                }
                // submitter
                if (motion.submitter && motion.submitter !== '') {
                    angular.forEach(User.getAll(), function (user) {
                        var user_short_name = [user.title, user.first_name, user.last_name].join(' ').trim();
                        if (user_short_name == motion.submitter.trim()) {
                            motion.submitters_id = [user.id];
                            motion.submitter = user.full_name;
                        }
                    });
                    if (!motion.submitters_id) {
                        motion.submitter_create = gettext('New participant will be created.');
                    }
                }
                // category
                if (motion.category && motion.category !== '') {
                    angular.forEach(Category.getAll(), function (category) {
                        // search for existing category
                        if (category.name == motion.category.trim()) {
                            motion.category_id = category.id;
                            motion.category = category.name;
                        }
                    });
                    if (!motion.category_id) {
                        motion.category_create = gettext('New category will be created.');
                    }
                }
                // Motion block
                if (motion.motionBlock && motion.motionBlock !== '') {
                    angular.forEach(MotionBlock.getAll(), function (block) {
                        // search for existing block
                        if (block.title == motion.motionBlock.trim()) {
                            motion.motion_block_id = block.id;
                            motion.motionBlock = block.title;
                        }
                    });
                    if (!motion.motion_block_id) {
                        motion.motionBlock_create = gettext('New motion block will be created.');
                    }
                }

                $scope.motions.push(motion);
            });
            $scope.calcStats();
        };

        $scope.calcStats = function () {
            $scope.motionsWillNotBeImported = 0;
            $scope.motionsWillBeImported = 0;

            $scope.motions.forEach(function(motion) {
                if (!motion.importerror && motion.selected) {
                    $scope.motionsWillBeImported++;
                } else {
                    $scope.motionsWillNotBeImported++;
                }
            });
        };

        // Counter for creations
        $scope.usersCreated = 0;
        $scope.categoriesCreated = 0;

        // import from csv file
        $scope.import = function () {
            $scope.csvImporting = true;

            // Reset counters
            $scope.usersCreated = 0;
            $scope.categoriesCreated = 0;
            $scope.motionBlocksCreated = 0;

            var importedUsers = [];
            var importedCategories = [];
            var importedMotionBlocks = [];
            // collect users, categories and motion blocks
            angular.forEach($scope.motions, function (motion) {
                if (motion.selected && !motion.importerror) {
                    // collect user if not exists
                    if (!motion.submitters_id && motion.submitter) {
                        var index = motion.submitter.indexOf(' ');
                        var user = {
                            first_name: motion.submitter.substr(0, index),
                            last_name: motion.submitter.substr(index+1),
                            groups_id: []
                        };
                        importedUsers.push(user);
                    }
                    // collect category if not exists
                    if (!motion.category_id && motion.category) {
                        var category = {
                            name: motion.category,
                            prefix: motion.category.charAt(0)
                        };
                        importedCategories.push(category);
                    }
                    // collect motion block if not exists
                    if (!motion.motion_block_id && motion.motionBlock) {
                        var motionBlock = {
                            title: motion.motionBlock,
                        };
                        importedMotionBlocks.push(motionBlock);
                    }
                }
            });

            // unique users, categories and motion blocks
            var importedUsersUnique = _.uniqWith(importedUsers, function (u1, u2) {
                return u1.first_name == u2.first_name &&
                    u1.last_name == u2.last_name;
            });
            var importedCategoriesUnique = _.uniqWith(importedCategories, function (c1, c2) {
                return c1.name == c2.name;
            });
            var importedMotionBlocksUnique = _.uniqWith(importedMotionBlocks, function (c1, c2) {
                return c1.title == c2.title;
            });

            // Promises for users and categories
            var createPromises = [];

            // create users and categories
            _.forEach(importedUsersUnique, function (user) {
                createPromises.push(User.create(user).then(
                    function (success) {
                        user.id = success.id;
                        $scope.usersCreated++;
                    }
                ));
            });
            _.forEach(importedCategoriesUnique, function (category) {
                createPromises.push(Category.create(category).then(
                    function (success) {
                        category.id = success.id;
                        $scope.categoriesCreated++;
                    }
                ));
            });
            _.forEach(importedMotionBlocksUnique, function (motionBlock) {
                createPromises.push(MotionBlock.create(motionBlock).then(
                    function (success) {
                        motionBlock.id = success.id;
                        $scope.motionBlocksCreated++;
                    }
                ));
            });

            // wait for users and categories to create
            $q.all(createPromises).then( function() {
                angular.forEach($scope.motions, function (motion) {
                    if (motion.selected && !motion.importerror) {
                        // now, add user
                        if (!motion.submitters_id && motion.submitter) {
                            var index = motion.submitter.indexOf(' ');
                            var first_name = motion.submitter.substr(0, index);
                            var last_name = motion.submitter.substr(index+1);

                            // search for user, set id.
                            _.forEach(importedUsersUnique, function (user) {
                                if (user.first_name == first_name &&
                                    user.last_name == last_name) {
                                    motion.submitters_id = [user.id];
                                }
                            });
                        }
                        // add category
                        if (!motion.category_id && motion.category) {
                            var name = motion.category;

                            // search for category, set id.
                            _.forEach(importedCategoriesUnique, function (category) {
                                if (category.name == name) {
                                    motion.category_id = category.id;
                                }
                            });
                        }
                        // add motion block
                        if (!motion.motion_block_id && motion.motionBlock) {
                            var title = motion.motionBlock;

                            // search for motion block
                            _.forEach(importedMotionBlocksUnique, function (motionBlock) {
                                if (motionBlock.title == title) {
                                    motion.motion_block_id = motionBlock.id;
                                }
                            });
                        }


                        // finally create motion
                        Motion.create(motion).then(
                            function(success) {
                                motion.imported = true;
                            }
                        );
                    }
                });
            });
            $scope.csvimported = true;
        };
        $scope.clear = function () {
            $scope.motions = [];
        };
        // download CSV example file
        $scope.downloadCSVExample = function () {
            MotionCsvExport.downloadExample();
        };
    }
])


.controller('CategoryListCtrl', [
    '$scope',
    'Category',
    'ngDialog',
    'CategoryForm',
    function($scope, Category, ngDialog, CategoryForm) {
        Category.bindAll({}, $scope, 'categories');

        // setup table sorting
        $scope.sortColumn = 'name';
        $scope.reverse = false;
        // function to sort by clicked column
        $scope.toggleSort = function ( column ) {
            if ( $scope.sortColumn === column ) {
                $scope.reverse = !$scope.reverse;
            }
            $scope.sortColumn = column;
        };

        // delete selected category
        $scope.delete = function (category) {
            Category.destroy(category.id);
        };
        $scope.editOrCreate = function (category) {
            ngDialog.open(CategoryForm.getDialog(category));
        };
    }
])

.controller('CategoryCreateCtrl', [
    '$scope',
    'Category',
    'CategoryForm',
    'ErrorMessage',
    function($scope, Category, CategoryForm, ErrorMessage) {
        $scope.model = {};
        $scope.alert = {};
        $scope.formFields = CategoryForm.getFormFields();
        $scope.save = function (category) {
            Category.create(category).then(
                function(success) {
                    $scope.closeThisDialog();
                },
                function (error) {
                    $scope.alert = ErrorMessage.forAlert(error);
                }
            );
        };
    }
])

.controller('CategoryUpdateCtrl', [
    '$scope',
    'Category',
    'categoryId',
    'CategoryForm',
    'ErrorMessage',
    function($scope, Category, categoryId, CategoryForm, ErrorMessage) {
        $scope.alert = {};
        $scope.model = angular.copy(Category.get(categoryId));
        $scope.formFields = CategoryForm.getFormFields();
        $scope.save = function (category) {
            Category.inject(category);
            Category.save(category).then(
                function(success) {
                    $scope.closeThisDialog();
                },
                function (error) {
                    // save error: revert all changes by restore
                    // (refresh) original category object from server
                    Category.refresh(category);
                    $scope.alert = ErrorMessage.forAlert(error);
                }
            );
        };
    }
])

.controller('CategorySortCtrl', [
    '$scope',
    '$stateParams',
    '$http',
    'Category',
    'categoryId',
    'Motion',
    'ErrorMessage',
    function($scope, $stateParams, $http, Category, categoryId, Motion, ErrorMessage) {
        Category.bindOne(categoryId, $scope, 'category');
        Motion.bindAll({}, $scope, 'motions');
        $scope.filter = { category_id: categoryId,
                          parent_id: null,
                          orderBy: 'identifier' };

        $scope.$watch(function () {
            return Motion.lastModified();
        }, function () {
            var motions = Motion.filter($scope.filter);
            $scope.items = _.map(motions, function (motion) {
                return {
                    id: motion.id,
                    item: motion
                };
            });
        });

        $scope.alert = {};
        // Numbers all motions in this category by the given order in $scope.items
        $scope.numbering = function () {
            // Create a list of all motion ids in the current order.
            var sorted_motions = [];
            $scope.items.forEach(function (item) {
                sorted_motions.push(item.item.id);
            });

            // renumber them
            $http.post('/rest/motions/category/' + $scope.category.id + '/numbering/',
                {'motions': sorted_motions} ).then(
            function (success) {
                $scope.alert = { type: 'success', msg: success.data.detail, show: true };
            }, function (error) {
                $scope.alert = ErrorMessage.forAlert(error);
            });
        };
    }
])

//mark all motions config strings for translation in javascript
.config([
    'gettext',
    function (gettext) {
        gettext('Motions');

        // subgroup General
        gettext('General');
        gettext('Workflow of new motions');
        gettext('Identifier');
        gettext('Numbered per category');
        gettext('Serially numbered');
        gettext('Set it manually');
        gettext('Motion preamble');
        gettext('The assembly may decide:');
        gettext('Default line numbering');
        /// Line numbering: Outside
        gettext('Outside');
        /// Line numbering: Inline
        gettext('Inline');
        /// Line numbering: None
        gettext('None');
        gettext('Line length');
        gettext('The maximum number of characters per line. Relevant when line numbering is enabled. Min: 40');
        gettext('Hide reason on projector');
        gettext('Hide meta information box on projector');
        gettext('Hide recommendation on projector');
        gettext('Stop submitting new motions by non-staff users');
        gettext('Allow to disable versioning');
        gettext('Name of recommender');
        gettext('Default text version for change recommendations');
        gettext('Will be displayed as label before selected recommendation. Use an empty value to disable the recommendation system.');

        // subgroup Amendments
        gettext('Amendments');
        gettext('Activate amendments');
        gettext('Prefix for the identifier for amendments');
        gettext('Apply text for new amendments');
        gettext('The title of the motion is always applied.');

        // subgroup Supporters
        gettext('Supporters');
        gettext('Number of (minimum) required supporters for a motion');
        gettext('Choose 0 to disable the supporting system.');
        gettext('Remove all supporters of a motion if a submitter edits his ' +
                'motion in early state');

        // subgroup Supporters
        gettext('Comments');
        gettext('Comment fields for motions');
        gettext('Public');
        gettext('Private');

        // subgroup Voting and ballot papers
        gettext('Voting and ballot papers');
        gettext('The 100 % base of a voting result consists of');
        gettext('Yes/No/Abstain');
        gettext('Yes/No');
        gettext('All valid ballots');
        gettext('All casted ballots');
        gettext('Disabled (no percents)');
        gettext('Required majority');
        gettext('Default method to check whether a motion has reached the required majority.');
        gettext('Simple majority');
        gettext('Two-thirds majority');
        gettext('Three-quarters majority');
        gettext('Disabled');
        gettext('Number of ballot papers (selection)');
        gettext('Number of all delegates');
        gettext('Number of all participants');
        gettext('Use the following custom number');
        gettext('Custom number of ballot papers');

        // subgroup PDF and DOCX
        gettext('Title for PDF and DOCX documents (all motions)');
        gettext('Preamble text for PDF and DOCX documents (all motions)');
        gettext('Sort categories by');
        gettext('Include the sequential number in PDF and DOCX');

        // misc strings (used dynamically in templates by translate filter)
        gettext('needed');
    }
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.poll.majority', [])

.value('MajorityMethodChoices', [
    {'value': 'simple_majority', 'display_name': 'Simple majority'},
    {'value': 'two-thirds_majority', 'display_name': 'Two-thirds majority'},
    {'value': 'three-quarters_majority', 'display_name': 'Three-quarters majority'},
    {'value': 'disabled', 'display_name': 'Disabled'},
])

.factory('MajorityMethods', [
    function () {
        return {
            'simple_majority': function (vote, base) {
                return Math.ceil(-(base / 2 - vote)) - 1;
            },
            'two-thirds_majority': function (vote, base) {
                var result = -(base * 2 - vote * 3) / 3;
                if (result % 1 !== 0) {
                    result = Math.ceil(result) - 1;
                }
                return result;
            },
            'three-quarters_majority': function (vote, base) {
                var result = -(base * 3 - vote * 4) / 4;
                if (result % 1 !== 0) {
                    result = Math.ceil(result) - 1;
                }
                return result;
            },
            'disabled': function () {
                return undefined;
            },
        };
    }
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.topics', [])

.factory('Topic', [
    'DS',
    'jsDataModel',
    'gettext',
    function(DS, jsDataModel, gettext) {
        var name = 'topics/topic';
        return DS.defineResource({
            name: name,
            useClass: jsDataModel,
            verboseName: gettext('Topic'),
            methods: {
                getResourceName: function () {
                    return name;
                },
                getAgendaTitle: function () {
                    return this.title;
                },
                getCSVExportText: function () {
                    return this.text;
                },
            },
            relations: {
                belongsTo: {
                    'agenda/item': {
                        localKey: 'agenda_item_id',
                        localField: 'agenda_item',
                    }
                },
                hasMany: {
                    'mediafiles/mediafile': {
                        localField: 'attachments',
                        localKeys: 'attachments_id',
                    }
                }
            }
        });
    }
])

.run(['Topic', function(Topic) {}]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.topics.csv', [])

.factory('TopicsCsvExample', [
    'gettextCatalog',
    'CsvDownload',
    function (gettextCatalog, CsvDownload) {
        var makeHeaderline = function () {
            var headerline = ['Title', 'Text', 'Duration', 'Comment', 'Internal item'];
            return _.map(headerline, function (entry) {
                return gettextCatalog.getString(entry);
            });
        };
        return {
            downloadExample: function () {
                var csvRows = [makeHeaderline(),
                    // example entries
                    ['Demo 1', 'Demo text 1', '1:00', 'test comment', ''],
                    ['Break', '', '0:10', '', '1'],
                    ['Demo 2', 'Demo text 2', '1:30', '', '']

                ];
                CsvDownload(csvRows, gettextCatalog.getString('agenda-example') + '.csv');
            },
        };
    }
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.topics.projector', ['OpenSlidesApp.topics'])

.config([
    'slidesProvider',
    function (slidesProvider) {
        slidesProvider.registerSlide('topics/topic', {
            template: 'static/templates/topics/slide_topic.html'
        });
    }
])

.controller('SlideTopicCtrl', [
    '$scope',
    'Topic',
    function($scope, Topic) {
        // Attention! Each object that is used here has to be dealt on server side.
        // Add it to the coresponding get_requirements method of the ProjectorElement
        // class.
        var id = $scope.element.id;
        Topic.bindOne(id, $scope, 'topic');
    }
]);

})();

(function () {

'use strict';

angular.module('OpenSlidesApp.topics.site', ['OpenSlidesApp.topics', 'OpenSlidesApp.topics.csv'])

.config([
    '$stateProvider',
    'gettext',
    function($stateProvider, gettext) {
        $stateProvider
            .state('topics', {
                url: '/topics',
                abstract: true,
                template: "<ui-view/>",
                data: {
                    title: gettext('Topics'),
                },
            })
            .state('topics.topic', {
                abstract: true,
                template: "<ui-view/>",
            })
            .state('topics.topic.detail', {
                resolve: {
                    topicId: ['$stateParams', function($stateParams) {
                        return $stateParams.id;
                    }],
                }
            })
            // redirects to topic detail and opens topic edit form dialog, uses edit url,
            // used by ui-sref links from agenda only
            // (from topic controller use TopicForm factory instead to open dialog in front
            // of current view without redirect)
            .state('topics.topic.detail.update', {
                onEnter: ['$stateParams', '$state', 'ngDialog',
                    function($stateParams, $state, ngDialog) {
                        ngDialog.open({
                            template: 'static/templates/topics/topic-form.html',
                            controller: 'TopicUpdateCtrl',
                            className: 'ngdialog-theme-default wide-form',
                            closeByEscape: false,
                            closeByDocument: false,
                            resolve: {
                                topicId: function() {
                                    return $stateParams.id;
                                },
                            },
                            preCloseCallback: function() {
                                $state.go('topics.topic.detail', {topic: $stateParams.id});
                                return true;
                            }
                        });
                    }],
            })
            .state('topics.topic.import', {
                url: '/import',
                controller: 'TopicImportCtrl',
            });
    }
])

.factory('TopicForm', [
    '$filter',
    'gettextCatalog',
    'operator',
    'Editor',
    'Mediafile',
    'Agenda',
    'AgendaTree',
    function ($filter, gettextCatalog, operator, Editor, Mediafile, Agenda, AgendaTree) {
        return {
            // ngDialog for topic form
            getDialog: function (topic) {
                return {
                    template: 'static/templates/topics/topic-form.html',
                    controller: (topic) ? 'TopicUpdateCtrl' : 'TopicCreateCtrl',
                    className: 'ngdialog-theme-default wide-form',
                    closeByEscape: false,
                    closeByDocument: false,
                    resolve: {
                        topicId: function () {return topic ? topic.id: void 0;}
                    },
                };
            },
            getFormFields: function (isCreateForm) {
                var images = Mediafile.getAllImages();
                var formFields = [
                {
                    key: 'title',
                    type: 'input',
                    templateOptions: {
                        label: gettextCatalog.getString('Title'),
                        required: true
                    }
                },
                {
                    key: 'text',
                    type: 'editor',
                    templateOptions: {
                        label: gettextCatalog.getString('Text')
                    },
                    data: {
                        ckeditorOptions: Editor.getOptions(images)
                    }
                }];
                // attachments
                if (Mediafile.getAll().length > 0) {
                    formFields.push({
                        key: 'attachments_id',
                        type: 'select-multiple',
                        templateOptions: {
                            label: gettextCatalog.getString('Attachment'),
                            options: $filter('orderBy')(Mediafile.getAll(), 'title_or_filename'),
                            ngOptions: 'option.id as option.title_or_filename for option in to.options',
                            placeholder: gettextCatalog.getString('Select or search an attachment ...')
                        }
                    });
                }

                // show as agenda item + parent item
                if (isCreateForm) {
                    formFields.push({
                        key: 'showAsAgendaItem',
                        type: 'checkbox',
                        templateOptions: {
                            label: gettextCatalog.getString('Show as agenda item'),
                            description: gettextCatalog.getString('If deactivated it appears as internal item on agenda.')
                        },
                        hide: !operator.hasPerms('agenda.can_manage')
                    });
                    formFields.push({
                        key: 'agenda_parent_id',
                        type: 'select-single',
                        templateOptions: {
                            label: gettextCatalog.getString('Parent item'),
                            options: AgendaTree.getFlatTree(Agenda.getAll()),
                            ngOptions: 'item.id as item.getListViewTitle() for item in to.options | notself : model.agenda_item_id',
                            placeholder: gettextCatalog.getString('Select a parent item ...')
                        },
                        hide: !operator.hasPerms('agenda.can_manage')
                    });
                }

                return formFields;
            }
        };
    }
])

.controller('TopicDetailCtrl', [
    '$scope',
    'ngDialog',
    'TopicForm',
    'Topic',
    'topicId',
    'Projector',
    'ProjectionDefault',
    'WebpageTitle',
    'gettextCatalog',
    function($scope, ngDialog, TopicForm, Topic, topicId, Projector, ProjectionDefault, WebpageTitle,
        gettextCatalog) {
        $scope.$watch(function () {
            return Topic.lastModified(topicId);
        }, function () {
            $scope.topic = Topic.get(topicId);
            WebpageTitle.updateTitle(gettextCatalog.getString('Topic') + ' ' +
                $scope.topic.agenda_item.getTitle());
        });
        $scope.$watch(function () {
            return Projector.lastModified();
        }, function () {
            var projectiondefault = ProjectionDefault.filter({name: 'topics'})[0];
            if (projectiondefault) {
                $scope.defaultProjectorId = projectiondefault.projector_id;
            }
        });
        $scope.openDialog = function (topic) {
            ngDialog.open(TopicForm.getDialog(topic));
        };
    }
])

.controller('TopicCreateCtrl', [
    '$scope',
    '$state',
    'Topic',
    'TopicForm',
    'Agenda',
    'ErrorMessage',
    function($scope, $state, Topic, TopicForm, Agenda, ErrorMessage) {
        $scope.topic = {};
        $scope.model = {};
        $scope.model.showAsAgendaItem = true;
        // get all form fields
        $scope.formFields = TopicForm.getFormFields(true);
        // save form
        $scope.save = function (topic) {
            topic.agenda_type = topic.showAsAgendaItem ? 1 : 2;
            // The attribute topic.agenda_parent_id is set by the form, see form definition.
            Topic.create(topic).then(
                function (success) {
                    $scope.closeThisDialog();
                }, function (error) {
                    $scope.alert = ErrorMessage.forAlert(error);
                }
            );
        };
    }
])

.controller('TopicUpdateCtrl', [
    '$scope',
    '$state',
    'Topic',
    'TopicForm',
    'Agenda',
    'topicId',
    'ErrorMessage',
    function($scope, $state, Topic, TopicForm, Agenda, topicId, ErrorMessage) {
        var topic = Topic.get(topicId);
        $scope.alert = {};
        // set initial values for form model by create deep copy of topic object
        // so list/detail view is not updated while editing
        $scope.model = angular.copy(topic);
        // get all form fields
        $scope.formFields = TopicForm.getFormFields();

        // save form
        $scope.save = function (topic) {
            // inject the changed topic (copy) object back into DS store
            Topic.inject(topic);
            // save changed topic object on server
            Topic.save(topic).then(
                function(success) {
                    $scope.closeThisDialog();
                },
                function (error) {
                    // save error: revert all changes by restore
                    // (refresh) original topic object from server
                    Topic.refresh(topic);
                    $scope.alert = ErrorMessage.forAlert(error);
                }
            );
        };
    }
])

.controller('TopicImportCtrl', [
    '$scope',
    'gettext',
    'Agenda',
    'Topic',
    'HumanTimeConverter',
    'TopicsCsvExample',
    function($scope, gettext, Agenda, Topic, HumanTimeConverter, TopicsCsvExample) {
        // Big TODO: Change wording from "item" to "topic".
        // import from textarea
        $scope.importByLine = function () {
            if ($scope.itemlist) {
                $scope.titleItems = _.filter($scope.itemlist[0].split("\n"));
                $scope.importcounter = 0;
                _.forEach($scope.titleItems, function(title, index) {
                    var item = {title: title};
                    item.agenda_type = 1;  // The new topic is not hidden.
                    item.agenda_weight = 1000 + index;
                    // TODO: create all items in bulk mode
                    Topic.create(item).then(
                        function(success) {
                            $scope.importcounter++;
                        }
                    );
                });
            }
        };

        // *** CSV import ***
        $scope.csvConfig = {
            accept: '.csv, .txt',
            encodingOptions: ['UTF-8', 'ISO-8859-1'],
            parseConfig: {
                skipEmptyLines: true,
            },
        };
        var FIELDS = ['title', 'text', 'duration', 'comment', 'is_hidden'];
        $scope.items = [];
        $scope.onCsvChange = function (csv) {
            $scope.items = [];

            var items = [];
            _.forEach(csv.data, function (row) {
                if (row.length > 1) {
                    var filledRow = _.zipObject(FIELDS, row);
                    items.push(filledRow);
                }
            });

            _.forEach(items, function (item, index) {
                item.selected = true;

                if (!item.title) {
                    item.importerror = true;
                    item.title_error = gettext('Error: Title is required.');
                }
                // duration
                if (item.duration) {
                    var time = HumanTimeConverter.humanTimeToSeconds(item.duration, {hours: true})/60;
                    if (time <= 0) { // null instead of 0 or negative duration
                        time = null;
                    }
                    item.duration = time;
                } else {
                    delete item.duration;
                }
                // is_hidden
                if (item.is_hidden) {
                    if (item.is_hidden == '1') {
                        item.type = 2;
                    } else {
                        item.type = 1;
                    }
                } else {
                    item.type = 1;
                }
                // set weight for right csv row order
                // (Use 1000+ to protect existing items and prevent collision
                // with new items which use weight 10000 as default.)
                item.weight = 1000 + index;
                $scope.items.push(item);
            });
            $scope.calcStats();
        };

        $scope.calcStats = function () {
            $scope.itemsWillNotBeImported = 0;
            $scope.itemsWillBeImported = 0;

            $scope.items.forEach(function(item) {
                if (item.selected && !item.importerror) {
                    $scope.itemsWillBeImported++;
                } else {
                    $scope.itemsWillNotBeImported++;
                }
            });
        };

        // import from csv file
        $scope.import = function () {
            $scope.csvImporting = true;
            angular.forEach($scope.items, function (item) {
                if (item.selected && !item.importerror) {
                    item.agenda_type = item.type;
                    item.agenda_comment = item.comment;
                    item.agenda_duration = item.duration;
                    item.agenda_weight = item.weight;
                    Topic.create(item).then(
                        function(success) {
                            item.imported = true;
                        }
                    );
                }
            });
            $scope.csvimported = true;
        };
        $scope.clear = function () {
            $scope.items = null;
        };
        // download CSV example file
        $scope.downloadCSVExample = function () {
            TopicsCsvExample.downloadExample();
        };
     }
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.users', [])

.factory('User', [
    'DS',
    'Group',
    'jsDataModel',
    'gettext',
    'gettextCatalog',
    'Config',
    function(DS, Group, jsDataModel, gettext, gettextCatalog, Config) {
        var name = 'users/user';
        return DS.defineResource({
            name: name,
            useClass: jsDataModel,
            verboseName: gettext('Participants'),
            verboseNamePlural: gettext('Participants'),
            computed: {
                full_name: function () {
                    return this.get_full_name();
                },
                short_name: function () {
                    return this.get_short_name();
                },
            },
            methods: {
                getResourceName: function () {
                    return name;
                },
                /*
                 * Returns a short form of the name.
                 *
                 * Example:
                 * - Dr. Max Mustermann
                 * - Professor Dr. Enders, Christoph
                 */
                get_short_name: function() {
                    var title = _.trim(this.title),
                        firstName = _.trim(this.first_name),
                        lastName = _.trim(this.last_name),
                        name = '';
                    if (Config.get('users_sort_by') && Config.get('users_sort_by').value == 'last_name') {
                        if (lastName && firstName) {
                            name += [lastName, firstName].join(', ');
                        } else {
                            name += lastName || firstName;
                        }
                    } else {
                        name += [firstName, lastName].join(' ');
                    }
                    if (name.trim() === '') {
                        name = this.username;
                    }
                    if (title !== '') {
                        name = title + ' ' + name;
                    }
                    return name.trim();
                },
                /*
                 * Returns a long form of the name.
                 *
                 * Example:
                 * - Dr. Max Mustermann (Villingen)
                 * - Professor Dr. Enders, Christoph (Leipzig)
                 */
                get_full_name: function() {
                    var name = this.get_short_name(),
                        structure_level = _.trim(this.structure_level),
                        number = _.trim(this.number),
                        addition = [];

                    // addition: add number and structure level
                    if (structure_level) {
                        addition.push(structure_level);
                    }
                    if (number) {
                        addition.push(
                            /// abbreviation for number
                            gettextCatalog.getString('No.') + ' ' + number
                        );
                    }
                    if (addition.length > 0) {
                        name += ' (' + addition.join(' · ') + ')';
                    }
                    return name.trim();
                },
                getPerms: function() {
                    var allPerms = [];
                    var allGroups = [];
                    if (this.groups_id) {
                        allGroups = this.groups_id.slice(0);
                    }
                    if (allGroups.length === 0) {
                        allGroups.push(1); // add default group
                    }
                    _.forEach(allGroups, function(groupId) {
                        var group = Group.get(groupId);
                        if (group) {
                            _.forEach(group.permissions, function(perm) {
                                allPerms.push(perm);
                            });
                        }
                    });
                    return _.uniq(allPerms);
                },
                // link name which is shown in search result
                getSearchResultName: function () {
                    return this.get_full_name();
                },
                // subtitle of search result
                getSearchResultSubtitle: function () {
                    return "Participant";
                },
            },
            relations: {
                hasMany: {
                    'users/group': {
                        localField: 'groups',
                        localKey: 'groups_id',
                    }
                }
            }
        });
    }
])

.factory('Group', [
    'DS',
    function(DS) {
        var name = 'users/group';
        var permissions;
        return DS.defineResource({
            name: name,
        });
    }
])

.factory('PersonalNote', [
    'DS',
    function (DS) {
        var name = 'users/personal-note';
        return DS.defineResource({
            name: name,
            relations: {
                hasOne: {
                    'users/user': {
                        localField: 'user',
                        localKey: 'user_id',
                    }
                }
            }
        });
    }
])

.run([
    'User',
    'Group',
    'PersonalNote',
    function(User, Group) {}
])

// Mark strings for translation in JavaScript.
.config([
    'gettext',
    function (gettext) {
        // default group names (from users/signals.py)
        gettext('Default');
        gettext('Delegates');
        gettext('Staff');
        gettext('Committees');
    }
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.users.csv', [])

.factory('UserCsvExport', [
    '$filter',
    'Group',
    'gettextCatalog',
    'CsvDownload',
    function ($filter, Group, gettextCatalog, CsvDownload) {
        var makeHeaderline = function () {
            var headerline = ['Title', 'Given name', 'Surname', 'Structure level', 'Participant number', 'Groups',
                'Comment', 'Is active', 'Is present', 'Is a committee', 'Initial password', 'Email'];
            return _.map(headerline, function (entry) {
                return gettextCatalog.getString(entry);
            });
        };
        return {
            export: function (users) {
                var csvRows = [
                    makeHeaderline()
                ];
                _.forEach(users, function (user) {
                    var groups = _.map(user.groups_id, function (id) {
                        return gettextCatalog.getString(Group.get(id).name);
                    }).join(',');
                    var row = [];
                    row.push('"' + user.title + '"');
                    row.push('"' + user.first_name + '"');
                    row.push('"' + user.last_name + '"');
                    row.push('"' + user.structure_level + '"');
                    row.push('"' + user.number + '"');
                    row.push('"' + groups + '"');
                    row.push('"' + user.comment + '"');
                    row.push(user.is_active ? '1' : '0');
                    row.push(user.is_present ? '1' : '0');
                    row.push(user.is_committee ? '1' : '0');
                    row.push('"' + user.default_password + '"');
                    row.push('"' + user.email + '"');
                    csvRows.push(row);
                });
                CsvDownload(csvRows, gettextCatalog.getString('participants') + '.csv');
            },

            downloadExample: function () {
                // try to get an example with two groups and one with one group
                var groups = $filter('orderBy')(Group.getAll(), 'id');
                var csvGroups = '';
                var csvGroup = '';
                if (groups.length >= 3) { // do not pick groups[0], this is the default group
                    csvGroups = '"' + gettextCatalog.getString(groups[1].name) +
                                ', ' + gettextCatalog.getString(groups[2].name) + '"';
                }
                if (groups.length >= 2) {
                    csvGroup = gettextCatalog.getString(groups[groups.length - 1].name); // take last group
                }

                var csvRows = [makeHeaderline(),
                    // example entries
                    ['Dr.', 'Max', 'Mustermann', 'Berlin','1234567890', csvGroups, 'xyz', '1', '1', '', 'initialPassword', ''],
                    ['', 'John', 'Doe', 'Washington','75/99/8-2', csvGroup, 'abc', '1', '1', '', '', 'john.doe@email.com'],
                    ['', 'Fred', 'Bloggs', 'London', '', '', '', '', '', '', '', ''],
                    ['', '', 'Executive Board', '', '', '', '', '', '', '1', '', ''],

                ];
                CsvDownload(csvRows, gettextCatalog.getString('participants-example') + '.csv');
            }
        };
    }
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.users.pdf', ['OpenSlidesApp.core.pdf'])

.factory('UserListContentProvider', [
    'gettextCatalog',
    'PDFLayout',
    'Group',
    function(gettextCatalog, PDFLayout, Group) {

        var createInstance = function(userList) {
            var groups = Group.getAll();

            //use the Predefined Functions to create the title
            var title = PDFLayout.createTitle(gettextCatalog.getString('List of participants'));

            //function to generate the user list
            var createUserList = function() {
                var userJsonList = [];

                angular.forEach(userList, function (user, counter) {

                    //parse for the group names
                    var userGroups = [];
                    angular.forEach(user.groups_id, function (id) {
                        if (id) {
                            angular.forEach(groups, function(group) {
                                if (id == group.id) {
                                    userGroups.push(gettextCatalog.getString(group.name));
                                }
                            });
                        }
                    });

                    var userJsonObj = [
                        {
                            text: "" + (counter+1),
                            style: PDFLayout.flipTableRowStyle(userJsonList.length)
                        },
                        {
                            text: user.short_name,
                            style: PDFLayout.flipTableRowStyle(userJsonList.length)
                        },
                        {
                            text: user.structure_level,
                            style: PDFLayout.flipTableRowStyle(userJsonList.length)
                        },
                        {
                            text: userGroups.join(', '),
                            style: PDFLayout.flipTableRowStyle(userJsonList.length)
                        }
                    ];
                    userJsonList.push(userJsonObj);
                });

                var userTableBody = [
                    [
                        {
                            text: '#',
                            style: 'tableHeader'
                        },
                        {
                            text: gettextCatalog.getString('Name'),
                            style: 'tableHeader'
                        },
                        {
                            text: gettextCatalog.getString('Structure level'),
                            style: 'tableHeader'
                        },
                        {
                            text: gettextCatalog.getString('Groups'),
                            style: 'tableHeader'
                        }
                    ]
                ];
                userTableBody = userTableBody.concat((userJsonList));

                var userTableJsonString = {
                    table: {
                        widths: ['auto', '*', 'auto', 'auto'],
                        headerRows: 1,
                        body: userTableBody
                    },
                    layout: 'headerLineOnly'
                };

                return userTableJsonString;
            };

            var getContent = function() {
                return [
                    title,
                    createUserList()
                ];
            };

            return {
                getContent: getContent
            };
        };

        return {
            createInstance: createInstance
        };
    }
])

.factory('UserAccessDataListContentProvider', [
    'gettextCatalog',
    'PDFLayout',
    'Config',
    function(gettextCatalog, PDFLayout, Config) {

        var createInstance = function(userList) {

            var createUserHeadLine = function(user) {
                var titleLine = [];
                titleLine.push({
                    text: user.get_short_name(),
                    style: 'userDataTitle'
                });
                if (user.structure_level) {
                    titleLine.push({
                        text: user.structure_level,
                        style: 'userDataHeading'
                    });
                }
                return titleLine;
            };

            var createAccessDataContent = function(user) {
                // wlan access data
                var columnWifi = [
                    {
                        text: gettextCatalog.getString('WLAN access data'),
                        style: 'userDataHeading'
                    },
                    {
                        text: gettextCatalog.getString('WLAN name (SSID)') + ':',
                        style: 'userDataTopic'
                    },
                    {
                        text: Config.get('users_pdf_wlan_ssid').value || '-',
                        style: 'userDataValue'
                    },
                    {
                        text: gettextCatalog.getString('WLAN password') + ':',
                        style: 'userDataTopic'
                    },
                    {
                        text: Config.get('users_pdf_wlan_password').value || '-',
                        style: 'userDataValue'
                    },
                    {
                        text: gettextCatalog.getString('WLAN encryption') + ':',
                        style: 'userDataTopic'
                    },
                    {
                        text: Config.get('users_pdf_wlan_encryption').value || '-',
                        style: 'userDataValue'
                    },
                    {
                        text: '\n'
                    }
                ];
                // wifi qr code
                if (Config.get('users_pdf_wlan_ssid').value && Config.get('users_pdf_wlan_encryption').value) {
                    var wifiQrCode = 'WIFI:S:' + Config.get('users_pdf_wlan_ssid').value +
                        ';T:' + Config.get('users_pdf_wlan_encryption').value +
                        ';P:' + Config.get('users_pdf_wlan_password').value + ';;';
                    columnWifi.push(
                        {
                            qr: wifiQrCode,
                            fit: 120,
                            margin: [0, 0, 0, 8]
                        },
                        {
                            text: gettextCatalog.getString('Scan this QR code to connect to WLAN.'),
                            style: 'small'
                        }
                    );
                }

                // openslides access data
                var columnOpenSlides = [
                    {
                        text: gettextCatalog.getString('OpenSlides access data'),
                        style: 'userDataHeading'
                    },
                    {
                        text: gettextCatalog.getString('Username') + ':',
                        style: 'userDataTopic'
                    },
                    {
                        text: user.username,
                        style: 'userDataValue'
                    },
                    {
                        text: gettextCatalog.getString('Initial password') + ':',
                        style: 'userDataTopic'
                    },
                    {
                        text: user.default_password,
                        style: 'userDataValue'
                    },
                    {
                        text: 'URL:',
                        style: 'userDataTopic'
                    },
                    {
                        text: Config.get('users_pdf_url').value  || '-',
                        link: Config.get('users_pdf_url').value,
                        style: 'userDataValue'
                    },
                    {
                        text: '\n'
                    }
                ];
                // url qr code
                if (Config.get('users_pdf_url').value) {
                    columnOpenSlides.push(
                        {
                            qr: Config.get('users_pdf_url').value,
                            fit: 120,
                            margin: [0, 0, 0, 8]
                        },
                        {
                            text: gettextCatalog.getString('Scan this QR code to open URL.'),
                            style: 'small'
                        }
                    );
                }

                var accessDataColumns = {
                    columns: [
                        columnWifi,
                        columnOpenSlides,
                    ],
                    margin: [0, 20]
                };

                return accessDataColumns;
            };

            var createWelcomeText = function() {
                return [
                    {
                        text:   Config.translate(Config.get('users_pdf_welcometitle').value),
                        style: 'userDataHeading'
                    },
                    {
                        text:   Config.translate(Config.get('users_pdf_welcometext').value),
                        style: 'userDataTopic'
                    }
                ];
            };

            var getContent = function() {
                var content = [];
                angular.forEach(userList, function (user, index) {
                    content.push(createUserHeadLine(user));
                    content.push(createAccessDataContent(user));
                    content.push(createWelcomeText());
                    // No pagebreak after the last user
                    if (index !== userList.length - 1) {
                        content.push({
                            text: '',
                            pageBreak: 'after'
                        });
                    }
                });

                return [
                    content
                ];
            };

            return {
                getContent: getContent
            };
        };

        return {
            createInstance: createInstance
        };
    }
])

.factory('UserPdfExport', [
    'gettextCatalog',
    'UserListContentProvider',
    'UserAccessDataListContentProvider',
    'PdfMakeDocumentProvider',
    'PdfCreate',
    'Messaging',
    function (gettextCatalog, UserListContentProvider, UserAccessDataListContentProvider,
        PdfMakeDocumentProvider, PdfCreate, Messaging) {
        return {
            exportUserList: function (users) {
                var filename = gettextCatalog.getString('List of participants') + '.pdf';
                var userListContentProvider = UserListContentProvider.createInstance(users);
                PdfMakeDocumentProvider.createInstance(userListContentProvider).then(
                    function (documentProvider) {
                        PdfCreate.download(documentProvider, filename);
                    }, function (error) {
                        Messaging.addMessage(error.msg, 'error');
                    }
                );
            },
            exportUserAccessDataList: function (users) {
                var filename = gettextCatalog.getString('List of access data') + '.pdf';
                var userAccessDataListContentProvider = UserAccessDataListContentProvider.createInstance(
                    users);
                // no footer here
                PdfMakeDocumentProvider.createInstance(userAccessDataListContentProvider, true).then(
                    function (documentProvider) {
                        PdfCreate.download(documentProvider, filename, true);
                    }, function (error) {
                        Messaging.addMessage(error.msg, 'error');
                    }
                );
            }
        };
    }
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.users.projector', ['OpenSlidesApp.users'])

.config([
    'slidesProvider',
    function(slidesProvider) {
        slidesProvider.registerSlide('users/user', {
            template: 'static/templates/users/slide_user.html',
        });
    }
])

.controller('SlideUserCtrl', [
    '$scope',
    'User',
    function($scope, User) {
        // Attention! Each object that is used here has to be dealt on server side.
        // Add it to the coresponding get_requirements method of the ProjectorElement
        // class.
        var id = $scope.element.id;
        User.bindOne(id, $scope, 'user');
    }
]);

}());

(function () {

'use strict';

angular.module('OpenSlidesApp.users.site', [
    'OpenSlidesApp.users',
    'OpenSlidesApp.core.pdf',
    'OpenSlidesApp.users.pdf',
    'OpenSlidesApp.users.csv',
])

.config([
    'mainMenuProvider',
    'gettext',
    function (mainMenuProvider, gettext) {
        mainMenuProvider.register({
            'ui_sref': 'users.user.list',
            'img_class': 'user',
            'title': gettext('Participants'),
            'weight': 500,
            'perm': 'users.can_see_name',
        });
    }
])
.config([
    'SearchProvider',
    'gettext',
    function (SearchProvider, gettext) {
        SearchProvider.register({
            'verboseName': gettext('Participants'),
            'collectionName': 'users/user',
            'urlDetailState': 'users.user.detail',
            'weight': 500,
        });
    }
])

.config([
    '$stateProvider',
    'gettext',
    function($stateProvider, gettext) {
        $stateProvider
        .state('users', {
            url: '/users',
            abstract: true,
            template: "<ui-view/>",
            data: {
                title: gettext('Participants'),
                basePerm: 'users.can_see_name',
            },
        })
        .state('users.user', {
            abstract: true,
            template: "<ui-view/>",
        })
        .state('users.user.list', {})
        .state('users.user.create', {})
        .state('users.user.detail', {
            resolve: {
                userId: ['$stateParams', function($stateParams) {
                    return $stateParams.id;
                }]
            }
        })
        .state('users.user.change-password', {
            url: '/change-password/{id}',
            controller: 'UserChangePasswordCtrl',
            templateUrl: 'static/templates/users/user-change-password.html',
            resolve: {
                userId: ['$stateParams', function($stateParams) {
                    return $stateParams.id;
                }]
            }
        })
        .state('users.user.import', {
            url: '/import',
            controller: 'UserImportCtrl',
        })
        .state('users.user.presence', {
            url: '/presence',
            controller: 'UserPresenceCtrl',
        })
        // groups
        .state('users.group', {
            url: '/groups',
            abstract: true,
            template: "<ui-view/>",
            data: {
                title: gettext('Groups'),
            },
        })
        .state('users.group.list', {})
        .state('login', {
            template: null,
            url: '/login',
            params: {
                guest_enabled: false,
                msg: null,
            },
            onEnter: ['$state', '$stateParams', 'ngDialog', 'LoginDialog', function($state, $stateParams, ngDialog, LoginDialog) {
                LoginDialog.id = ngDialog.open({
                    template: 'static/templates/core/login-form.html',
                    controller: 'LoginFormCtrl',
                    showClose: $stateParams.guest_enabled,
                    closeByEscape: $stateParams.guest_enabled,
                    closeByDocument: $stateParams.guest_enabled,
                }).id;
            }],
            data: {
                title: 'Login',
            },
        });
    }
])

.value('LoginDialog', {})

/*
 * Directive to check for permissions
 *
 * This is the Code from angular.js ngIf.
 *
 * TODO: find a way not to copy the code.
*/
.directive('osPerms', [
    '$animate',
    function($animate) {
        return {
            multiElement: true,
            transclude: 'element',
            priority: 600,
            terminal: true,
            restrict: 'A',
            $$tlb: true,
            link: function($scope, $element, $attr, ctrl, $transclude) {
                var block, childScope, previousElements, perms;
                if ($attr.osPerms[0] === '!') {
                    perms = _.trimStart($attr.osPerms, '!');
                } else {
                    perms = $attr.osPerms;
                }
                $scope.$watch(
                    function (scope) {
                        return scope.operator && scope.operator.hasPerms(perms);
                    },
                    function (value) {
                        if ($attr.osPerms[0] === '!') {
                            value = !value;
                        }
                        if (value) {
                            if (!childScope) {
                                $transclude(function(clone, newScope) {
                                    childScope = newScope;
                                    clone[clone.length++] = document.createComment(' end ngIf: ' + $attr.ngIf + ' ');
                                    // Note: We only need the first/last node of the cloned nodes.
                                    // However, we need to keep the reference to the jqlite wrapper as it might be changed later
                                    // by a directive with templateUrl when its template arrives.
                                    block = {
                                        clone: clone
                                    };
                                    $animate.enter(clone, $element.parent(), $element);
                                });
                            }
                        } else {
                            if (previousElements) {
                                previousElements.remove();
                                previousElements = null;
                            }
                            if (childScope) {
                                childScope.$destroy();
                                childScope = null;
                            }
                            if (block) {
                                previousElements = getBlockNodes(block.clone);
                                $animate.leave(previousElements).then(function() {
                                    previousElements = null;
                                });
                                block = null;
                            }
                        }
                    }
                );
            }
        };
    }
])

.factory('PasswordGenerator', [
    function () {
        return {
            generate: function (length) {
                if (!length) {
                    length = 8;
                }
                var chars = 'abcdefghijkmnpqrstuvwxyzABCDEFGHJKLMNPQRSTUVWXYZ23456789',
                pw = '';
                for (var i = 0; i < length; ++i) {
                    pw += chars.charAt(Math.floor(Math.random() * chars.length));
                }
                return pw;
            }
        };
    }
])

.factory('PersonalNoteManager', [
    'PersonalNote',
    'operator',
    function (PersonalNote, operator) {
        var _getPersonalNoteObject = function (resourceName) {
            var personalNote = _.find(PersonalNote.getAll(), function (pn) {
                return pn.user_id === operator.user.id;
            });
            if (!personalNote) {
                personalNote = {
                    notes: {},
                };
            }
            if (!personalNote.notes[resourceName]) {
                personalNote.notes[resourceName] = {};
            }
            return personalNote;
        };
        var get = function (resourceName, id) {
            return _getPersonalNoteObject(resourceName).notes[resourceName][id];
        };
        var save = function (resourceName, id, note) {
            var personalNote = _getPersonalNoteObject(resourceName);
            personalNote.notes[resourceName][id] = note;
            if (personalNote.id) {
                return PersonalNote.save(personalNote);
            } else {
                return PersonalNote.create(personalNote);
            }
        };
        return {
            getNote: function (obj) {
                if (typeof obj.getResourceName === 'undefined') {
                    throw 'The Object has to be a js data model!';
                }
                return get(obj.getResourceName(), obj.id);
            },
            saveNote: function (obj, note) {
                if (typeof obj.getResourceName === 'undefined') {
                    throw 'The Object has to be a js data model!';
                }
                return save(obj.getResourceName(), obj.id, note);
            },
        };
    }
])

// Service for generic assignment form (create and update)
.factory('UserForm', [
    '$http',
    'gettextCatalog',
    'Editor',
    'Group',
    'Mediafile',
    'PasswordGenerator',
    function ($http, gettextCatalog, Editor, Group, Mediafile, PasswordGenerator) {
        return {
            // ngDialog for user form
            getDialog: function (user) {
                return {
                    template: 'static/templates/users/user-form.html',
                    controller: (user) ? 'UserUpdateCtrl' : 'UserCreateCtrl',
                    className: 'ngdialog-theme-default wide-form',
                    closeByEscape: false,
                    closeByDocument: false,
                    resolve: {
                        userId: function () {return user ? user.id : void 0;},
                    }
                };
            },
            // angular-formly fields for user form
            getFormFields: function (hideOnCreateForm) {
                var images = Mediafile.getAllImages();
                return [
                {
                    className: "row",
                    fieldGroup: [
                        {
                            key: 'title',
                            type: 'input',
                            className: "col-xs-2 no-padding-left",
                            templateOptions: {
                                label: gettextCatalog.getString('Title')
                            }
                        },
                        {
                            key: 'first_name',
                            type: 'input',
                            className: "col-xs-5 no-padding",
                            templateOptions: {
                                label: gettextCatalog.getString('Given name')
                            }
                        },
                        {
                            key: 'last_name',
                            type: 'input',
                            className: "col-xs-5 no-padding-right",
                            templateOptions: {
                                label: gettextCatalog.getString('Surname')
                            }
                        }
                    ]
                },
                {
                    key: 'email',
                    type: 'input',
                    templateOptions: {
                        label: gettextCatalog.getString('Email')
                    },
                },
                {
                    className: "row",
                    fieldGroup: [
                        {
                            key: 'structure_level',
                            type: 'input',
                            className: "col-xs-9 no-padding-left",
                            templateOptions: {
                                label: gettextCatalog.getString('Structure level'),
                            }
                        },
                        {   key: 'number',
                            type: 'input',
                            className: "col-xs-3 no-padding-left no-padding-right",
                            templateOptions: {
                                label:gettextCatalog.getString('Participant number')
                            }
                        }
                    ]
                },
                {
                    key: 'username',
                    type: 'input',
                    templateOptions: {
                        label: gettextCatalog.getString('Username')
                    },
                    hide: hideOnCreateForm
                },
                {
                    key: 'groups_id',
                    type: 'select-multiple',
                    templateOptions: {
                        label: gettextCatalog.getString('Groups'),
                        options: Group.filter({where: {id: {'>': 1}}}),
                        ngOptions: "option.id as option.name | translate for option in to.options | orderBy: 'id'",
                        placeholder: gettextCatalog.getString('Select or search a group ...')
                    }
                },
                {
                    key: 'default_password',
                    type: 'input',
                    templateOptions: {
                        label: gettextCatalog.getString('Initial password'),
                        description: gettextCatalog.getString('Initial password can not be changed.'),
                        addonRight: {
                            text: gettextCatalog.getString('Generate'),
                            class: 'fa fa-magic',
                            onClick:function (options, scope) {
                                scope.$parent.model.default_password = PasswordGenerator.generate();
                            }
                        }
                    },
                    hide: !hideOnCreateForm
                },
                {
                    key: 'comment',
                    type: 'input',
                    templateOptions: {
                        label: gettextCatalog.getString('Comment'),
                        description: gettextCatalog.getString('Only for internal notes.')
                    }
                },
                {
                    key: 'more',
                    type: 'checkbox',
                    templateOptions: {
                        label: gettextCatalog.getString('Show extended fields')
                    }
                },
                {
                    template: '<hr class="smallhr">',
                    hideExpression: '!model.more'
                },
                {
                    key: 'is_present',
                    type: 'checkbox',
                    templateOptions: {
                        label: gettextCatalog.getString('Is present'),
                        description: gettextCatalog.getString('Designates whether this user is in the room or not.')
                    },
                    defaultValue: true,
                    hideExpression: '!model.more'
                },
                {
                    key: 'is_active',
                    type: 'checkbox',
                    templateOptions: {
                        label: gettextCatalog.getString('Is active'),
                        description: gettextCatalog.getString(
                            'Designates whether this user should be treated as ' +
                            'active. Unselect this instead of deleting the account.')
                    },
                    defaultValue: true,
                    hideExpression: '!model.more'
                },
                {
                    key: 'is_committee',
                    type: 'checkbox',
                    templateOptions: {
                        label: gettextCatalog.getString('Is a committee'),
                        description: gettextCatalog.getString(
                            'Designates whether this user should be treated as a committee.')
                    },
                    defaultValue: false,
                    hideExpression: '!model.more'
                },
                {
                    key: 'about_me',
                    type: 'editor',
                    templateOptions: {
                        label: gettextCatalog.getString('About me'),
                    },
                    data: {
                        ckeditorOptions: Editor.getOptions(images)
                    },
                    hideExpression: '!model.more'
                }
                ];
            }
        };
    }
])

.factory('UserProfileForm', [
    'gettextCatalog',
    'Editor',
    'Mediafile',
    function (gettextCatalog, Editor, Mediafile) {
        return {
            // ngDialog for user form
            getDialog: function () {
                return {
                    template: 'static/templates/users/profile-password-form.html',
                    controller: 'UserProfileCtrl',
                    className: 'ngdialog-theme-default wide-form',
                    closeByEscape: false,
                    closeByDocument: false,
                };
            },
            // angular-formly fields for user form
            getFormFields: function (hideOnCreateForm) {
                var images = Mediafile.getAllImages();
                return [
                {
                    key: 'username',
                    type: 'input',
                    templateOptions: {
                        label: gettextCatalog.getString('Username'),
                        required: true
                    },
                },
                {
                    key: 'email',
                    type: 'input',
                    templateOptions: {
                        label: gettextCatalog.getString('Email')
                    },
                },
                {
                    key: 'about_me',
                    type: 'editor',
                    templateOptions: {
                        label: gettextCatalog.getString('About me'),
                    },
                    data: {
                        ckeditorOptions: Editor.getOptions(images)
                    },
                }
                ];
            }
        };
    }
])

.factory('UserPasswordForm', [
    'gettextCatalog',
    function (gettextCatalog) {
        return {
            // ngDialog for user form
            getDialog: function () {
                return {
                    template: 'static/templates/users/profile-password-form.html',
                    controller: 'UserPasswordCtrl',
                    className: 'ngdialog-theme-default',
                    closeByEscape: false,
                    closeByDocument: false,
                };
            },
            // angular-formly fields for user form
            getFormFields: function (hideOnCreateForm) {
                return [
                {
                    key: 'oldPassword',
                    type: 'password',
                    templateOptions: {
                        label: gettextCatalog.getString('Old password'),
                        required: true
                    },
                },
                {
                    key: 'newPassword',
                    type: 'password',
                    templateOptions: {
                        label: gettextCatalog.getString('New password'),
                        required: true
                    },
                },
                {
                    key: 'newPassword2',
                    type: 'password',
                    templateOptions: {
                        label: gettextCatalog.getString('Confirm new password'),
                        required: true
                    },
                },
                ];
            }
        };
    }
])

.controller('UserListCtrl', [
    '$scope',
    '$http',
    'ngDialog',
    'UserForm',
    'User',
    'Group',
    'PasswordGenerator',
    'Projector',
    'ProjectionDefault',
    'Config',
    'UserCsvExport',
    'osTableFilter',
    'osTableSort',
    'osTablePagination',
    'gettext',
    'UserPdfExport',
    'InvitationEmails',
    'ErrorMessage',
    function($scope, $http, ngDialog, UserForm, User, Group, PasswordGenerator,
        Projector, ProjectionDefault, Config, UserCsvExport, osTableFilter, osTableSort,
        osTablePagination, gettext, UserPdfExport, InvitationEmails, ErrorMessage) {
        $scope.$watch(function () {
            return User.lastModified();
        }, function () {
            $scope.users = _.orderBy(User.getAll(), ['first_name']);
            _.forEach($scope.users, function (user) {
                user.has_last_email_send = !!user.last_email_send;
            });
            if ($scope.updateUsers) {
                $scope.updateUsers();
            }
        });
        Group.bindAll({where: {id: {'>': 1}}}, $scope, 'groups');
        $scope.$watch(function () {
            return Projector.lastModified();
        }, function () {
            var projectiondefault = ProjectionDefault.filter({name: 'users'})[0];
            if (projectiondefault) {
                $scope.defaultProjectorId = projectiondefault.projector_id;
            }
        });
        $scope.alert = {};

        // Filtering
        $scope.filter = osTableFilter.createInstance('UserTableFilter');

        if (!$scope.filter.existsStorageEntry()) {
            $scope.filter.multiselectFilters = {
                group: [],
            };
            $scope.filter.booleanFilters = {
                isPresent: {
                    value: undefined,
                    displayName: gettext('Present'),
                    choiceYes: gettext('Is present'),
                    choiceNo: gettext('Is not present'),
                    needExtraPermission: true,
                },
                isActive: {
                    value: undefined,
                    displayName: gettext('Active'),
                    choiceYes: gettext('Is active'),
                    choiceNo: gettext('Is not active'),
                    needExtraPermission: true,
                },
                isCommittee: {
                    value: undefined,
                    displayName: gettext('Committee'),
                    choiceYes: gettext('Is a committee'),
                    choiceNo: gettext('Is not a committee'),
                },
                hasLastEmailSend: {
                    value: undefined,
                    displayName: gettext('Last email send'),
                    choiceYes: gettext('Got an email'),
                    choiceNo: gettext("Didn't get an email"),
                },
            };
        }
        $scope.filter.propertyList = ['first_name', 'last_name', 'username', 'title',
            'number', 'comment', 'structure_level'];
        $scope.filter.propertyDict = {
            'groups_id' : function (group_id) {
                return Group.get(group_id).name;
            },
        };
        $scope.getItemId = {
            group: function (user) {return user.groups_id;},
        };
        // Sorting
        $scope.sort = osTableSort.createInstance('UserTableSort');
        if (!$scope.sort.column) {
            $scope.sort.column = $scope.config('users_sort_by');
        }
        $scope.sortOptions = [
            {name: 'first_name',
             display_name: gettext('Given name')},
            {name: 'last_name',
             display_name: gettext('Surname')},
            {name: 'is_present',
             display_name: gettext('Present')},
            {name: 'is_active',
             display_name: gettext('Active')},
            {name: 'is_committee',
             display_name: gettext('Committee')},
            {name: 'number',
             display_name: gettext('Number')},
            {name: 'structure_level',
             display_name: gettext('Structure level')},
            {name: 'comment',
             display_name: gettext('Comment')},
            {name: 'last_email_send',
             display_name: gettext('Last email send')},
        ];

        // pagination
        $scope.pagination = osTablePagination.createInstance('UserTablePagination');

        // Toggle group from user
        $scope.toggleGroup = function (user, group) {
            if (_.indexOf(user.groups_id, group.id) > -1) {
                user.groups_id = _.filter(user.groups_id, function (group_id) {
                    return group_id != group.id;
                });
            } else {
                user.groups_id.push(group.id);
            }
            $scope.save(user);
        };
        // open new/edit dialog
        $scope.openDialog = function (user) {
            ngDialog.open(UserForm.getDialog(user));
        };
        // save changed user
        $scope.save = function (user) {
            User.save(user).then(
                function(success) {
                    //user.quickEdit = false;
                    $scope.alert.show = false;
                },
                function(error){
                    $scope.alert = ErrorMessage.forAlert(error);
                });
        };
        // delete single user
        $scope.delete = function (user) {
            User.destroy(user.id);
        };
        // *** select mode functions ***
        $scope.isSelectMode = false;
        // check all checkboxes
        $scope.checkAll = function () {
            $scope.selectedAll = !$scope.selectedAll;
            _.forEach($scope.usersFiltered, function (user) {
                user.selected = $scope.selectedAll;
            });
        };
        // uncheck all checkboxes if isSelectMode is closed
        $scope.uncheckAll = function () {
            if (!$scope.isSelectMode) {
                $scope.selectedAll = false;
                angular.forEach($scope.users, function (user) {
                    user.selected = false;
                });
            }
        };
        var selectModeAction = function (predicate) {
            angular.forEach($scope.usersFiltered, function (user) {
                if (user.selected) {
                    predicate(user);
                }
            });
            $scope.isSelectMode = false;
            $scope.uncheckAll();
        };
        // delete all selected users
        $scope.deleteMultiple = function () {
            selectModeAction(function (user) {
                $scope.delete(user);
            });
        };
        // add group for selected users
        $scope.addGroupMultiple = function (group) {
            if (group) {
                selectModeAction(function (user) {
                    user.groups_id.push(group);
                    User.save(user);
                });
            }
        };
        // remove group for selected users
        $scope.removeGroupMultiple = function (group) {
            if (group) {
                selectModeAction(function (user) {
                    var groupIndex = _.indexOf(user.groups_id, parseInt(group));
                    if (groupIndex > -1) {
                        user.groups_id.splice(groupIndex, 1);
                        User.save(user);
                    }
                });
            }
        };
        // generate new passwords
        $scope.generateNewPasswordsMultiple = function () {
            selectModeAction(function (user) {
                var newPassword = PasswordGenerator.generate();
                user.default_password = newPassword;
                User.save(user);
                $http.post(
                    '/rest/users/user/' + user.id + '/reset_password/',
                    {'password': newPassword}
                );
            });
        };
        // set boolean properties (is_active, is_present, is_committee)
        $scope.setBoolPropertyMultiple = function (property, value) {
            selectModeAction(function (user) {
                user[property] = value;
                User.save(user);
            });
        };
        // Send invitation emails
        $scope.sendInvitationEmails = function () {
            InvitationEmails.send($scope.usersFiltered).then(function (success) {
                $scope.alert = success;
                $scope.isSelectMode = false;
                $scope.uncheckAll();
            }, function (error) {
                $scope.alert = ErrorMessage.forAlert(error);
                $scope.isSelectMode = false;
                $scope.uncheckAll();
            });
        };

        // Export as PDF
        $scope.pdfExportUserList = function () {
            UserPdfExport.exportUserList($scope.usersFiltered);
        };
        $scope.pdfExportUserAccessDataList = function () {
            UserPdfExport.exportUserAccessDataList($scope.usersFiltered);
        };
        // Export as a csv file
        $scope.csvExport = function () {
            UserCsvExport.export($scope.usersFiltered);
        };
    }
])

.factory('InvitationEmails', [
    '$http',
    'User',
    'Config',
    'gettextCatalog',
    function ($http, User, Config, gettextCatalog) {
        return {
            // Returns the request promise. If it was successfull, a nice message for
            // an alert is generated and the alert-object is returned.
            send: function (users) {
                var user_ids = _
                    .chain(users)
                    .filter(function (user) {
                        return user.selected;
                    })
                    .map(function (user) {
                        return user.id;
                    })
                    .value();
                var subject = gettextCatalog.getString(Config.get('users_email_subject').value);
                var message = gettextCatalog.getString(Config.get('users_email_body').value);

                return $http.post('/rest/users/user/mass_invite_email/', {
                    user_ids: user_ids,
                    subject: subject,
                    message: message,
                }).then(function (success) {
                    var numEmails = success.data.count;
                    var noEmailIds = success.data.no_email_ids;
                    var type = 'success', msg;
                    if (numEmails === 0) {
                        type = 'danger';
                        msg = gettextCatalog.getString('No emails were send.');
                    } else if (numEmails === 1) {
                        msg = gettextCatalog.getString('One email was send sucessfully.');
                    } else {
                        msg = gettextCatalog.getString('%num% emails were send sucessfully.').replace('%num%', numEmails);
                    }

                    if (noEmailIds.length) {
                        type = 'warning';
                        msg += ' ';

                        if (noEmailIds.length === 1) {
                            msg += gettextCatalog.getString('The user %user% has no email, so the invitation email could not be send.');
                        } else {
                            msg += gettextCatalog.getString('The users %user% have no email, so the invitation emails could not be send.');
                        }

                        // This one builds a username string like "user1, user2 and user3" with the full names.
                        var lastUsername, userString = _
                            .chain(noEmailIds)
                            .map(function (id) {
                                var user = User.get(id);
                                return user ? user.get_full_name() : '';
                            })
                            .filter(function (username) {
                                return username;
                            })
                            .tap(function (names) {
                                if (names.length !== 1) {
                                    lastUsername = names.pop();
                                }
                            })
                            .join(', ')
                            .thru(function (names) {
                                return lastUsername ? names + ' ' + gettextCatalog.getString('and') + ' ' + lastUsername : names;
                            })
                            .value();
                        msg = msg.replace('%user%', userString);
                    }

                    return {
                        msg: msg,
                        type: type,
                        show: true,
                    };
                });
            },
        };
    }
])

.controller('UserDetailCtrl', [
    '$scope',
    'ngDialog',
    'UserForm',
    'User',
    'userId',
    'Group',
    'Projector',
    'ProjectionDefault',
    'gettextCatalog',
    'WebpageTitle',
    function($scope, ngDialog, UserForm, User, userId, Group, Projector, ProjectionDefault, gettextCatalog,
        WebpageTitle) {
        Group.bindAll({where: {id: {'>': 1}}}, $scope, 'groups');
        $scope.$watch(function () {
            return User.lastModified(userId);
        }, function () {
            $scope.user = User.get(userId);
            WebpageTitle.updateTitle(gettextCatalog.getString('Participant') + ' ' + $scope.user.get_short_name());
        });
        $scope.$watch(function () {
            return Projector.lastModified();
        }, function () {
            var projectiondefault = ProjectionDefault.filter({name: 'users'})[0];
            if (projectiondefault) {
                $scope.defaultProjectorId = projectiondefault.projector_id;
            }
        });

        // open edit dialog
        $scope.openDialog = function (user) {
            ngDialog.open(UserForm.getDialog(user));
        };
    }
])

.controller('UserCreateCtrl', [
    '$scope',
    '$state',
    'User',
    'UserForm',
    'ErrorMessage',
    function($scope, $state, User, UserForm, ErrorMessage) {
        $scope.alert = {};
        // get all form fields
        $scope.formFields = UserForm.getFormFields(true);

        // save user
        $scope.save = function (user) {
            if (!user.groups_id) {
                user.groups_id = [];
            }
            User.create(user).then(
                function (success) {
                    $scope.closeThisDialog();
                },
                function (error) {
                    $scope.alert = ErrorMessage.forAlert(error);
                }
            );
        };
    }
])

.controller('UserUpdateCtrl', [
    '$scope',
    '$state',
    'User',
    'UserForm',
    'userId',
    'ErrorMessage',
    function($scope, $state, User, UserForm, userId, ErrorMessage) {
        $scope.alert = {};
        // set initial values for form model by create deep copy of user object
        // so list/detail view is not updated while editing
        $scope.model = angular.copy(User.get(userId));

        // get all form fields
        $scope.formFields = UserForm.getFormFields();

        // save user
        $scope.save = function (user) {
            if (!user.groups_id) {
                user.groups_id = [];
            }
            // inject the changed user (copy) object back into DS store
            User.inject(user);
            // save change user object on server
            User.save(user).then(
                function (success) {
                    $scope.closeThisDialog();
                },
                function (error) {
                    // save error: revert all changes by restore
                    // (refresh) original user object from server
                    User.refresh(user);
                    $scope.alert = ErrorMessage.forAlert(error);
                }
            );
        };
    }
])

.controller('UserProfileCtrl', [
    '$scope',
    'Editor',
    'User',
    'operator',
    'UserProfileForm',
    'gettext',
    'ErrorMessage',
    function($scope, Editor, User, operator, UserProfileForm, gettext, ErrorMessage) {
        $scope.model = angular.copy(operator.user);
        $scope.title = gettext('Edit profile');
        $scope.formFields = UserProfileForm.getFormFields();
        $scope.save = function (user) {
            User.inject(user);
            User.save(user).then(
                function(success) {
                    $scope.closeThisDialog();
                },
                function(error) {
                    // save error: revert all changes by restore
                    // (refresh) original user object from server
                    User.refresh(user);
                    $scope.alert = ErrorMessage.forAlert(error);
                }
            );
        };
    }
])

.controller('UserChangePasswordCtrl', [
    '$scope',
    '$state',
    '$http',
    'User',
    'userId',
    'gettextCatalog',
    'PasswordGenerator',
    'ErrorMessage',
    function($scope, $state, $http, User, userId, gettextCatalog, PasswordGenerator, ErrorMessage) {
        User.bindOne(userId, $scope, 'user');
        $scope.alert = {};
        $scope.generatePassword = function () {
            $scope.new_password = PasswordGenerator.generate();
        };
        $scope.save = function (user) {
            if ($scope.new_password !== '') {
                $http.post(
                    '/rest/users/user/' + user.id + '/reset_password/',
                    {'password': $scope.new_password}
                ).then(
                    function (success) {
                        $scope.alert = {type: 'success', msg: success.data.detail, show: true};
                        $scope.new_password = '';
                    },
                    function (error) {
                        $scope.alert = ErrorMessage.forAlert(error);
                    }
                );
            }
        };
    }
])

.directive("showPassword", function() {
    return function linkFn(scope, elem, attrs) {
        scope.$watch(attrs.showPassword, function(value) {
            if (value) {
                elem.attr("type", "text");
            } else {
                elem.attr("type", "password");
            }
        });
    };
})

.controller('UserPasswordCtrl', [
    '$scope',
    '$state',
    '$http',
    'gettext',
    'UserPasswordForm',
    'ErrorMessage',
    function($scope, $state, $http, gettext, UserPasswordForm, ErrorMessage) {
        $scope.title = 'Change password';
        $scope.alert = {};
        $scope.model = {};
        $scope.formFields = UserPasswordForm.getFormFields();
        $scope.save = function (data) {
            if (data.newPassword != data.newPassword2) {
                data.newPassword = data.newPassword2 = '';
                $scope.alert = {
                    type: 'danger',
                    msg: gettext('Password confirmation does not match.'),
                    show: true,
                };
            } else {
                $http.post(
                    '/users/setpassword/',
                    {'old_password': data.oldPassword, 'new_password': data.newPassword}
                ).then(
                    function (success) {
                        $scope.closeThisDialog();
                    },
                    function (error) {
                        // Error, e. g. wrong old password.
                        $scope.model = {};
                        $scope.alert = ErrorMessage.forAlert(error);
                    }
                );
            }
        };
    }
])

.controller('UserPresenceCtrl', [
    '$scope',
    'User',
    'gettextCatalog',
    'ErrorMessage',
    function ($scope, User, gettextCatalog, ErrorMessage) {
        $scope.alert = {};
        $('#userNumber').focus();

        $scope.changeState = function () {
            if (!$scope.number) {
                return;
            }
            var enteredNumber = $scope.number.trim();
            var user = _.find(User.getAll(), function (user) {
                return user.number === enteredNumber;
            });
            if (user) {
                user.is_present = !user.is_present;
                User.save(user).then(function (success) {
                    var messageText = user.full_name + ' ' + gettextCatalog.getString('is now') + ' ';
                    messageText += gettextCatalog.getString(user.is_present ? 'present' : 'not present') + '.';
                    $scope.alert = {
                        msg: messageText,
                        show: true,
                        type: 'success',
                    };
                    $scope.number = '';
                }, function (error) {
                    $scope.alert = ErrorMessage.forAlert(error);
                });
            } else {
                $scope.alert = {
                    msg: gettextCatalog.getString('Cannot find the participant with the participant number') + ' "' + enteredNumber + '".',
                    show: true,
                    type: 'danger',
                };
            }
            $('#userNumber').focus();
        };
    }
])

.controller('UserImportCtrl', [
    '$scope',
    '$http',
    '$q',
    'gettext',
    'gettextCatalog',
    'User',
    'Group',
    'UserCsvExport',
    'osTablePagination',
    'ErrorMessage',
    function($scope, $http, $q, gettext, gettextCatalog, User, Group, UserCsvExport,
        osTablePagination, ErrorMessage) {
        // import from textarea
        $scope.importByLine = function () {
            var usernames = $scope.userlist[0].split("\n");
            var users = _.map(usernames, function (name) {
                // Split each full name in first and last name.
                // The last word is set as last name, rest is the first name(s).
                // (e.g.: "Max Martin Mustermann" -> last_name = "Mustermann")
                var names = name.split(" ");
                var last_name = names.slice(-1)[0];
                var first_name = names.slice(0, -1).join(" ");
                return {
                    first_name: first_name,
                    last_name: last_name,
                    groups_id: [],
                };
            });
            $http.post('/rest/users/user/mass_import/', {
                users: users
            }).then(function (success) {
                $scope.alert = {
                    show: true,
                    type: 'success',
                    msg: success.data.detail,
                };
            }, function (error) {
                $scope.alert = ErrorMessage.forAlert(error);
            });
        };

        // pagination
        $scope.pagination = osTablePagination.createInstance('UserImportTablePagination', 100);

        // Duplicates
        $scope.duplicateActions = [
            gettext('keep original'),
            gettext('override new'),
            gettext('create duplicate')
        ];

        // *** csv import ***
        $scope.csvConfig = {
            accept: '.csv, .txt',
            encodingOptions: ['UTF-8', 'ISO-8859-1'],
            parseConfig: {
                skipEmptyLines: true,
            },
        };

        var FIELDS = ['title', 'first_name', 'last_name', 'structure_level', 'number',
        'groups', 'comment', 'is_active', 'is_present', 'is_committee', 'default_password', 'email'];
        $scope.users = [];
        $scope.onCsvChange = function (csv) {
            $scope.csvImporting = false;
            // All user objects are already loaded via the resolve statement from ui-router.
            var users = User.getAll();
            $scope.users = [];

            var csvUsers = [];
            _.forEach(csv.data, function (row) {
                if (row.length >= 2) {
                    var filledRow = _.zipObject(FIELDS, row);
                    csvUsers.push(filledRow);
                }
            });
            $scope.duplicates = 0;
            _.forEach(csvUsers, function (user, index) {
                user.importTrackId = index;
                user.selected = true;
                if (!user.first_name && !user.last_name) {
                    user.importerror = true;
                    user.name_error = gettext('Error: Given name or surname is required.');
                }
                // number
                if (!user.number) {
                    user.number = "";
                }
                // groups
                user.groups_id = []; // will be overwritten if there are groups
                if (user.groups) {
                    user.groups = user.groups.split(',');
                    user.groups = _.map(user.groups, function (group) {
                        return _.trim(group); // remove whitespaces on start or end
                    });

                    // All group objects are already loaded via the resolve statement from ui-router.
                    var allGroups = Group.getAll();
                    // in allGroupsNames ar all original group names and translated names if a
                    // translation exists (e.g. for default group Delegates)
                    var allGroupsNames = [];
                    _.forEach(allGroups, function (group) {
                        var groupTranslation = gettextCatalog.getString(group.name);
                        if (group.name !== groupTranslation) {
                            allGroupsNames.push(groupTranslation);
                        }
                        allGroupsNames.push(group.name);
                    });
                    user.groupsToCreate = _.difference(user.groups, allGroupsNames);

                    // for template:
                    user.groupsNotToCreate = _.difference(user.groups, user.groupsToCreate);
                } else {
                    user.groups = [];
                }
                user.is_active = (user.is_active !== undefined && user.is_active === '1');
                user.is_present = (user.is_present !== undefined && user.is_present === '1');
                user.is_committee = (user.is_committee !== undefined && user.is_committee === '1');

                // Check for duplicates
                user.duplicate = false;
                users.forEach(function(user_) {
                    user_.fullname = [
                        user_.title,
                        user_.first_name,
                        user_.last_name,
                        user_.structure_level].join(' ').trim();
                    user.fullname = [
                        user.title,
                        user.first_name,
                        user.last_name,
                        user.structure_level].join(' ').trim();
                    if (user_.fullname === user.fullname) {
                        if (user.duplicate) {
                            // there are multiple duplicates!
                            user.duplicate_info += '\n' + gettextCatalog.getString('There are more than one duplicates of this user!');
                        } else {
                            user.duplicate = true;
                            user.duplicateAction = $scope.duplicateActions[1];
                            user.duplicate_info = '';
                            if (user_.title)
                                user.duplicate_info += user_.title + ' ';
                            if (user_.first_name)
                                user.duplicate_info += user_.first_name;
                            if (user_.first_name && user_.last_name)
                                user.duplicate_info += ' ';
                            if (user_.last_name)
                                user.duplicate_info += user_.last_name;
                            user.duplicate_info += ' (';
                            if (user_.number)
                                user.duplicate_info += gettextCatalog.getString('Number') + ': ' + user_.number + ', ';
                            if (user_.structure_level)
                                user.duplicate_info += gettextCatalog.getString('Structure level') + ': ' + user_.structure_level + ', ';
                            user.duplicate_info += gettextCatalog.getString('Username') + ': ' + user_.username + ') '+
                                gettextCatalog.getString('already exists.');

                            $scope.duplicates++;
                        }
                    }
                });
                $scope.users.push(user);
            });
            $scope.calcStats();
        };

        // Stats
        $scope.calcStats = function() {
            // not imported: if importerror or duplicate->keep original
            $scope.usersWillNotBeImported = 0;
            // imported: all others
            $scope.usersWillBeImported = 0;

            $scope.users.forEach(function(user) {
                if (!user.selected || user.importerror || (user.duplicate && user.duplicateAction == $scope.duplicateActions[0])) {
                    $scope.usersWillNotBeImported++;
                } else {
                    $scope.usersWillBeImported++;
                }
            });
        };

        $scope.setGlobalAction = function (action) {
            $scope.users.forEach(function (user) {
                if (user.duplicate)
                    user.duplicateAction = action;
            });
            $scope.calcStats();
        };

        // import from csv file
        $scope.import = function () {
            $scope.csvImporting = true;

            // collect all needed groups and create non existing groups
            var groupsToCreate = [];
            _.forEach($scope.users, function (user) {
                if (user.selected && !user.importerror && user.groups.length) {
                    _.forEach(user.groupsToCreate, function (group) { // Just append groups, that are not listed yet.
                        if (_.indexOf(groupsToCreate, group) == -1) {
                            groupsToCreate.push(group);
                        }
                    });
                }
            });
            var createPromises = [];
            $scope.groupsCreated = 0;
            _.forEach(groupsToCreate, function (groupname) {
                var group = {
                    name: groupname,
                    permissions: []
                };
                createPromises.push(Group.create(group).then( function (success) {
                    $scope.groupsCreated++;
                }));
            });

            $q.all(createPromises).then(function () {
                // reload allGroups, now all new groups are created
                var allGroups = Group.getAll();
                var existingUsers = User.getAll();

                // For option 'delete existing user' on duplicates
                var deletePromises = [];
                // Array of users for mass import
                var usersToBeImported = [];

                _.forEach($scope.users, function (user) {
                    if (user.selected && !user.importerror) {
                        // Assign all groups
                        _.forEach(user.groups, function(csvGroup) {
                            allGroups.forEach(function (allGroup) {
                                // check with and without translation
                                if (csvGroup === allGroup.name ||
                                    csvGroup === gettextCatalog.getString(allGroup.name)) {
                                    user.groups_id.push(allGroup.id);
                                }
                            });
                        });

                        // Do nothing on duplicateAction==duplicateActions[0] (keep original)
                        if (user.duplicate && (user.duplicateAction == $scope.duplicateActions[1])) {
                            // delete existing user
                            existingUsers.forEach(function(user_) {
                                user_.fullname = [
                                    user_.title,
                                    user_.first_name,
                                    user_.last_name,
                                    user_.structure_level].join(' ').trim();
                                user.fullname = [
                                    user.title,
                                    user.first_name,
                                    user.last_name,
                                    user.structure_level].join(' ').trim();
                                if (user_.fullname === user.fullname) {
                                    deletePromises.push(User.destroy(user_.id));
                                }
                            });
                            usersToBeImported.push(user);
                        } else if (!user.duplicate ||
                                   (user.duplicateAction == $scope.duplicateActions[2])) {
                            // create user
                            usersToBeImported.push(user);
                        }
                    }
                });
                $q.all(deletePromises).then(function () {
                    $http.post('/rest/users/user/mass_import/', {
                        users: usersToBeImported
                    }).then(function (success) {
                        _.forEach(success.data.importedTrackIds, function (trackId) {
                            _.find($scope.users, function (user) {
                                return user.importTrackId === trackId;
                            }).imported = true;
                        });
                        $scope.csvimported = true;
                    }, function (error) {
                        $scope.alert = ErrorMessage.forAlert(error);
                    });
                });
            });
        };
        $scope.clear = function () {
            $scope.users = null;
        };
        $scope.excludeImportedUsers = function () {
            $scope.users = _.filter($scope.users, function (user) {
                return !user.imported;
            });
            $scope.csvImporting = false;
            $scope.calcStats();
        };
        $scope.someImportedUsers = function () {
            return _.some($scope.users, function (user) {
                return user.imported;
            });
        };
        // download CSV example file
        $scope.downloadCSVExample = function () {
            UserCsvExport.downloadExample();
        };
    }
])

.controller('GroupListCtrl', [
    '$scope',
    '$http',
    '$filter',
    'operator',
    'Group',
    'permissions',
    'gettext',
    'Agenda',
    'Assignment',
    'Mediafile',
    'Motion',
    'User',
    'ngDialog',
    'OpenSlidesPlugins',
    function($scope, $http, $filter, operator, Group, permissions, gettext, Agenda,
        Assignment, Mediafile, Motion, User, ngDialog, OpenSlidesPlugins) {
        $scope.permissions = permissions;

        $scope.$watch(function() {
            return Group.lastModified();
        }, function() {
            $scope.groups = $filter('orderBy')(Group.getAll(), 'id');

            // find all groups with the 2 dangerous permissions
            var groups_danger = [];
            $scope.groups.forEach(function (group) {
                if ((_.indexOf(group.permissions, 'users.can_see_name') > -1) &&
                    (_.indexOf(group.permissions, 'users.can_manage') > -1)){
                    if (operator.isInGroup(group)){
                        groups_danger.push(group);
                    }
                }
            });
            // if there is only one dangerous group, block it.
            $scope.group_danger = groups_danger.length == 1 ? groups_danger[0] : null;
        });

        // Dict to map plugin name -> display_name
        var pluginTranslation = {};
        _.forEach(OpenSlidesPlugins.getAll(), function (plugin) {
            pluginTranslation[plugin.name] = plugin.display_name;
        });
        $scope.apps = [];
        // Create the main clustering with appname->permissions
        angular.forEach(permissions, function(perm) {
            var permissionApp = perm.value.split('.')[0]; // get appname

            // To insert perm in the right spot in $scope.apps
            var insert = function (id, perm, verboseName) {
                if (!$scope.apps[id]) {
                    $scope.apps[id] = {
                        app_name: verboseName,
                        app_visible: true,
                        permissions: []
                    };
                }
                $scope.apps[id].permissions.push(perm);
            };

            switch(permissionApp) {
                case 'core': // id 0 (projector) and id 6 (general)
                    if (perm.value.indexOf('projector') > -1) {
                        insert(0, perm, gettext('Projector'));
                    } else {
                        insert(6, perm, gettext('General'));
                    }
                    break;
                case 'agenda': // id 1
                    insert(1, perm, Agenda.verboseName);
                    break;
                case 'motions': // id 2
                    insert(2, perm, Motion.verboseNamePlural);
                    break;
                case 'assignments': // id 3
                    insert(3, perm, Assignment.verboseNamePlural);
                    break;
                case 'mediafiles': // id 4
                    insert(4, perm, Mediafile.verboseNamePlural);
                    break;
                case 'users': // id 5
                    insert(5, perm, User.verboseNamePlural);
                    break;
                default: // plugins: id>5
                    var display_name = pluginTranslation[permissionApp] || permissionApp.charAt(0).toUpperCase() +
                        permissionApp.slice(1);
                    // does the app exists?
                    var result = -1;
                    angular.forEach($scope.apps, function (app, index) {
                        if (app.app_name === display_name)
                            result = index;
                    });
                    var id = result == -1 ? $scope.apps.length : result;
                    insert(id, perm, display_name);
                    break;
            }
        });

        // sort each app: first all permission with 'see', then 'manage', then the rest
        // save the permissions in different lists an concat them in the right order together
        // Special Users: the two "see"-permissions are normally swapped. To create the right
        // order, we could simply reverse the whole permissions.
        angular.forEach($scope.apps, function (app, index) {
            if(index == 5) { // users
                app.permissions.reverse();
            } else { // rest
                var see = [];
                var manage = [];
                var others = [];
                angular.forEach(app.permissions, function (perm) {
                    if (perm.value.indexOf('see') > -1) {
                        see.push(perm);
                    } else if (perm.value.indexOf('manage') > -1) {
                        manage.push(perm);
                    } else {
                        others.push(perm);
                    }
                });
                app.permissions = see.concat(manage.concat(others));
            }
        });

        // check if the given group has the given permission
        $scope.hasPerm = function (group, permission) {
            return _.indexOf(group.permissions, permission.value) > -1;
        };

        // The current user is not allowed to lock himself out of the configuration:
        // - if the permission is 'users.can_manage' or 'users.can_see'
        // - if the user is in only one group with these permissions (group_danger is set)
        $scope.danger = function (group, permission){
            if ($scope.group_danger){
                if (permission.value == 'users.can_see_name' ||
                    permission.value == 'users.can_manage'){
                    return $scope.group_danger == group;
                }
            }
            return false;
        };

        // delete selected group
        $scope.delete = function (group) {
            Group.destroy(group.id);
        };

        // save changed permission
        $scope.changePermission = function (group, perm) {
            if (!$scope.danger(group, perm)) {
                if (!$scope.hasPerm(group, perm)) { // activate perm
                    group.permissions.push(perm.value);
                } else {
                    // delete perm in group.permissions
                    group.permissions = _.filter(group.permissions, function(value) {
                        return value != perm.value; // remove perm
                    });
                }
                Group.save(group);
            }
        };

        $scope.openDialog = function (group) {
            ngDialog.open({
                template: 'static/templates/users/group-edit.html',
                controller: group ? 'GroupRenameCtrl' : 'GroupCreateCtrl',
                className: 'ngdialog-theme-default wide-form',
                closeByEscape: false,
                closeByDocument: false,
                resolve: {
                    group: function () {return group;},
                }
            });
        };
    }
])

.controller('GroupRenameCtrl', [
    '$scope',
    'Group',
    'group',
    'gettextCatalog',
    'ErrorMessage',
    function($scope, Group, group, gettextCatalog, ErrorMessage) {
        $scope.group = group;
        $scope.new_name = gettextCatalog.getString(group.name);

        $scope.alert = {};
        $scope.save = function() {
            var old_name = gettextCatalog.getString($scope.group.name);
            $scope.group.name = $scope.new_name;
            Group.save($scope.group).then(
                function (success) {
                    $scope.closeThisDialog();
                },
                function (error) {
                   $scope.alert = ErrorMessage.forAlert(error);
                   $scope.group.name = old_name;
                }
            );
        };
    }
])

.controller('GroupCreateCtrl', [
    '$scope',
    'Group',
    'ErrorMessage',
    function($scope, Group, ErrorMessage) {
        $scope.new_name = '';
        $scope.alert = {};

        $scope.save = function() {
            var group = {
                name: $scope.new_name,
                permissions: []
            };

            Group.create(group).then(
                function (success) {
                    $scope.closeThisDialog();
                },
                function (error) {
                   $scope.alert = ErrorMessage.forAlert(error);
                }
            );
        };
    }
])

.factory('UserMenu', [
    '$http',
    'OpenSlides',
    'ngDialog',
    'UserProfileForm',
    'UserPasswordForm',
    function ($http, OpenSlides, ngDialog, UserProfileForm, UserPasswordForm) {
        return {
            logout: function () {
                $http.post('/users/logout/').then(function (response) {
                    // Success: User logged out, so reboot OpenSlides.
                    OpenSlides.reboot();
                });
            },
            editProfile: function () {
                ngDialog.open(UserProfileForm.getDialog());
            },
            changePassword: function () {
                ngDialog.open(UserPasswordForm.getDialog());
            },
        };
    }
])

.controller('userMenu', [
    '$scope',
    'UserMenu',
    function($scope, UserMenu) {
        $scope.logout = UserMenu.logout;
        $scope.editProfile = UserMenu.editProfile;
        $scope.changePassword = UserMenu.changePassword;

    }
])

.controller('LoginFormCtrl', [
    '$rootScope',
    '$scope',
    '$http',
    '$state',
    '$stateParams',
    '$q',
    'operator',
    'gettext',
    'autoupdate',
    'mainMenu',
    'DS',
    'ngDialog',
    function ($rootScope, $scope, $http, $state, $stateParams, $q, operator, gettext,
        autoupdate, mainMenu, DS, ngDialog) {
        $scope.alerts = [];

        if ($stateParams.msg) {
            $scope.alerts.push({
                type: 'danger',
                msg: $stateParams.msg,
            });
        }

        // check if guest login is allowed
        $scope.guestAllowed = $rootScope.guest_enabled;

        // get login info-text from server
        $http.get('/users/login/').then(function (success) {
            if(success.data.info_text) {
                $scope.alerts.push({
                    type: 'success',
                    msg: success.data.info_text
                });
            }
        });
        // check if cookies are enabled
        if (!navigator.cookieEnabled) {
            $scope.alerts.push({
                type: 'danger',
                msg: gettext('You have to enable cookies to use OpenSlides.'),
            });
        }

        // close alert function
        $scope.closeAlert = function(index) {
            $scope.alerts.splice(index, 1);
        };
        // login
        $scope.login = function () {
            $scope.closeThisDialog();
            $scope.alerts = [];
            var data = { 'username': $scope.username, 'password': $scope.password };
            if (!navigator.cookieEnabled) {
                data.cookies = false;
            }
            $http.post('/users/login/', data).then(
                function (response) {
                    // Success: User logged in.
                    // Clear store and reset deferred first message, if guests was enabled before.
                    DS.clear();
                    autoupdate.firstMessageDeferred = $q.defer();
                    // The next lines are partly the same lines as in core/start.js
                    autoupdate.newConnect();
                    autoupdate.firstMessageDeferred.promise.then(function () {
                        operator.setUser(response.data.user_id, response.data.user);
                        $rootScope.operator = operator;
                        mainMenu.updateMainMenu();
                        $state.go('home');
                        $rootScope.openslidesBootstrapDone = true;
                    });
                },
                function (error) {
                    // Error: Username or password is not correct.
                    $state.transitionTo($state.current, {msg: error.data.detail}, {
                          reload: true, inherit: false, notify: true
                    });
                }
            );
        };
        // guest login
        $scope.guestLogin = function () {
            $scope.closeThisDialog();
            $state.go('home');
        };
    }
])

// Mark all users strings for translation in JavaScript.
.config([
    'gettext',
    function (gettext) {
        // permission strings (see models.py of each Django app)
        // agenda
        gettext('Can see agenda');
        gettext('Can manage agenda');
        gettext('Can manage list of speakers');
        gettext('Can see hidden items and time scheduling of agenda');
        gettext('Can put oneself on the list of speakers');
        // assignments
        gettext('Can see elections');
        gettext('Can nominate another participant');
        gettext('Can nominate oneself');
        gettext('Can manage elections');
        // core
        gettext('Can see the projector');
        gettext('Can manage the projector');
        gettext('Can see the front page');
        gettext('Can manage tags');
        gettext('Can manage configuration');
        gettext('Can use the chat');
        gettext('Can manage the chat');
        gettext('Can manage logos and fonts');
        // mediafiles
        gettext('Can see the list of files');
        gettext('Can upload files');
        gettext('Can manage files');
        gettext('Can see hidden files');
        // motions
        gettext('Can see motions');
        gettext('Can create motions');
        gettext('Can support motions');
        gettext('Can manage motions');
        gettext('Can see comments');
        gettext('Can manage comments');
        // users
        gettext('Can see names of users');
        gettext('Can see extra data of users (e.g. present and comment)');
        gettext('Can manage users');

        // config strings in users/config_variables.py
        gettext('General');
        gettext('Sort name of participants by');
        gettext('Enable participant presence view');
        gettext('Participants');
        gettext('Given name');
        gettext('Surname');
        gettext('PDF');
        gettext('Welcome to OpenSlides');
        gettext('Title for access data and welcome PDF');
        gettext('[Place for your welcome and help text.]');
        gettext('Help text for access data and welcome PDF');
        gettext('System URL');
        gettext('Used for QRCode in PDF of access data.');
        gettext('WLAN name (SSID)');
        gettext('Used for WLAN QRCode in PDF of access data.');
        gettext('WLAN password');
        gettext('Used for WLAN QRCode in PDF of access data.');
        gettext('WLAN encryption');
        gettext('Used for WLAN QRCode in PDF of access data.');
        gettext('WEP');
        gettext('WPA/WPA2');
        gettext('No encryption');
        gettext('Email');
        gettext('Email sender');
        gettext('Email subject');
        gettext('Your login for {event_name}');
        gettext('You can use {event_name} as a placeholder.');
        gettext('Email body');
        gettext('Dear {name},\n\nthis is your OpenSlides login for the event {event_name}:\n\n    {url}\n    username: {username}\n    password: {password}\n\nThis email was generated automatically.');
        gettext('Use these placeholders: {name}, {event_name}, {url}, {username}, {password}. The url referrs to the system url.');
    }
]);


// this is code from angular.js. Find a way to call this function from this file
function getBlockNodes(nodes) {
  // TODO(perf): just check if all items in `nodes` are siblings and if they are return the original
  //             collection, otherwise update the original collection.
  var node = nodes[0];
  var endNode = nodes[nodes.length - 1];
  var blockNodes = [node];

  do {
    node = node.nextSibling;
    if (!node) break;
    blockNodes.push(node);
  } while (node !== endNode);

  return $(blockNodes);
}

}());

//# sourceMappingURL=data:application/json;charset=utf8;base64,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